#!/usr/bin/perl -w

use strict;
use warnings;

use List::MoreUtils qw/uniq/;

use Test::More;
use Test::Exception;

use Data::Dumper;
use Encode;

BEGIN {
    use FindBin qw/$Bin/;
    require "$Bin/_prepare.pl";
};
use utf8;

###################################
# get_new_id

# errors
dies_ok {get_new_id('ClientID');} "no auto inc, shard_key";

dies_ok {get_new_id('uid');} "no auto inc, chain_key";
dies_ok {get_new_id('uid', ClientID => 123);} "no auto inc, chain_key";

dies_ok {get_new_id('login', uid => 213);} "no auto inc";

save_shard(ClientID => 100, shard => 7);
save_shard(uid => 10, ClientID => 100);

# no chain_keys
dies_ok {get_new_id('bids_id', ClientID => 1);} "auto inc, extra chain_key";
my @bids_ids;
push @bids_ids, get_new_id('bids_id');
ok($bids_ids[-1] > 0, "auto inc");
push @bids_ids, get_new_id('bids_id');
ok($bids_ids[-1] > 0, "auto inc");
push @bids_ids, get_new_id('bids_id');
ok($bids_ids[-1] > 0, "auto inc");
is(scalar(@bids_ids), scalar(uniq @bids_ids), 'all ids uniq');
is(scalar(@bids_ids), scalar(grep {/^\d+$/} @bids_ids), 'all ids is numbers');

# chain_key
dies_ok {get_new_id('cid');} "auto inc, no chain_key";
dies_ok {get_new_id('cid', uid => 666);} "no shard info for uid=666";
my @cids;
push @cids, get_new_id('cid', ClientID => 10);
ok($cids[-1] > 0, "auto inc, correct chain_key");
push @cids, get_new_id('cid', ClientID => 10);
ok($cids[-1] > 0, "auto inc, correct chain_key");
push @cids, get_new_id('cid', ClientID => 10);
ok($cids[-1] > 0, "auto inc, correct chain_key");
push @cids, get_new_id('cid', uid => 10);
ok($cids[-1] > 0, "auto inc, skip chain_key");
is(get_shard(cid => $cids[-1]), 7, "correct shard for skip chain key"); 
is(scalar(@cids), 4, 'all ids');
is(scalar(@cids), scalar(uniq @cids), 'all ids uniq');
is(scalar(@cids), scalar(grep {/^\d+$/} @cids), 'all ids is numbers');


# multi
for my $cnt (1..5) {
    my @new_cids = @{get_new_id_multi('cid', $cnt, ClientID => $cnt)};
    is(scalar(@new_cids), $cnt, "multi, cnt=$cnt");
    is(
        get_one_field_sql(UT, ["SELECT count(*) from shard_inc_cid", WHERE => {cid => \@new_cids, ClientID => $cnt}]),
        $cnt,
        'correct chain_val'
        );
    push @cids, @new_cids;
    is(
        get_one_field_sql(UT, "SELECT count(*) from shard_inc_cid"),
        scalar(@cids),
        'correct rows num'
        );
}
is(scalar(@cids), scalar(uniq @cids), 'all ids uniq');
is(scalar(@cids), scalar(grep {/^\d+$/} @cids), 'all ids is numbers');

done_testing();

