#!/usr/bin/perl -w

use strict;
use warnings;

use Test::More;
use Test::Exception;
use Test::Deep;

use POSIX qw/ceil/;

use utf8;

BEGIN {
    use FindBin qw/$Bin/;
    require "$Bin/_prepare.pl";
};

dies_ok { sharded_chunks( xxx => 111, 1 ) };
dies_ok { sharded_chunks( cid => 111, jhkjh => 7) };
dies_ok { foreach_shard cid => [], adsf => 324, sub {} };

sub _unchunk
{
    my %ret;
    for my $c (@_) {
        for my $id (@{$c->{cid}}) {
            $ret{$id} = $c->{shard};
        }
    }
    return %ret;
}

my %cid2shard = (
    100 => 1,
    101 => 1,
    102 => 2,
    103 => 1,
    104 => 2,
    105 => 3,
    106 => 3,
    107 => 3,
);

my $ClientID = 1;
save_shard(ClientID => 42, shard => 0);
save_shard(cid => 200, ClientID => 42);

while (my ($cid, $shard) = each %cid2shard) {
    save_shard(ClientID => $ClientID, shard => $shard);
    save_shard(cid => $cid, ClientID => $ClientID);

    my $uid = 1000 + $ClientID;
    save_shard(uid => $uid, ClientID => $ClientID);

    my $login = "login_for_cid_$cid";
    save_shard(login => $login, uid => $uid);

    $ClientID++;
}

for my $n (0 .. 4) {
    my @sharded_cids = sharded_chunks(cid => [ keys %cid2shard ], $n);
    cmp_deeply(\%cid2shard, { _unchunk @sharded_cids }, "expected unchunk, n = $n");
    ok( @sharded_cids == (grep { scalar @{$_->{cid}} <= $n } @sharded_cids), "length <= $n") if $n > 0;
}

for my $n (0 .. 4) {
    my %c = ();
    my $iter = sharded_chunks_iter(cid => [keys %cid2shard], $n);
    while (my ($shard, $c) = $iter->()) {
        $c{$_} = $shard for @$c;
        ok( scalar @$c <= $n, "length <= $n") if $n > 0;
    }
    cmp_deeply(\%c, \%cid2shard, "iter, n = $n");
}

for my $n (0 .. 4) {
    my %c = ();
    foreach_shard cid => [keys %cid2shard], chunk_size => $n, sub {
        my ($shard, $c) = @_;
        $c{$_} = $shard for @$c;
        ok( scalar @$c <= $n, "length <= $n") if $n > 0;
    };
    cmp_deeply(\%c, \%cid2shard, "iter, n = $n");
}


# массив хэшей
cmp_deeply(
    [sharded_chunks(cid => [{cid => 100, i => 1}, {cid => 101, i=>2}, {cid => 102, i => 3}])],
    [
     {shard => 1, cid => [{cid => 100, i => 1}, {cid => 101, i => 2}]},
     {shard => 2, cid => [{cid => 102, i => 3}]},
    ]
    );

# by - строка
cmp_deeply(
    [sharded_chunks(
         cid => [{ccid => 100, i => 1}, {ccid => 101, i=>2}, {ccid => 102, i => 3}],
         by => 'ccid'
     )],
    [
     {shard => 1, cid => [{ccid => 100, i => 1}, {ccid => 101, i => 2}]},
     {shard => 2, cid => [{ccid => 102, i => 3}]},
    ]
    );
cmp_deeply(
    [sharded_chunks(
         cid => [],
         by => 'ccid'
     )],
    [
    ]
    );

# by - функция
cmp_deeply(
    [sharded_chunks(
         cid => [{ccid => [100], i => 1}, {ccid => [101], i=>2}, {ccid => [102], i => 3}],
         by => sub {$_->{ccid}->[0]}
     )],
    [
     {shard => 1, cid => [{ccid => [100], i => 1}, {ccid => [101], i => 2}]},
     {shard => 2, cid => [{ccid => [102], i => 3}]},
    ]
    );

# by - функция
cmp_deeply(
    [sharded_chunks(
         cid => [{ccid => [100], i => 1}, {ccid => [101], i=>2}, {ccid => [102], i => 3}],
         by => sub {shift->{ccid}->[0]}
     )],
    [
     {shard => 1, cid => [{ccid => [100], i => 1}, {ccid => [101], i => 2}]},
     {shard => 2, cid => [{ccid => [102], i => 3}]},
    ]
    );

# by - функция
cmp_deeply(
    [sharded_chunks(
         cid => [[100, 1], [101, 2], [102, 3]],
         by => sub {$_->[0]}
     )],
    [
     {shard => 1, cid => [[100, 1], [101, 2]]},
     {shard => 2, cid => [[102, 3]]},
    ]
    );

# by - функция, указана несуществующая кампания
cmp_deeply(
    [sharded_chunks(
         cid => [{ccid => [199], i => 1}, {ccid => [101], i=>2}, {ccid => [102], i => 3}],
         by => sub {$_->{ccid}->[0]}
     )],
    [
     {shard => 1, cid => [{ccid => [101], i => 2}]},
     {shard => 2, cid => [{ccid => [102], i => 3}]},
    ]
    );

# by - функция, указана несуществующая кампания, указан with_undef_shard
cmp_deeply(
    [sharded_chunks(
         cid => [{ccid => [199], i => 1}, {ccid => [101], i=>2}, {ccid => [102], i => 3}, {ccid => [200], i => 4}],
         by => sub {$_->{ccid}->[0]},
         with_undef_shard => 1
     )],
    [
     {shard => undef, cid => [{ccid => [199], i => 1}]},
     {shard => 0, cid => [{ccid => [200], i => 4}]},
     {shard => 1, cid => [{ccid => [101], i => 2}]},
     {shard => 2, cid => [{ccid => [102], i => 3}]},
    ]
    );

# by - функция, указана несуществующая кампания, указан with_undef_shard
cmp_deeply(
    [sharded_chunks(
         shard => [1,1,1,24],
     )],
    [
     {shard => 1},
     {shard => 24},
    ]
    );

# если массив хэшей, и id неопределён - падаем
dies_ok {sharded_chunks(cid => [{}])};
dies_ok {sharded_chunks(cid => [{cid => 100}], by => 'ccid')};
dies_ok {sharded_chunks(cid => [{cid => 100}], by => sub {$_->{ccid}})};

# шардирование по нецифровым полям
cmp_deeply(
    [ sharded_chunks(login => ['login_for_cid_100', 'login_for_cid_101', 'login_for_cid_102']) ],
    [
     {shard => 1, login => ['login_for_cid_100', 'login_for_cid_101']},
     {shard => 2, login => ['login_for_cid_102']},
    ]
    );

cmp_deeply(
    [sharded_chunks(cid => [ keys(%cid2shard), 200, 300 ], with_undef_shard => 1, chunk_size => 100)],
    [
       {shard => undef, cid => [300]},
       {shard => 0, cid => [200]},
       {shard => 1, cid => [100, 101, 103]},
       {shard => 2, cid => [102, 104]},
       {shard => 3, cid => [105, 106, 107]},
    ],
    'sharded_chunks with zero and undef shards'
    );

done_testing();

