#!/usr/bin/perl -w

=pod

    $Id$

=cut

use strict;
use warnings;

use Test::More;
use Test::Exception;
use POSIX qw/strftime :sys_wait_h/;

use Yandex::DBTools;
use Yandex::DBUnitTest qw/:all/;

use utf8;

local $Yandex::DBTools::DONT_SEND_LETTERS = 1;

my $LOCK = join '_', 'test_lock', strftime("%Y%m%d%H%M%S", localtime), $$;

my $dbh = get_dbh(UT);
my $dbh2 = $dbh->clone();

sub get2 {
    my $res = [ $dbh2->selectrow_array(shift, {}, @_) ];
    warn $dbh->errstr if $dbh->err;
    return $res->[0];
}

ok(get_lock_sql($dbh, $LOCK), 'get lock');

ok(!get2("select get_lock(?, 0)", $LOCK), 'get lock for dbh2');

dies_ok { get_lock_sql($dbh, $LOCK) } 'dies on get same lock';
dies_ok { get_lock_sql($dbh, "NEW_$LOCK") } 'dies on get another lock';

ok(release_lock_sql($dbh, $LOCK), 'release lock');

ok(get2("select get_lock(?, 0)", $LOCK), 'get lock for dbh2');
ok(get2("select release_lock(?)", $LOCK), 'release lock for dbh2');

{
    my $lock;
    lives_ok { $lock = sql_lock_guard($dbh, $LOCK, 1) } 'sql_lock_guard on $dbh';
    ok(!get2("select get_lock(?, 0)", $LOCK), 'get lock for dbh2');
}
ok(get2("select get_lock(?, 0)", $LOCK), 'get lock for dbh2');
dies_ok {sql_lock_guard($dbh, $LOCK, 0);} "guard dies when can't get lock";

ok(get2("select release_lock(?)", $LOCK), 'release lock for dbh2');

{
    get_dbh(SHUT_3)->{PrintError} = 0;
    my $limit = get_one_field_sql(SHUT_3, 'select @@max_allowed_packet');
    my $lock;
    lives_ok { $lock = sql_lock_guard(SHUT_3, $LOCK, 1) } 'sql_lock_guard on SHUT_3';
    ok(!get2("select get_lock(?, 0)", $LOCK), 'get lock for dbh2');

    is(Yandex::DBTools::has_broken_lock(SHUT_3), 0, 'lock not broken');
    dies_ok {
        get_one_field_sql(SHUT_3, "SELECT 1 -- ".('x' x $limit));
    };
    is(Yandex::DBTools::has_broken_lock(SHUT_3), 1, 'lock broken');
    throws_ok {
        get_one_field_sql(SHUT_3, "SELECT 132");
    } qr/Reconnect under sql-lock/, 'dies on reconnect under sql-lock';
}
# проверяем, что информация о локе в SHUT_3 удалилась при разрушении lock_guard
is(scalar(keys(%Yandex::DBTools::LOCK_DBH)), 0, 'no locks into %LOCK_DBH');
%Yandex::DBTools::LOCK_DBH = ();

# кейс со снятием sql-lock-guard, если $dbh->{Active}=0
{
    get_dbh(SHUT_3)->{PrintError} = 0;
    my $lock;
    lives_ok { $lock = sql_lock_guard(SHUT_3, $LOCK, 1) } 'sql_lock_guard on SHUT_3';
    ok(!get2("select get_lock(?, 0)", $LOCK), 'get lock for dbh2');
    get_dbh(SHUT_3)->disconnect();
}
# проверяем, что информация о локе в SHUT_3 удалилась при разрушении lock_guard
is(scalar(keys(%Yandex::DBTools::LOCK_DBH)), 0, 'no locks into %LOCK_DBH');
%Yandex::DBTools::LOCK_DBH = ();

# кейс со реконнектом под локом, если $dbh->{Active}=0
{
    get_dbh(SHUT_3)->{PrintError} = 0;
    my $lock;
    lives_ok { $lock = sql_lock_guard(SHUT_3, $LOCK, 1) } 'sql_lock_guard on SHUT_3';
    ok(!get2("select get_lock(?, 0)", $LOCK), 'get lock for dbh2');
    is(Yandex::DBTools::has_broken_lock(SHUT_3), 0, 'lock not broken');
    get_dbh(SHUT_3)->disconnect();
    is(Yandex::DBTools::has_broken_lock(SHUT_3), 1, 'lock broken');

    throws_ok {
        get_one_field_sql(SHUT_3, "SELECT 132");
    } qr/Reconnect under sql-lock/, 'dies on reconnect under sql-lock on inactive dbh';
}
# проверяем, что информация о локе в SHUT_3 удалилась при разрушении lock_guard
is(scalar(keys(%Yandex::DBTools::LOCK_DBH)), 0, 'no locks into %LOCK_DBH');
%Yandex::DBTools::LOCK_DBH = ();

dies_ok {
        get_lock_sql(SHUT(shard => [1,2]));
} 'get_lock_sql dies on several dbnames';

my $lock;
lives_ok { $lock = sql_lock_guard(SHUT_4, $LOCK, 1) } 'sql_lock_guard on SHUT_4';

sub _do_some_request {
    return get_one_field_sql(shift, 'SELECT 1');
}

_do_some_request(SHUT_1);
_do_some_request(SHUT_4);
lives_ok {
        my $child_pid = fork();
        if ($child_pid) {
            waitpid($child_pid, 0);
            ok(WIFEXITED($?) && WEXITSTATUS($?) == 0, 'child connection works after fork with alive connection without sql-lock');
            _do_some_request(SHUT_1);
        } else {
            _do_some_request(SHUT_1);
            exit 0;
        }
} 'prior-fork connections withouth locks must be usable after fork';

# не разрешаем выполнять запросы на форкнутых соединениях со взятым локом
lives_ok {
        my $child_pid = fork();
        if ($child_pid) {
            waitpid($child_pid, 0);
            ok(WIFEXITED($?) && WEXITSTATUS($?) == 0, 'child dies after fork with alive connection with sql-lock');
            _do_some_request(SHUT_4);
        } else {
            $SIG{__DIE__} = sub { exit 255 };
            _do_some_request(SHUT_4);
            exit 0;
        }
} 'prior-fork connections with locks must be usable after fork';

# не разрешаем отпускать лок в форке, если взят был родителем
lives_ok {
        my $child_pid = fork();
        if ($child_pid) {
            waitpid($child_pid, 0);
            ok(WIFEXITED($?) && WEXITSTATUS($?) == 42, 'child dies after fork with alive connection with sql-lock');
            _do_some_request(SHUT_4);
        } else {
            $SIG{__DIE__} = sub { exit($_[0] =~ /Can't release sql-lock .* in forked connection/ ? 42 : 255)};
            release_lock_sql(SHUT_4, $LOCK);
            exit 0;
        }
} 'release_lock_sql dies during release lock, obtained before fork';

done_testing();
