#!/usr/bin/perl -w

use strict;
use warnings;

use Test::Exception;
use Test::More;

use Yandex::DBTools;
use Yandex::DBUnitTest qw/:all/;
use constant PPC => 'ppc';

use utf8;

$Yandex::DBTools::QUOTE_DB = PPC;

is(sql_case(PPC, "cid", {123 => 124}),
    "CASE `cid` WHEN '123' THEN '124' ELSE NULL END");
is(sql_case(PPC, "c.cid", {123 => 124}),
    "CASE `c`.`cid` WHEN '123' THEN '124' ELSE NULL END");
is(sql_case(PPC, "c.cid", {}),
    "NULL");
is(sql_case(PPC, "cid", {qwe => "wer"}, default => undef),
    "CASE `cid` WHEN 'qwe' THEN 'wer' ELSE NULL END");
is(sql_case(PPC, "cid", {qwe => "wer"}, default__dont_quote => 'bi.price'),
    "CASE `cid` WHEN 'qwe' THEN 'wer' ELSE bi.price END");
is(sql_case(PPC, "cid", {qwe => "wer"}, default => "dsaf"),
    "CASE `cid` WHEN 'qwe' THEN 'wer' ELSE 'dsaf' END");
dies_ok {sql_case(PPC, field => {}, DEFAULT => 234)};
dies_ok {sql_case(PPC, field => {}, default => 234, dont_quote => 1)};
is(sql_case(PPC, "cid", {qwe => "wer"}, default => "dsaf", dont_quote_value => 1),
    "CASE `cid` WHEN 'qwe' THEN wer ELSE dsaf END");

like(sql_case(PPC, "cid", {map {$_ => $_} 1..100}, default => "dsaf", dont_quote_value => 1),
    qr/^IF/);
like(sql_case(PPC, "cid", {asd => 123, map {$_ => $_} 1..100}, default => "dsaf", dont_quote_value => 1),
    qr/^CASE/);

# создаём временную табличку для выполнения сформированных case
create_table(UT, "test", "f");
my $N = 100;
do_mass_insert_sql(PPC, "insert into test (f) values %s",
                   [(map {[$_]} 1..$N), (map {["str$_"]} 1..$N)]
    );

my @tests = (
    { 
        # параметры для sql_case
        params => [{map {$_=>$_+1} 1..50}],
        # регулярка для результата
        like => qr/^IF/,
        # аналогичное преобразование в перле для сравнения
        res => sub {$_ =~ /^\d+$/ && $_ <= 50 ? $_ + 1 : undef},
    },
    { params => [{map {$_=>"`f`+1"} 1..50}, dont_quote_value => 1],
      like => qr/^IF/,
      res => sub {$_ =~ /^\d+$/ && $_ <= 50 ? $_ + 1 : undef},
    },
    { params => [{map {$_=>$_+1} 1..50}, default => 42],
      like => qr/^IF/,
      res => sub {$_ =~ /^\d+$/ && $_ <= 50 ? $_ + 1 : 42},
    },
    { params => [{map {($_=>$_+1, "str$_" => "val$_")} 1..50}, default => 42],
      like => qr/^CASE/,
      res => sub {$_ =~ /^\d+$/ && $_ <= 50 ? $_ + 1 : $_ =~ /^str(\d+)$/ && $1 <= 50 ? "val$1" : 42},
    },
    { params => [{map {($_=>$_+1, "str$_" => "`f`")} 1..50}, default => 42, dont_quote_value => 1],
      like => qr/^CASE/,
      res => sub {$_ =~ /^\d+$/ && $_ <= 50 ? $_ + 1 : $_ =~ /^str(\d+)$/ && $1 <= 50 ? $_ : 42},
    },
    );
for my $i (0..$#tests) {
    my $test = $tests[$i];
    for my $with_db (0, 1) {
        my $sql_case = sql_case(($with_db ? PPC : ()), "f", @{$test->{params}});
        like($sql_case, $test->{like}, "test #$i: like");
        my %res = map {$_->{f} => $_->{v}} @{get_all_sql(PPC, "SELECT f, $sql_case v FROM test")};
        my %check_res = map {$_ => $test->{res}->()} keys %res;
        is_deeply(\%res, \%check_res, "test #$i: check values");
    }
}

done_testing;
