#!/usr/bin/perl -w

use strict;
use warnings;
use utf8;

use Yandex::DBTools;
use Yandex::DBUnitTest qw/:all/;
use Test::More;
use Yandex::Test::UTF8Builder;
use open ':std' => ':utf8';

$Yandex::DBTools::DONT_SEND_LETTERS = 1;

my @tests = (
                { 
                    str => '%value%',
                    starts_with => ['%value', ''],
                    doesnt_start_with => ['_value', '*value', '.value'],
                    contains => ['value', '%value%', 'value%', ''],
                    doesnt_contain => ['value_'],
                },
                {
                    str => '_value_',
                    starts_with => ['_value'],
                    doesnt_start_with => ['%value'],
                    contains => ['_value_', 'value_'],
                    doesnt_contain => ['value%', '%value%'],
                },
                {
                    str => '\%value\%',
                    starts_with => ['\%value'],
                    doesnt_start_with => ['%value'],
                    contains => ['value\%'],
                    doesnt_contain => ['value%', '%value%'],
                },
                {
                    str => '*value*',
                    starts_with => ['*value'],
                    doesnt_start_with => ['%value', '.value', '_value'],
                    contains => ['*value*', 'value*'],
                    doesnt_contain => ['value_', 'value.', 'value%']
                },
                {
                    str => '.value.',
                    starts_with => ['.value'],
                    doesnt_start_with => ['*value', '_value'],
                    contains => ['.value.', 'value.'],
                    doesnt_contain => ['value*', 'value%']
                },
                {
                    str => "ремонт",
                    starts_with => ["ремонт", "ремо", "Ремон"],
                    doesnt_start_with => [ "евроремонт" ],
                    contains => ["емонт", "Рем", "ремонт"],
                    doesnt_contain => ["чип"],
                }
            );


foreach my $test (@tests) {
    is (index(lc $test->{str}, lc $_), 0, "sanity check: '$test->{str}' starts with '$_'") foreach @{$test->{starts_with}};
    isnt (index(lc $test->{str}, lc $_), 0, "sanity check: '$test->{str}' doesn't start with '$_'") foreach @{$test->{doesnt_start_with}};
    isnt (index(lc $test->{str}, lc $_), -1, "sanity check: '$test->{str}' contains '$_'") foreach @{$test->{contains}};
    is (index(lc $test->{str}, lc $_), -1, "sanity check: '$test->{str}' doesn't contain '$_'") foreach @{$test->{doesnt_contain}};
}

my $id = 1;
$_->{id} = $id++ foreach @tests;

create_table(UT, 'test_table', 'id:pk', 'value:text');

do_insert_into_table(UT, 'test_table', {id => $_->{id}, value => $_->{str} }) foreach @tests;

sub get_one_id {
    my ($condition_hash) = @_;
    return get_one_field_sql(UT, ['select id from test_table', where => $condition_hash] );
}

sub contains_count {
    my ($cont) = @_;
    my $cond = {value__contains_any => $cont};
    return get_one_field_sql(UT, ['select count(id) from test_table', where => $cond] );
}

foreach my $test (@tests) {
    my ($id, $str) = ($test->{id}, $test->{str});
    ok (get_one_id( {value__starts_with => $_, id => $id}), "'$str' starts with '$_'") foreach @{$test->{starts_with}};
    ok (!get_one_id( {value__starts_with => $_, id => $id}), "'$str' doesn't start with '$_'") foreach @{$test->{doesnt_start_with}};
    ok (get_one_id( {value__contains => $_, id => $id}), "'$str' contains '$_'") foreach @{$test->{contains}};
    ok (!get_one_id( {value__contains => $_, id => $id}), "'$str' doesn't contain '$_'") foreach @{$test->{doesnt_contain}};
}

cmp_ok(contains_count( ["РЕМОНТ"] ), "eq", 1);
cmp_ok(contains_count( [".VALUE"] ), "eq", 1);
cmp_ok(contains_count( [".Valu", "ремо"] ), "eq", 2);
cmp_ok(contains_count( [".valu", "РЕмо"] ), "eq", 2);

done_testing();
