#!/usr/bin/perl -w

# $Id$

=head1 DBUnitTest.t

    Юнит-тесты модуля для юнит-тестов.
    Звучит странно, но проверить что всё работает как задумано - никогда не бывает лишним.
    Можно воспринимать эти тесты как примеры разных кейсов для:
        %dataset, описывающего содержимое баз данных
        &init_test_dataset - создающей таблицы и заполняющей их данными
        &replace_test_data - заменяющей в одной(!) таблице (но во всех шардах) данные на заданные
        &check_test_dataset - проверяющей, что все содержимое %dataset действительно существует в правильных базах/шардах

=cut

use strict;
use warnings;

use Test::More;
use Test::Deep;
use Test::Exception;

use Yandex::DBTools;
use Yandex::DBUnitTest qw/:all/;

use utf8;

$Yandex::DBTools::DONT_SEND_LETTERS = 1;

my $create_string = ['uid:bigint(20):pk', 'domain_login', 'manager_private_email', 'is_developer:tinyint(1)'];

# Ручное создание таблицы
sub _create_table{
    my ($db, $table) = @_;
    do_sql($db, q/
        CREATE TABLE /.$table.q/ (
            uid bigint not null  primary key
            , domain_login varchar(100)
            , manager_private_email varchar(100)
            , is_developer tinyint
        ) ENGINE=MyISAM/);
}
# Ручная выборка данных из таблицы
sub _real_rows{
    my ($db, $table) = @_;
    return get_all_sql($db, q/SELECT uid, manager_private_email, is_developer, domain_login FROM /.$table);
}

###########################################################
##### Вторя группа тестов - только шардированная база #####
###########################################################
# Использавно SHUT(shard => 'all') в качестве original_db потому, что для создания таблиц во всех шардах - нужно эти шарды знать.
# Просто SHUT не работает при STRICT_SHARD_DBNAMES
my $ut_db_rows = [
    {uid => 9, domain_login => 'root', manager_private_email => 'root@yandex-team.ru', is_developer => 1},
    {uid => 10, domain_login => 'test', manager_private_email => 'noreply@yandex-team.ru', is_developer => 0},
    {uid => 11, domain_login => 'unknown', manager_private_email => 'vasia_pupkin@mail.ru', is_developer => 0},
];
my %shut_db = (
    # Табличка с данными без указания шарда - должны "лечь" в первый шард
    half_sharded_table => {
        original_db => SHUT(shard => 'all'),
        create_string => $create_string,
        rows => $ut_db_rows,
    },
    # Табличка с разными данными в разных шардах (во втором шарде те же, данные что в первом примере)
    sharded_table => {
        original_db => SHUT(shard => 'all'),
        create_string => $create_string,
        rows => {
            1 => $ut_db_rows,
            2 => [
                {uid => 2, domain_login => 'zzz', manager_private_email => 'abuse@yandex-team.ru', is_developer => 0},
                {uid => 3, domain_login => 'aaa', manager_private_email => 'robot@yandex-team.ru', is_developer => 1},
                {uid => 4, domain_login => 'q', manager_private_email => 'q@rambler.ru', is_developer => 0},
            ],
        },
    },
);
# Табличка для тестов на шардовый ключ all
my %shut_all_db = (
    sharded_table => {
        original_db => SHUT(shard => 'all'),
        create_string => $create_string,
        rows => {
            all => $ut_db_rows,
        }
    },
);

# Тестовый набор данных
my $test_bag_1 = bag(map {superhashof($_)} @$ut_db_rows);
my $test_bag_2 = bag(map {superhashof($_)} @{ $shut_db{sharded_table}->{rows}->{2} });
my $extra_row = {uid => 42, domain_login => 'zombie', manager_private_email => 'zombie@hotmail.com', is_developer => -1};

sub _check_table_not_exists{
    my $msg_prefix = shift;
    open(my $stderr, '>&', \*STDERR);
    open(STDERR, '>/dev/null');    # Подавляем сообщения об ошибках системы

    # Проверяем что таблица half_sharded_table не попала в нешардированную базу
    dies_ok { exec_sql(UT, q/DESC half_sharded_table/) } qq/$msg_prefix - half_sharded_table should not exists in UT/;
    # Таблица sharded_table в нешардированную базу попасть не должна была ни при каком раскладе
    dies_ok { exec_sql(UT, q/DESC sharded_table/) } qq/$msg_prefix - sharded_table should not exists in UT/;
    # Таблица single_table в шардированную базу попасть не должна была ни при каком раскладе
    dies_ok { exec_sql(SHUT(shard => 'all'), q/DESC single_table/) } qq/$msg_prefix - single_table should not exists in SHUT/;


    *STDERR = $stderr;  # Возвращаем STDERR
}
sub _uids{
    my $shard = shift;
    return [ map {$_->{uid}} @{ $shut_db{sharded_table}->{rows}->{$shard} } ];
}
sub _check_sharded_data{
    my $msg_prefix = shift;

    # Проверяем нешардированные данные
    cmp_deeply(_real_rows(SHUT(shard => 1), 'half_sharded_table'), $test_bag_1, "SHUT: $msg_prefix - manually checked (not sharded data found in 1 shard)");
    cmp_deeply(_real_rows(SHUT(shard => 2), 'half_sharded_table'), [], "SHUT: $msg_prefix - manually checked (not sharded data not exists in 2 shard)");
    
    # Проверяем, что шардированные данные там, где нужно
    cmp_deeply(_real_rows(SHUT(shard => 1), 'sharded_table'), $test_bag_1, "SHUT: $msg_prefix - manually checked (sharded data1 found in 1 shard)");
    cmp_deeply(_real_rows(SHUT(shard => 2), 'sharded_table'), $test_bag_2, "SHUT: $msg_prefix - manually checked (sharded data2 found in 2 shard)");
    
    # Проверяем, что данные, предназнаечнные для одного шарда не попали в другой
    ok(!get_one_field_sql(SHUT(shard => 1), [q/SELECT count(*) FROM sharded_table WHERE/, {uid => _uids(2)}]), "SHUT: $msg_prefix - manually checked (sharded data2 not found in 1 shard)");
    ok(!get_one_field_sql(SHUT(shard => 2), [q/SELECT count(*) FROM sharded_table WHERE/, {uid => _uids(1)}]), "SHUT: $msg_prefix - manually checked (sharded data1 not found in 2 shard)");

    # Проверяем все сразу функцией
    check_test_dataset(\%shut_db, "SHUT: $msg_prefix - check_test_dataset");
}
sub _replace_sharded_data{
    lives_ok {
        replace_test_data(SHUT(shard => 'all'), 'half_sharded_table', $shut_db{half_sharded_table}->{rows});
        replace_test_data(SHUT(shard => 'all'), 'sharded_table', $shut_db{sharded_table}->{rows});
    } 'SHUT: replace_test_data';
}
sub _check_sharded_to_all_data{
    my $msg_prefix = shift;

    foreach my $db (dbnames(SHUT(shard => 'all'))) {
        $db =~ m/:(\d+)$/;
        my $num = $1;
        cmp_deeply(_real_rows($db, 'sharded_table'), $test_bag_1, "SHUT: all shards - $msg_prefix - manually checked (shard $num)");
    }
    check_test_dataset(\%shut_all_db, "SHUT: all shards - $msg_prefix - check_test_dataset");
}

################################################################################
# Шардированная база - создаем и заполняем таблицы вручную
lives_ok {
    _create_table(SHUT(shard => 'all'), 'half_sharded_table');
    foreach my $row ( @{ $shut_db{half_sharded_table}->{rows} } ) {
        do_insert_into_table(SHUT(shard => 1), 'half_sharded_table', $row);
    }
    _create_table(SHUT(shard => 'all'), 'sharded_table');
    foreach my $shard (keys %{$shut_db{sharded_table}->{rows}}) {
        foreach my $row ( @{ $shut_db{sharded_table}->{rows}->{$shard} } ) {
            do_insert_into_table(SHUT(shard => $shard), 'sharded_table', $row);
        }
    }
} 'SHUT: manually inserted data';

# Проверяем что таблички НЕ появились там, где не должны
_check_table_not_exists('SHUT: manually inserted data');

# Проверяем
_check_sharded_data('manually inserted data');

# Удаляем таблицы
lives_ok {
    do_sql(SHUT(shard => 'all'), q/DROP TABLE half_sharded_table/);
    do_sql(SHUT(shard => 'all'), q/DROP TABLE sharded_table/);
    %Yandex::DBUnitTest::CREATED_TABLES = ();
} 'SHUT: DROP TABLEs';
################################################################################
# Шардированная база - создаем и заполняем таблицы через init_test_dataset
lives_ok { init_test_dataset(\%shut_db) } 'SHUT: init_test_dataset';

# Проверяем что таблички НЕ появились там, где не должны
_check_table_not_exists('SHUT: init_test_dataset');

# Проверяем
_check_sharded_data('init_test_dataset');
################################################################################
# Шардированная база - данные восстанавливаются через init_test_dataset, а затем данные изменяются
lives_ok {
    do_update_table(SHUT(shard => 'all'), 'half_sharded_table', {domain_login => 'broken login'}, where => {uid => [3, 9, 11]})
} 'SHUT: modified data in half_sharded_table';
ok( !(Yandex::DBUnitTest::_check_test_dataset(\%shut_db))[0], "SHUT: modified data in half_sharded_table - check_test_dataset failed");

# Восстанавливаем данные, и портим в другой таблице
lives_ok { init_test_dataset(\%shut_db) } 'SHUT: init_test_dataset';
check_test_dataset(\%shut_db, "SHUT: init_test_dataset - check_test_dataset");
lives_ok {
    do_update_table(SHUT(shard => 'all'), 'sharded_table', {domain_login => 'bad login'}, where => {uid => [2, 4, 10]})
} 'SHUT: modified data in sharded_table';

# Проверяем, что не данные не совпали
ok( !(Yandex::DBUnitTest::_check_test_dataset(\%shut_db))[0], "SHUT: modified data in sharded_table - check_test_dataset failed");
################################################################################
# Шардированная база - данные восстанавливаются через replace_test_data, а затем добавляются лишние
_replace_sharded_data();

# Проверяем данные, затем добавляем лишних и снова проверяем
check_test_dataset(\%shut_db, 'SHUT: replace_test_data - check_test_dataset');
lives_ok {
    do_insert_into_table(SHUT(shard => 1), 'half_sharded_table', $extra_row);
} 'SHUT: replace_test_data - inserted more data in half_sharded_table';
ok( !(Yandex::DBUnitTest::_check_test_dataset(\%shut_db))[0], "SHUT: check_test_dataset failed on extra data in half_sharded_table");

# Заход 2, другая таблица
_replace_sharded_data();
check_test_dataset(\%shut_db, 'SHUT: replace_test_data - check_test_dataset');
lives_ok {
    do_insert_into_table(SHUT(shard => 'all'), 'sharded_table', $extra_row);
} 'SHUT: replace_test_data - inserted more data in sharded_table';
ok( !(Yandex::DBUnitTest::_check_test_dataset(\%shut_db))[0], "SHUT: check_test_dataset failed on extra data in sharded_table");

# заход 3. "А теперь оба окурка - двумя ногами - оп, оп, оп"
_replace_sharded_data();
check_test_dataset(\%shut_db, 'SHUT: replace_test_data - check_test_dataset');
lives_ok {
    do_insert_into_table(SHUT(shard => 1), 'half_sharded_table', $extra_row);
    do_insert_into_table(SHUT(shard => 'all'), 'sharded_table', $extra_row);
} 'SHUT: replace_test_data - inserted more data in both tables';
ok( !(Yandex::DBUnitTest::_check_test_dataset(\%shut_db))[0], "SHUT: check_test_dataset failed on extra data in both tables");

# Удаляем таблицы
lives_ok {
    do_sql(SHUT(shard => 'all'), q/DROP TABLE half_sharded_table/);
    do_sql(SHUT(shard => 'all'), q/DROP TABLE sharded_table/);
    %Yandex::DBUnitTest::CREATED_TABLES = ();
} 'SHUT: DROP TABLEs';
################################################################################
# Шардированная база - одинаковые данные в шардах - ручное создание/заполнение
lives_ok {
    _create_table(SHUT(shard => 'all'), 'sharded_table');
    foreach my $row ( @{ $shut_all_db{sharded_table}->{rows}->{all} } ) {
        do_insert_into_table(SHUT(shard => 'all'), 'sharded_table', $row);
    }
} 'SHUT: all shards - manually inserted data';

# Проверяем, что шардированные данные лежат в таблице во всех шардах
_check_sharded_to_all_data('manually inserted data');

# Удаляем таблицу
lives_ok {
    do_sql(SHUT(shard => 'all'), q/DROP TABLE sharded_table/);
    %Yandex::DBUnitTest::CREATED_TABLES = ();
} 'SHUT: DROP TABLE sharded_table';
################################################################################
# Шардированная база - одинаковые данные в шардах - создание/заполнение через init_test_dataset
lives_ok { init_test_dataset(\%shut_all_db); } 'SHUT: all shards - init_test_dataset';
_check_sharded_to_all_data('init_test_dataset');
################################################################################
# Шардированная база - одинаковые данные - портим данные в одном из шардов, восстаналиваем через init_test_dataset
lives_ok {
    do_update_table(SHUT(shard => 1), 'sharded_table', {domain_login => 'broken login'}, where => {uid => 9})
} 'SHUT: all shards - modified data in 1 shard';
ok( !(Yandex::DBUnitTest::_check_test_dataset(\%shut_all_db))[0], "SHUT: all shards - modified data in 1 shard - check_test_dataset failed");

lives_ok { init_test_dataset(\%shut_all_db); } 'SHUT: all shards - init_test_dataset';
check_test_dataset(\%shut_all_db, 'SHUT: all shards - init_test_dataset - check_test_dataset');
lives_ok {
    do_update_table(SHUT(shard => 2), 'sharded_table', {domain_login => 'broken login'}, where => {uid => 9})
} 'SHUT: all shards - modified data in 2 shard';

# Проверяем, что не данные не совпали
ok( !(Yandex::DBUnitTest::_check_test_dataset(\%shut_all_db))[0], "SHUT: all shards - modified data in 2 shard - check_test_dataset failed");
################################################################################
# Шардированная база - одинаковые данные - восстаналиваем данные через replace_test_data, потом добавляем лишних
lives_ok {
    replace_test_data(SHUT(shard => 'all'), 'sharded_table', $shut_all_db{sharded_table}->{rows});
} 'SHUT: all shards - replace_test_data';
check_test_dataset(\%shut_all_db, 'SHUT: all shards - replace_test_data - check_test_dataset');

lives_ok {
    do_insert_into_table(SHUT(shard => 1), 'sharded_table', $extra_row);
} 'SHUT: all shards - inserted extra data in 1 shard';
ok( !(Yandex::DBUnitTest::_check_test_dataset(\%shut_all_db))[0], "SHUT: all shards - inserted extra data in 1 shard - check_test_dataset failed");

# Заход 2
lives_ok {
    replace_test_data(SHUT(shard => 'all'), 'sharded_table', $shut_all_db{sharded_table}->{rows});
} 'SHUT: all shards - replace_test_data';
check_test_dataset(\%shut_all_db, 'SHUT: all shards - replace_test_data - check_test_dataset');
lives_ok {
    do_insert_into_table(SHUT(shard => 2), 'sharded_table', $extra_row);
} 'SHUT: all shards - inserted extra data in 2 shard';
ok( !(Yandex::DBUnitTest::_check_test_dataset(\%shut_all_db))[0], "SHUT: all shards - inserted extra data in 2 shard - check_test_dataset failed");

done_testing();
