#!/usr/bin/perl -w

# $Id$

=head1 DBUnitTest.t

    Юнит-тесты модуля для юнит-тестов.
    Звучит странно, но проверить что всё работает как задумано - никогда не бывает лишним.
    Можно воспринимать эти тесты как примеры разных кейсов для:
        %dataset, описывающего содержимое баз данных
        &init_test_dataset - создающей таблицы и заполняющей их данными
        &replace_test_data - заменяющей в одной(!) таблице (но во всех шардах) данные на заданные
        &check_test_dataset - проверяющей, что все содержимое %dataset действительно существует в правильных базах/шардах

=cut

use strict;
use warnings;

use Test::More tests => 16;
use Test::Deep;
use Test::Exception;

use Yandex::DBTools;
use Yandex::DBUnitTest qw/:all/;

use utf8;

$Yandex::DBTools::DONT_SEND_LETTERS = 1;

my $create_string = ['uid:bigint(20):pk', 'domain_login', 'manager_private_email', 'is_developer:tinyint(1)'];

##############################################################
##### Первая группа тестов - только нешардированная база #####
##############################################################
my %ut_db = (
    single_table => {
        original_db => UT,
        create_string => $create_string,
        rows => [
            {uid => 9, domain_login => 'root', manager_private_email => 'root@yandex-team.ru', is_developer => 1},
            {uid => 10, domain_login => 'test', manager_private_email => 'noreply@yandex-team.ru', is_developer => 0},
            {uid => 11, domain_login => 'unknown', manager_private_email => 'vasia_pupkin@mail.ru', is_developer => 0},
        ],
    },
);

# Тестовый набор данных
my $test_bag_1 = bag(map {superhashof($_)} @{ $ut_db{single_table}->{rows} });

# Ручное создание таблицы
sub _create_table{
    my ($db, $table) = @_;
    do_sql($db, q/
        CREATE TABLE /.$table.q/ (
            uid bigint not null  primary key
            , domain_login varchar(100)
            , manager_private_email varchar(100)
            , is_developer tinyint
        ) ENGINE=MyISAM/);
}
# Ручная выборка данных из таблицы
sub _real_rows{
    my ($db, $table) = @_;
    return get_all_sql($db, q/SELECT uid, manager_private_email, is_developer, domain_login FROM /.$table);
}
# Проверка данных двумя способами
sub _check_not_sharded_data {
    my $msg_prefix = shift;

    cmp_deeply(_real_rows(UT, 'single_table'), $test_bag_1, "UT: $msg_prefix - manually checked");
    check_test_dataset(\%ut_db, "UT: $msg_prefix - check_test_dataset");
}

################################################################################
# Нешардированная база - создаем и заполняем таблицу вручную
_create_table(UT, 'single_table');
foreach my $row ( @{ $ut_db{single_table}->{rows} } ) {
    do_insert_into_table(UT, 'single_table', $row);
}

# Проверяем
_check_not_sharded_data('manually inserted data');

# Удаляем таблицу
lives_ok {
    do_sql(UT, q/DROP TABLE single_table/);
    %Yandex::DBUnitTest::CREATED_TABLES = ();
} 'UT: DROP TABLE single_table';

################################################################################
# Нешардированная база - создаем и заполняем таблицу через init_test_dataset
lives_ok { init_test_dataset(\%ut_db) } 'UT: init_test_dataset';

# Проверяем
_check_not_sharded_data('init_test_dataset');

# Удаляем таблицу
lives_ok {
    do_sql(UT, q/DROP TABLE single_table/);
    %Yandex::DBUnitTest::CREATED_TABLES = ();
} 'UT: DROP TABLE single_table';

################################################################################
# Нешардированная база - создаем таблицу вручную, заполняем через replace_test_data
_create_table(UT, 'single_table');

# Заполняем неправильными (временными) данными
do_insert_into_table(UT, 'single_table', {uid => 42, domain_login => 'fake', manager_private_email => 'fake@universe', is_developer => -1});

# Проверяем, что в таблице не те данные, которые нас интересуют
ok(!eq_deeply(_real_rows(UT, 'single_table'), $test_bag_1), 'UT: replace_test_data - wrong data before replacing');

# Инициализируем таблицу правильными данными
lives_ok { replace_test_data(UT, 'single_table', $ut_db{single_table}->{rows}) } 'UT: replace_test_data';

# Проверяем
_check_not_sharded_data('replace_test_data');

# Удаляем таблицу
do_sql(UT, q/DROP TABLE single_table/);

########################
##### Прочие тесты #####
########################
# Если база нешардирована, а в dataset данные как для шардированной - то это точно не порядок
my %other_db = (
    strange_table => {
        original_db => UT,
        create_string => $create_string,
        rows => {
            all => $ut_db{single_table}->{rows},
        }
    },
);
dies_ok {
    init_test_dataset(\%other_db);
} q/UT: can't initialize not sharded DB with sharded data/;

# пойдем обходным путем - создадим табличку руками и попробуем проверить.
lives_ok {
    _create_table(UT, 'strange_table');
    foreach my $row ( @{ $other_db{strange_table}->{rows}->{all} } ) {
        do_insert_into_table(UT, 'strange_table', $row);
    }
} 'UT: manually created strange table with sharded data in not sharded DB';

ok( !(Yandex::DBUnitTest::_check_test_dataset(\%other_db))[0], "UT: manually created strange table with sharded data in not sharded DB - check_test_dataset failed");
# Удаляем таблицу
do_sql(UT, q/DROP TABLE strange_table/);

# Проверяем что не падаем, если имя поля в таблице - зарезервированное ключевое слово MySQL
my $keyword_check_dataset = {
    test_table => {
        original_db => UT,
        create_string => ['id:bigint(20):pk', 'precision', 'separator', 'real:tinyint(1)'],
        rows => [
            {id => 15, precision => 'bad',     separator => '|||', real => 1},
            {id => 16, precision => 'good',    separator => '---', real => 0},
            {id => 17, precision => 'unknown', separator => '///', real => 0},
        ],
    }
};
lives_ok {
    init_test_dataset($keyword_check_dataset);
    check_test_dataset($keyword_check_dataset, "UT: MySQL reserved keywords in test dataset");
} "DBUnitTest doesn't dies on queries with MySQL reserved keywords";
