#!/usr/bin/perl -w

# $Id$

=head1 DBUnitTest.t

    Юнит-тесты модуля для юнит-тестов.
    Звучит странно, но проверить что всё работает как задумано - никогда не бывает лишним.
    Можно воспринимать эти тесты как примеры разных кейсов для:
        %dataset, описывающего содержимое баз данных
        &init_test_dataset - создающей таблицы и заполняющей их данными
        &replace_test_data - заменяющей в одной(!) таблице (но во всех шардах) данные на заданные
        &check_test_dataset - проверяющей, что все содержимое %dataset действительно существует в правильных базах/шардах

=cut

use strict;
use warnings;

use Test::More;
use Test::Deep;
use Test::Exception;

use Yandex::DBTools;
use Yandex::DBUnitTest qw/:all/;

use POSIX qw(:sys_wait_h);

use utf8;

$Yandex::DBTools::DONT_SEND_LETTERS = 1;

my $create_string = ['uid:bigint(20):pk', 'domain_login', 'manager_private_email', 'is_developer:tinyint(1)'];
my $create_string2 = ['uid:bigint(20)', 'setting', 'value', 'unique:uid,setting'];

####################################################################
##### Третья группа тестов - сочетание шардированных и нет баз #####
####################################################################
my $ut_db_rows = [
    {uid => 9, domain_login => 'root', manager_private_email => 'root@yandex-team.ru', is_developer => 1},
    {uid => 10, domain_login => 'test', manager_private_email => 'noreply@yandex-team.ru', is_developer => 0},
    {uid => 11, domain_login => 'unknown', manager_private_email => 'vasia_pupkin@mail.ru', is_developer => 0},
];
my %combined_db = (
    single_table => {
        original_db => UT,
        create_string => $create_string2,
        rows => [
            {uid => 2, setting => 'colour', value => 'red'},
            {uid => 2, setting => 'font', value => 'Times'},
            {uid => 3, setting => 'currency', value => 'USD'},
            {uid => 3, setting => 'colour', value => 'blue'},
            {uid => 4, setting => 'currency', value => 'RUR'},
            {uid => 5, setting => 'colour', value => 'orange'},
            {uid => 5, setting => 'currency', value => 'UAH'},
        ],
    },
    # Табличка с данными без указания шарда - должны "лечь" в первый шард
    half_sharded_table => {
        original_db => SHUT(shard => 'all'),
        create_string => $create_string,
        rows => $ut_db_rows,
    },
    # Табличка с разными данными в разных шардах (во втором шарде те же, данные что в первом примере)
    sharded_table => {
        original_db => SHUT(shard => 'all'),
        create_string => $create_string,
        rows => {
            1 => $ut_db_rows,
            2 => [
                {uid => 2, domain_login => 'zzz', manager_private_email => 'abuse@yandex-team.ru', is_developer => 0},
                {uid => 3, domain_login => 'aaa', manager_private_email => 'robot@yandex-team.ru', is_developer => 1},
                {uid => 4, domain_login => 'q', manager_private_email => 'q@rambler.ru', is_developer => 0},
            ],
        },
    },
);

# Тестовые данные
my $test_bag_1 = bag(map {superhashof($_)} @$ut_db_rows);
my $test_bag_2 = bag(map {superhashof($_)} @{ $combined_db{sharded_table}->{rows}->{2} });
my $test_bag_3 = bag(map {superhashof($_)} @{ $combined_db{single_table}->{rows} });
my $extra_row = {uid => 42, domain_login => 'zombie', manager_private_email => 'zombie@hotmail.com', is_developer => -1};
my $another_extra_row = {uid => 24, setting => 'animal', value => 'elephant'};

# Ручное создание таблицы
sub _create_table{
    my ($db, $table) = @_;
    do_sql($db, q/
        CREATE TABLE /.$table.q/ (
            uid bigint not null  primary key
            , domain_login varchar(100)
            , manager_private_email varchar(100)
            , is_developer tinyint
        ) ENGINE=MyISAM/);
}
sub _create_another_table{
    my ($db, $table) = @_;
    do_sql($db, q/
        CREATE TABLE /.$table.q/ (
            uid bigint
            , setting varchar(100)
            , value varchar(100)
            , unique (uid,setting)
            ) ENGINE=MyISAM/);
};
# Ручная выборка данных из таблицы
sub _real_rows{
    my ($db, $table) = @_;
    return get_all_sql($db, q/SELECT uid, manager_private_email, is_developer, domain_login FROM /.$table);
}
sub _another_real_rows{
    my ($db, $table) = @_;
    return get_all_sql($db, q/SELECT uid, setting, value FROM /.$table);
}
sub _uids{
    my $shard = shift;
    return [ map {$_->{uid}} @{ $combined_db{sharded_table}->{rows}->{$shard} } ];
}
# Проверка данных
sub _check_table_not_exists{
    my $msg_prefix = shift;
    open(my $stderr, '>&', \*STDERR);
    open(STDERR, '>/dev/null');    # Подавляем сообщения об ошибках системы

    # Проверяем что таблица half_sharded_table не попала в нешардированную базу
    dies_ok { exec_sql(UT, q/DESC half_sharded_table/) } qq/$msg_prefix - half_sharded_table should not exists in UT/;
    # Таблица sharded_table в нешардированную базу попасть не должна была ни при каком раскладе
    dies_ok { exec_sql(UT, q/DESC sharded_table/) } qq/$msg_prefix - sharded_table should not exists in UT/;
    # Таблица single_table в шардированную базу попасть не должна была ни при каком раскладе
    dies_ok { exec_sql(SHUT(shard => 'all'), q/DESC single_table/) } qq/$msg_prefix - single_table should not exists in SHUT/;


    *STDERR = $stderr;  # Возвращаем STDERR
}
sub _check_combined_data{
    my $msg_prefix = shift;

    # Проверяем нешардированные данные
    cmp_deeply(_another_real_rows(UT, 'single_table'), $test_bag_3, "UT&SHUT: $msg_prefix - manually checked");
    cmp_deeply(_real_rows(SHUT(shard => 1), 'half_sharded_table'), $test_bag_1, "UT&SHUT: $msg_prefix - manually checked (not sharded data found in 1 shard)");
    cmp_deeply(_real_rows(SHUT(shard => 2), 'half_sharded_table'), [], "UT&SHUT: $msg_prefix - manually checked (not sharded data not exists in 2 shard)");
    
    # Проверяем, что шардированные данные там, где нужно
    cmp_deeply(_real_rows(SHUT(shard => 1), 'sharded_table'), $test_bag_1, "UT&SHUT: $msg_prefix - manually checked (sharded data1 found in 1 shard)");
    cmp_deeply(_real_rows(SHUT(shard => 2), 'sharded_table'), $test_bag_2, "UT&SHUT: $msg_prefix - manually checked (sharded data2 found in 2 shard)");
    
    # Проверяем, что данные, предназнаечнные для одного шарда не попали в другой
    ok(!get_one_field_sql(SHUT(shard => 1), [q/SELECT count(*) FROM sharded_table WHERE/, {uid => _uids(2)}]), "UT&SHUT: $msg_prefix - manually checked (sharded data1 not found in 1 shard)");
    ok(!get_one_field_sql(SHUT(shard => 2), [q/SELECT count(*) FROM sharded_table WHERE/, {uid => _uids(1)}]), "UT&SHUT: $msg_prefix - manually checked (sharded data2 not found in 2 shard)");

    # Проверяем все сразу функцией
    check_test_dataset(\%combined_db, "UT&SHUT: $msg_prefix - check_test_dataset");
}
sub _replace_combined_data{
    lives_ok {
        replace_test_data(UT, 'single_table', $combined_db{single_table}->{rows});
        replace_test_data(SHUT(shard => 'all'), 'half_sharded_table', $combined_db{half_sharded_table}->{rows});
        replace_test_data(SHUT(shard => 'all'), 'sharded_table', $combined_db{sharded_table}->{rows});
    } 'UT&SHUT: replace_test_data';
}

################################################################################
# Разные базы - создаем и заполняем таблицы вручную
lives_ok {
    _create_another_table(UT, 'single_table');
    foreach my $row ( @{ $combined_db{single_table}->{rows} } ) {
        do_insert_into_table(UT, 'single_table', $row);
    }
    _create_table(SHUT(shard => 'all'), 'half_sharded_table');
    foreach my $row ( @{ $combined_db{half_sharded_table}->{rows} } ) {
        do_insert_into_table(SHUT(shard => 1), 'half_sharded_table', $row);
    }
    _create_table(SHUT(shard => 'all'), 'sharded_table');
    foreach my $shard (keys %{$combined_db{sharded_table}->{rows}}) {
        foreach my $row ( @{ $combined_db{sharded_table}->{rows}->{$shard} } ) {
            do_insert_into_table(SHUT(shard => $shard), 'sharded_table', $row);
        }
    }
} 'UT&SHUT: manually inserted data';

# Проверяем что таблички НЕ появились там, где не должны
_check_table_not_exists('UT&SHUT: manually inserted data');

# Проверяем
_check_combined_data('manually inserted data');

# Удаляем таблицы
lives_ok {
    do_sql(UT, q/DROP TABLE single_table/);
    do_sql(SHUT(shard => 'all'), q/DROP TABLE half_sharded_table/);
    do_sql(SHUT(shard => 'all'), q/DROP TABLE sharded_table/);
    %Yandex::DBUnitTest::CREATED_TABLES = ();
} 'UT&SHUT: DROP TABLEs';

################################################################################
# Разные базы - создаем и заполняем таблицы через init_test_dataset
lives_ok { init_test_dataset(\%combined_db) } 'UT&SHUT: init_test_dataset';

# Проверяем что таблички НЕ появились там, где не должны
_check_table_not_exists('UT&SHUT: init_test_dataset');

# Проверяем
_check_combined_data('init_test_dataset');

################################################################################
# Разные базы - данные восстанавливаются через init_test_dataset, а затем данные изменяются
lives_ok {
    do_update_table(SHUT(shard => 'all'), 'half_sharded_table', {domain_login => 'broken login'}, where => {uid => [3, 9, 11]})
} 'UT&SHUT: modified data in half_sharded_table';
ok( !(Yandex::DBUnitTest::_check_test_dataset(\%combined_db))[0], "UT&SHUT: modified data in half_sharded_table - check_test_dataset failed");

# Восстанавливаем данные, и портим в другой таблице
lives_ok { init_test_dataset(\%combined_db) } 'UT&SHUT: init_test_dataset';
check_test_dataset(\%combined_db, "UT&SHUT: init_test_dataset - check_test_dataset");
lives_ok {
    do_update_table(SHUT(shard => 'all'), 'sharded_table', {domain_login => 'bad login'}, where => {uid => [2, 4, 10]})
} 'UT&SHUT: modified data in sharded_table';
ok( !(Yandex::DBUnitTest::_check_test_dataset(\%combined_db))[0], "UT&SHUT: modified data in sharded_table - check_test_dataset failed");

# Восстанавливаем данные, и портим в третьей таблице
lives_ok { init_test_dataset(\%combined_db) } 'UT&SHUT: init_test_dataset';
check_test_dataset(\%combined_db, "UT&SHUT: init_test_dataset - check_test_dataset");
lives_ok {
    do_update_table(UT, 'single_table', {value => 'green'}, where => {setting => 'colour', value => ['red', 'blue']})
} 'UT&SHUT: modified data in single_table';
ok( !(Yandex::DBUnitTest::_check_test_dataset(\%combined_db))[0], "UT&SHUT: modified data in single_table - check_test_dataset failed");

################################################################################
# Разные базы - данные восстанавливаются через replace_test_data, а затем добавляются лишние
_replace_combined_data();

# Проверяем данные, затем добавляем лишних и снова проверяем
check_test_dataset(\%combined_db, 'UT&SHUT: replace_test_data - check_test_dataset');
lives_ok { do_insert_into_table(SHUT(shard => 1), 'half_sharded_table', $extra_row); } 'UT&SHUT: inserted extra data in half_sharded_table';
ok( !(Yandex::DBUnitTest::_check_test_dataset(\%combined_db))[0], "UT&SHUT: inserted extra data in half_sharded_table - check_test_dataset failed");

# Заход 2, другая таблица
_replace_combined_data();
check_test_dataset(\%combined_db, 'UT&SHUT: replace_test_data - check_test_dataset');
lives_ok { do_insert_into_table(SHUT(shard => 2), 'sharded_table', $extra_row); } 'UT&SHUT: inserted extra data in sharded_table';
ok( !(Yandex::DBUnitTest::_check_test_dataset(\%combined_db))[0], "UT&SHUT: inserted extra data in sharded_table - check_test_dataset failed");

# Заход 3, третья таблица
_replace_combined_data();
check_test_dataset(\%combined_db, 'UT&SHUT: replace_test_data - check_test_dataset');
lives_ok { do_insert_into_table(UT, 'single_table', $another_extra_row); } 'UT&SHUT: inserted extra data in single_table';
ok( !(Yandex::DBUnitTest::_check_test_dataset(\%combined_db))[0], "UT&SHUT: inserted extra data in single_table - check_test_dataset failed");

################################################################################
# Форки - базу и созданные таблицы удаляем только там, где создали
my %parent_dataset = (
    "DBUnitTest_combined_parent_$$" => {
        original_db => UT,
        create_string => $create_string,
    },
);
my $child_table_name = "DBUnitTest_combined_child_$$";
my %child_dataset = (
    $child_table_name => {
        original_db => UT,
        create_string => $create_string,
    },
);
init_test_dataset(\%parent_dataset);
my $child_pid = fork();
if ($child_pid) {
    waitpid($child_pid, 0);
    # дитё не должно умереть и вернуть нулевой статус
    ok(WIFEXITED($?) && WEXITSTATUS($?) == 0, 'child exited successfully');
} else {
    # создаём таблицу и выходим. при этом не должны удалить ничего из созданного в родителе, только эту таблицу.
    init_test_dataset(\%child_dataset);
    exit 0;
}
# родительская таблица должна остаться
check_test_dataset(\%parent_dataset, 'check_test_dataset for parent');
# а вот таблица, созданная в чайлде, должна была удалиться вместе с чайлдом
ok(!is_table_exists(UT, $child_table_name), "table created from child dies with it's exit");

done_testing;
