#!/usr/bin/perl -w

use strict;
use warnings;

use Test::More;
use Test::Deep;
use Test::Exception;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::DBUnitTest qw/:all/;
use Cwd qw/abs_path/;

use utf8;

$Yandex::DBTools::DONT_SEND_LETTERS = 1;

%Yandex::DBShards::SHARD_KEYS = (
    ClientID => {
        table => 'shard_client_id',
        shard_key => 'shard',
        not_zero  => 1,
    },
    uid => {
        table => "shard_uid",
        chain_key => "ClientID",
    },
    cid => {
        table => "shard_inc_cid",
        autoinc => 1,
        chain_key => "ClientID",
    },
);
$Yandex::DBShards::SHARD_DB = UT;

BEGIN { use_ok('Yandex::Overshard', 'enrich_data') };

my %db = (
    campaigns => {
        original_db => SHUT(shard => 'all'),
        create_string => [ qw/cid:pk ManagerUID/ ],
        rows => {
            1 => [
                { cid => 1, ManagerUID => 1 },
                { cid => 2, ManagerUID => 1 },
                { cid => 3, ManagerUID => 2 },
            ],
            2 => [
                { cid => 4, ManagerUID => 1 },
                { cid => 5, ManagerUID => 2 },
                { cid => 6, ManagerUID => undef },
            ],
        },
    },
    users => {
        original_db => SHUT(shard => 'all'),
        create_string => [ qw/uid:int:pk login / ],
        rows => {
            1 => [
                { uid => 1, login => 'manager1' },
            ],
            2 => [
                { uid => 2, login => 'manager2' },
                { uid => 3, login => 'manager3' },
            ],
        },
    },
    shard_client_id => {
        original_db => UT,
        create_string => [ 'ClientID:int:pk', 'shard:int:not_null' ],
        rows => [
            { ClientID => 1, shard => 1 },
            { ClientID => 2, shard => 2 },
            { ClientID => 3, shard => 2 },
            { ClientID => 4, shard => 1 },
            { ClientID => 5, shard => 2 },
        ],
    },
    shard_uid => {
        original_db => UT,
        create_string => [ 'uid:int:pk', 'ClientID:int:not_null' ],
        rows => [
            { ClientID => 1, uid => 1 },
            { ClientID => 2, uid => 2 },
            { ClientID => 3, uid => 3 },
            { ClientID => 4, uid => 4 },
            { ClientID => 5, uid => 5 },
        ],
    },
    shard_inc_cid => {
        original_db => UT,
        create_string => [ 'cid:int:pk:auto', 'ClientID:int:not_null' ],
        rows => [
            { ClientID => 4, cid => 1 },
            { ClientID => 4, cid => 2 },
            { ClientID => 4, cid => 3 },
            { ClientID => 5, cid => 4 },
            { ClientID => 5, cid => 5 },
            { ClientID => 5, cid => 6 },
        ],
        
    },
);

init_test_dataset(\%db);
get_shard(uid => 1);

my @tests = (
    # [ $query, $opt, $sub, $result_data, $test_name ]
    [ 
        "select c.cid, c.ManagerUID from campaigns c where cid in (1,4)", 
        { using => 'ManagerUID', map_fields => { ManagerLogin => 'ManagerLogin' } },
        sub { 
            my $uids = shift;
            return get_hashes_hash_sql(SHUT(uid => $uids), [ "select uid as ManagerUID, login as ManagerLogin from users u", where => { uid => SHARD_IDS }]);
        },
        [ { cid => 1, ManagerUID => 1, ManagerLogin => 'manager1' }, { cid => 4, ManagerUID => 1, ManagerLogin => 'manager1' } ] ,
        'Camps in 2 shards, same manager'
    ],
    [
      "select c.cid, c.ManagerUID from campaigns c where cid in (4, 6)",
      { using => 'ManagerUID', map_fields => { ManagerLogin => 'ManagerLogin' } },
        sub { 
            my $uids = shift;
            return get_hashes_hash_sql(SHUT(uid => $uids), [ "select uid as ManagerUID, login as ManagerLogin from users u", where => { uid => SHARD_IDS }]);
        },
      [ { cid => 4, ManagerUID => 1, ManagerLogin => 'manager1' }, { cid => 6, ManagerUID => undef, ManagerLogin => undef } ],
      'camp with manager = null'
    ],
    [ 
        "select c.cid, c.ManagerUID from campaigns c where cid in (1,4)", 
        { using => 'ManagerUID', map_fields => { ManagerLogin => 'ManagerLogin' } },
        sub { 
            my $uids = shift;
            return get_all_sql(SHUT(uid => $uids), [ "select uid as ManagerUID, login as ManagerLogin from users u", where => { uid => SHARD_IDS }]);
        },
        [ { cid => 1, ManagerUID => 1, ManagerLogin => 'manager1' }, { cid => 4, ManagerUID => 1, ManagerLogin => 'manager1' } ] ,
        'sub returns array of hashes'
    ],
);
my @tests_die = (
    [
        "select c.cid, c.ManagerUID from campaigns c where cid = 1",
        { using => 'ManagerUID' },
        'conflicting names'
    ],
    [
        "select c.cid, c.ManagerUID from campaigns c where cid = 1",
        { using => 'ManagerUID', invalid_opt => 1 },
        'invalid opt'
    ],
);

for my $t (@tests) {
    my ($query, $opt, $sub, $res, $name) = @$t;
    my $data = get_all_sql(SHUT(shard => 'all'), $query);
    enrich_data($data, %$opt, $sub);
    cmp_deeply($data, $res, $name);
}

for my $t (@tests_die) {
    my $data = get_all_sql(SHUT(shard => 'all'), $t->[0]);
    dies_ok sub { enrich_data($data, %{$t->[1]}, sub {
        my $uids = shift;
        return get_hashes_hash_sql(SHUT(uid => $uids), [ "select uid as ManagerUID, login as ManagerLogin, 0 as cid 
            from users u", where => { uid => SHARD_IDS }]);
    }) }, $t->[2];
}


done_testing();
