#!/usr/bin/perl -w

use strict;
use warnings;

use Test::More;
use Test::Deep;
use Test::Exception;

use Data::Dumper;

use Yandex::DBTools;
use Yandex::DBUnitTest qw/:all/;

use Yandex::ListUtils qw/xsort/;

use utf8;

$Yandex::DBTools::DONT_SEND_LETTERS = 1;

BEGIN { use_ok('Yandex::Overshard', 'overshard') };


do_sql(UT, "drop table if exists test_overshard");
do_sql(UT, "create temporary table test_overshard (
    i1 int(10),
    i2 int(10),
    name varchar(32)
) Engine = MyISAM");

do_mass_insert_sql(UT, "INSERT INTO test_overshard (i1, i2, name) VALUES %s",
    [ map { [ int(rand(1000))+1, int(rand(10000))+1, (int($_/5) % 2 ? 'S' : 's')."tr_".(int($_/5)) ] } (1 .. 10)]
);

# имитируем запрос из двух шардов
sub _get_data
{
    my $query = shift;
    my $r1 = get_all_sql(UT, "$query limit 5 offset 0");
    my $r2 = get_all_sql(UT, "$query limit 5 offset 5");
    push @$r1, @$r2;
    return $r1;
}

my $sql = get_all_sql(UT, "select name, sum(i1) as i1, min(i2) as i2 from test_overshard group by name order by name");
my $shard = overshard group => 'name', sum => 'i1', min => 'i2', order => 'name', _get_data("select name, i1, i2 from test_overshard");
cmp_deeply($sql, $shard, "group, order, min, sum");

$sql = get_all_sql(UT, "select name, sum(i1) as i1, min(i2) as i2 from test_overshard group by name order by name desc");
$shard = overshard group => 'name', sum => 'i1', min => 'i2', order => '-name', _get_data("select name, i1, i2 from test_overshard");
cmp_deeply($sql, $shard, "group, order desc, min, sum");

$sql = get_all_sql(UT, "select name from test_overshard order by name limit 3");
$shard = overshard limit => 3, order => 'name', _get_data("select name from test_overshard");
cmp_deeply($sql, $shard, "limit");

$sql = get_all_sql(UT, "select name from test_overshard order by name limit 3 offset 7");
$shard = overshard limit => 3, offset => 7, order => 'name', _get_data("select name from test_overshard");
cmp_deeply($sql, $shard, "limit, offset");

$sql = get_all_sql(UT, "select count(*) as cnt, name from test_overshard group by name order by name");
$shard = overshard group => 'name', count => 'cnt', order => 'name', _get_data("select 0 as cnt, name from test_overshard");
cmp_deeply($sql, $shard, 'count');

# похоже, что mysql плюет на сортировку в group_concat :(
#$sql = get_all_sql(UT, "select name, group_concat(i1) as i1_concat from test_overshard group by name order by name, i1");
#$shard = overshard group => 'name', group_concat_array => 'i1_concat', order => [ 'name', '-i1_concat' ], 
#    _get_data("select name, i1 as i1_concat from test_overshard");
#
#cmp_deeply($sql, $shard, 'group_concat');

$sql = get_all_sql(UT, "select name, floor(i1/3) as i1, min(i2) as i2 from test_overshard group by name, floor(i1/3) order by name, i1");
$shard = overshard group => [qw/name i1/], min => 'i2', order => [qw/name i1:num/], _get_data("select name, floor(i1/3) as i1, i2 from test_overshard");
cmp_deeply($sql, $shard, 'multiple group');

$sql = get_all_sql(UT, "select name, floor(i1/3) as i1, min(i2) as i2 from test_overshard group by name, floor(i1/3) order by name, i1 desc");
$shard = overshard group => [qw/name i1/], min => 'i2', order => [qw/name -i1:num/], _get_data("select name, floor(i1/3) as i1, i2 from test_overshard");
cmp_deeply($sql, $shard, 'multiple group, order desc');

$sql = get_all_sql(UT, "select name, max(-i1) i1, min(-i2) i2 from test_overshard order by name");
$shard = overshard group => [], max => 'i1', min => 'i2', order => 'name', 
    _get_data("select name, max(-i1) as i1, min(-i2) as i2 from test_overshard");
cmp_deeply($sql, $shard, 'min, max on negative; empty group');

$sql = get_all_sql(UT, "select min(name) as min_name, max(name) as max_name from test_overshard");
$shard = overshard group => [], minstr => 'min_name', maxstr => 'max_name', 
    _get_data("select name as min_name, name as max_name from test_overshard");
cmp_deeply($sql, $shard, 'min, max on strings');

$sql = get_all_sql(UT, "select min(name collate utf8_bin) as min_name, max(name collate utf8_bin) as max_name from test_overshard");
$shard = overshard group => [], minstr_cs => 'min_name', maxstr_cs => 'max_name', 
    _get_data("select name as min_name, name as max_name from test_overshard");
cmp_deeply($sql, $shard, 'min, max on strings (case sensitive)');

dies_ok sub { overshard unknown_key => [], [] },  'dies on invalid key';
dies_ok sub { overshard group => {  }, [] },  'dies on invalid key (not array/string)';

$sql = get_all_sql(UT, 'select name, NULL as n from test_overshard group by name order by name');
$shard = overshard group => 'name', order => 'name', _get_data("select name, NULL as n from test_overshard");
cmp_deeply($sql, $shard, 'null in group values');

$sql = get_all_sql(UT, 'select if(i1 <= 5, i1, null) as i1, count(i1) as cnt_i1 from test_overshard group by 1 order by 1');
$shard = overshard group => 'i1', order => 'i1', count => 'cnt_i1', _get_data('select if(i1 <= 5, i1, null) as i1, 0 as cnt_i1 from test_overshard');
cmp_deeply($sql, $shard, 'null in group keys');

$sql = get_all_sql(UT, 'select name, avg(i1) as i1_avg from test_overshard group by name order by name');
$shard = overshard group => ['name'], avg => 'i1_avg', order => 'name', _get_data('select name, avg(i1) i1_avg from test_overshard group by name');
# need to turn strings to float values: "2.500" -> 2.5
$sql = [ map { $_->{i1_avg} += 0; $_ } @$sql ];
cmp_deeply($sql, $shard, 'avg');

for my $limit_offset (
    [10, 9], [10, 10], [10, 100], [100, 100],
    [0, 100], [100, 0],
) {
    my ($limit, $offset) = @$limit_offset;
    for my $cs (0, 1) {
        $sql = get_all_sql(UT, 'select name from test_overshard order by name '.($cs ? 'collate utf8_bin' : '').' limit ? offset ?', $limit, $offset);
        $shard = overshard limit => $limit, offset => $offset, order => 'name'.($cs ? ':str_cs' : ''), _get_data('select name from test_overshard');
        cmp_deeply($sql, $shard, "limit $limit, $offset, cs:$cs");
    }
}

# distinct
$sql = get_all_sql(UT, 'select distinct name from test_overshard order by name');
$shard = overshard distinct => 1, order => 'name', _get_data('select name from test_overshard order by name');
cmp_deeply($sql, $shard, 'distinct');


# group_array, group_array_distinct, group_concat_distinct
my $test_dataset = [
    {key1 => "value1", key2 => 123, key3 => 456, key4 => "a"},
    {key1 => "value1", key2 => 666, key3 => 777, key4 => "b"},
    {key1 => "value2", key2 => 777, key3 => 888, key4 => "c"},
    {key1 => "value2", key2 => 777, key3 => 999, key4 => "d"},
    {key1 => "zzz", key2 => 999, key3 => 999, key4 => "e,f,f,e"},
    {key1 => "zzz", key2 => 999, key3 => 999, key4 => "f"},
    {key1 => "zzz", key2 => 999, key3 => 999, key4 => "f"},
];

{
local $Yandex::Overshard::GROUP_CONCAT_SEPARATOR = undef;
cmp_bag(
    overshard(group => [qw/key1 key2/], group_array => [qw/key3/], $test_dataset),
    [
        {key1 => "value1", key2 => 123, key3 => [456], key4 => "a"},
        {key1 => "value1", key2 => 666, key3 => [777], key4 => "b"},
        {key1 => "value2", key2 => 777, key3 => bag(888,999), key4 => re(q/^(?:c|d)$/)},
        {key1 => "zzz", key2 => 999, key3 => bag(999,999,999), key4 => re(q/^(?:e,f,f,e|f)$/)},
    ],
    'group array for one column without GROUP_CONCAT_SEPARATOR'
);
cmp_bag(
    overshard(group => [qw/key1 key2/], group_array => [qw/key3 key4/], $test_dataset),
    [
        {key1 => "value1", key2 => 123, key3 => [456], key4 => ["a"]},
        {key1 => "value1", key2 => 666, key3 => [777], key4 => ["b"]},
        {key1 => "value2", key2 => 777, key3 => bag(888,999), key4 => bag("c", "d")},
        {key1 => "zzz", key2 => 999, key3 => bag(999,999,999), key4 => bag("e,f,f,e", "f", "f")},
    ],
    'group array for two columns without GROUP_CONCAT_SEPARATOR'
);
cmp_bag(
    overshard(group => [qw/key1 key2/], group_concat_distinct => [qw/key3 key4/], $test_dataset),
    [
        {key1 => "value1", key2 => 123, key3 => '456', key4 => "a"},
        {key1 => "value1", key2 => 666, key3 => '777', key4 => "b"},
        {key1 => "value2", key2 => 777, key3 => re('888,999|999,888'), key4 => re(q/^(?:c,d|d,c)$/)},
        {key1 => "zzz", key2 => 999, key3 => '999', key4 => re(q/^(?:e,f,f,e,f|f,e,f,f,e)$/)},
    ],
    'group_concat_distinct for two columns without GROUP_CONCAT_SEPARATOR'
);
cmp_bag(
    overshard(group => [qw/key1 key2/], group_array_distinct => [qw/key3 key4/], $test_dataset),
    [
        {key1 => "value1", key2 => 123, key3 => [456], key4 => ["a"]},
        {key1 => "value1", key2 => 666, key3 => [777], key4 => ["b"]},
        {key1 => "value2", key2 => 777, key3 => bag(888,999), key4 => bag("c", "d")},
        {key1 => "zzz", key2 => 999, key3 => bag(999), key4 => bag("e,f,f,e", "f")},
    ],
    'group_array_distinct for two columns without GROUP_CONCAT_SEPARATOR'
);
}

cmp_bag(
    overshard(group => [qw/key1 key2/], group_array => [qw/key3 key4/], $test_dataset),
    [
        {key1 => "value1", key2 => 123, key3 => [456], key4 => ["a"]},
        {key1 => "value1", key2 => 666, key3 => [777], key4 => ["b"]},
        {key1 => "value2", key2 => 777, key3 => bag(888,999), key4 => bag("c", "d")},
        {key1 => "zzz", key2 => 999, key3 => bag(999,999,999), key4 => bag("e", "f" , "f", "e", "f", "f")},
    ],
    'group array for two columns'
);
cmp_bag(
    overshard(group => [qw/key1 key2/], group_concat_distinct => [qw/key3 key4/], $test_dataset),
    [
        {key1 => "value1", key2 => 123, key3 => '456', key4 => "a"},
        {key1 => "value1", key2 => 666, key3 => '777', key4 => "b"},
        {key1 => "value2", key2 => 777, key3 => re('888,999|999,888'), key4 => re(q/^(?:c,d|d,c)$/)},
        {key1 => "zzz", key2 => 999, key3 => '999', key4 => re(q/^(?:e,f|f,e)$/)},
    ],
    'group_concat_distinct for two columns'
);
cmp_bag(
    overshard(group => [qw/key1 key2/], group_array_distinct => [qw/key3 key4/], $test_dataset),
    [
        {key1 => "value1", key2 => 123, key3 => [456], key4 => ["a"]},
        {key1 => "value1", key2 => 666, key3 => [777], key4 => ["b"]},
        {key1 => "value2", key2 => 777, key3 => bag(888,999), key4 => bag("c", "d")},
        {key1 => "zzz", key2 => 999, key3 => bag(999), key4 => bag("e", "f")},
    ],
    'group_array_distinct for two columns'
);

# count_distinct
cmp_bag(
    overshard(group => 'key1,key2', count_distinct => 'key3,key4', $test_dataset),
    [
        {key1 => "value1", key2 => 123, key3 => 1, key4 => 1},
        {key1 => "value1", key2 => 666, key3 => 1, key4 => 1},
        {key1 => "value2", key2 => 777, key3 => 2, key4 => 2},
        {key1 => "zzz", key2 => 999, key3 => 1, key4 => 2},
    ],
    'count distinct'
);

# having
dies_ok sub { overshard group => '', having => 1, [] }, 'invalid having';
dies_ok sub { overshard having => 1, [] }, 'having without group';
cmp_deeply(
    ( overshard group => 'id', having => sub { $_[0]->{key} > 1 }, [ { id => 0, key => 0 }, { id => 1, key => 1 }, { id => 2, key => 2 } ] ),
    [ { id => 2, key => 2 } ]
);



done_testing();
