package Model::Base;

use strict;
use warnings;
use utf8;

=head1 NAME

    Model::Base

=head1 DESCRIPTION

    Model::Base базовый класс для написания моделей, позволяет быстро описать
    поля модели, поддерживает проверку типов на лету.

    За счет предикатов has_$field позволяет на базе одного класса иметь модели
    с разным набором заданных полей, при этом иметь поля со значением undef для
    хранения БД столбцов со значением NULL, и отличать заданные в undef поля от
    незаданных вовсе.

    Модели на базе Model::Base решают проблему с отсутствием в Perl структур
    данных, с четко типизированными полями

    Объекты модели иммутабельны, для изменения полей сущестуют методы создающие
    копию объекта с изменением заданных полей

=head1 SYNOPSIS

    # Создаем класс модели
    package Model::Keyword;
    use Model::Base;

    define_types_fields(
        UnsignedInt => [
            'id'
            , 'group_id'
        ],

        String => [
            'text'
        ]
    );


    # другой способ задавать поля
    define_fields qw/Price Discount/ => 'Number';
    define_fields modified_time => 'Timestamp';
    define_fields normalized_text => 'Varchar255';

    done; # окончание определения модели

    # Используем класс модели

    use Test::More;
    use Test::Exception;;
    use DDP;

    use Model::Keyword;
    my $keyword;

    dies_ok(sub { $keyword = Model::Keyword->new(id => 'some id')});

    $keyword = Model::Keyword->new(id => 123, price => 10.1123);
    is_deeply([ $keyword->fields ],  [ 'id', 'price' ]);
    is_deeply($keyword->to_hash, { id => 123, price => 10.1123});

    ok(!$keyword->has_group_id);
    dies_ok(sub { $keyword->group_id }, "not defined in new()");

    dies_ok(sub {my $keyword = Model::Keyword->new('wrong_field' => 10)});

    ok($keyword->has_fields(qw/id price/));
    ok(!$keyword->has_fields(qw/id price group_id/));

    is_deeply(
        [ $keyword->missing_fields(qw/id price group_id text/) ]
        , [ qw/group_id text/ ]
    );

    my $clone = $keyword->clone;
    is_deeply($clone->to_hash, { price => 10.1123 } );

    my $price_update = $keyword->alter(price => 10.2);
    is_deeply($price_update->to_hash, { id => 123, price => 10.2 } );

    my $new_keyword = $keyword->alter_clone(price => 11);
    is_deeply($new_keyword->to_hash, { price => 11 } );

    $keyword = Model::Keyword->new(
        id => 1,
        group_id => 11,
        price => 42,
        text => 'some keyword and stuff',
        normalized_text => 'keyword some stuff',
        modified_time => '2013-07-15 12:00'
    );

    is_deeply($keyword->fields_to_hash(qw/id group_id price/), { id => 1, group_id => 11, price => 42 });

    done_testing;

=cut

our $VERSION = '0.00'; # not released yet

use Scalar::Util qw/blessed/;

use Mouse;
use Mouse::Exporter;

# sub-parts
use Model::Base::Types;
use Model::Base::Relations;
use Model::Base::DefineFields;

Mouse::Exporter->setup_import_methods(
    as_is => [
        # class defining sugar
        'define_fields',
        'define_types_fields',
        'done',

        # relationships defining sugar
        @Model::Base::Relations::METHODS,


        # model object interface
        'shove', # create object from hash without fields validation

        #   dumping
        'fields_to_hash',
        'to_hash',
        #   survey
        'fields',
        'simple_fields',
        'has_fields',
        'missing_fields',

        #   duplication helpers
        'clone',
        'alter',
        'alter_clone',
        # / model object interface


        # private stuff, exporting due to nature mouse export
        '_clone_drop_data',
        '_cut_to_drop',
        '_fields_aref',
        '_has_field_hashref',
        '_to_hash_value',
        '_to_hash_no_nested',
        '_is_link_field',

        # mostly private, field accessor helpers
        'trim_timestamp',
        'yesno',
    ],
    also  => 'Mouse',
);

has _has_field_hashref => (
    is => 'ro',
    isa => 'HashRef',
    lazy => 1,
    default => sub {
        return { map { $_ => 1 } shift->fields }
    },
);

has _fields_aref => (
    is => 'ro',
    isa => 'ArrayRef',
    lazy => 1,
    default => sub {
        my $self = shift;
        return [ grep
            {
                my $predicate = "has_$_";
                $self->meta->has_method($predicate) and $self->$predicate
            } ($self->meta->get_attribute_list)
        ];
    }
);


=head1 METHODS

=head2 new(%fields)

    Создает объект класса с полями указанными в %fields = ( field_name => value ..);
    В дальнейшем поля объекта иммутабельны

=head2 has_$field

    Возвращает true если поле было задано при создании объекта

    my $model1 = Model::Something->new(id => 123, name => 'Some name');
    $model1->has_id; # true
    my $model_new = Model::Something->new(name => 'Some name model object');
    $model_new->has_id; # false
    $model_new->has_name; # true
    $model_new->id; # died

    my $model2 = Model::Something->new(id => 123, name => undef);
    $model2->has_name; # true
    $model2->name; # undef

    Таким образом мы можем хранить в поле объекта undef значения, например для представления
    NULL из БД, и все равно знать является ли поле заданным

=head2 $field

    Возвращает значение поля $field, если значение не задано при создании объекта, будет брошен эксепшн
    значение заданное в undef в данном случае считается заданным.

=cut

=head2 define_fields @fields => $type

    Добавляет поля из @fields с типом $type

=cut

sub define_fields(@) {
    my $type = pop;
    my @fields = @_;
    my $meta = caller->meta;
    Model::Base::DefineFields::by_types($meta, $type => \@fields);
    return;
}

=head2 define_types_fields(%fields_by_types)
    Добавляем поля, переданные с группировкой по типу
    %fields_by_types = { type => [qw/field1 .. fieldN/], .., typeN => [..]}
=cut

sub define_types_fields {
    my %fields_by_types = @_;
    my $meta = caller->meta;
    Model::Base::DefineFields::by_types($meta, %fields_by_types);
    return;
}

=head2 done

    Финализация определения класса
    done(); должно быть в конце каждого файла модели;

=cut

sub done {
    my $meta = caller->meta;
    $meta->strict_constructor(1);
    return $meta->make_immutable();
}


=head2 shove($hashref), shove(%hash)

    Создает объект модели по хэшу (ссылке на хэш) параметров игнорируя все
    проверки полей. new и обрамляющие методы (BUILD, BUILDARGS и другие) не
    вызываются.

    Метод служит только для валидации, когда нам намерено нужны значения
    превышающие длинны полей и не совпадающие с типами, и крайне не
    рекомендован для использования где-то еще.

    После прохождения модели
    валидации необходимо зафиксировать результат вызовом new на to_hash();
=cut

sub shove {
    my $class = shift;
    my $params = @_ % 2 ? $_[0] : { @_ } ;
    my %reqs = map { $_->name => $_->is_required }  $class->meta->get_all_attributes;

    foreach my $name ( keys %reqs) {
        die "$name is required" if $reqs{$name} && !exists($params->{$name});
    }

    foreach my $p ( keys %$params ) {
        die "attribute $p unknown" unless exists $reqs{$p};
    }
    my $new_object = { %$params };

    return bless $new_object, ref $class ? ref $class :  $class;
}

# Object dumping

=head2 fields_to_hash(@fields)

    Возвращает ссылку на хэш с значениями полей моледи заданными в @fields
    my $banner_hashref = $banner->fields_to_hash(qw/id name title/);
    # { id => 123, name => 'Some banner name', title => 'some title' }

=cut

sub fields_to_hash {
    my ($self, @fields_to_dump) = @_;
    my $result = {};
    foreach(@fields_to_dump) {
        $result->{$_} =  $self->_to_hash_value($self->$_);
    }
    return $result;
}

=head2 to_hash

    Ссылка на хэш со всеми заданными в конструкторе полями модели
    Аналог $model->fields_to_hash($model->fields)

=cut

sub to_hash {
    my $self = shift;
    return $self->fields_to_hash( $self->fields );
}

sub _to_hash_no_nested {
    my $self = shift;
    return {map {
        $_ => $self->$_
    } $self->simple_fields};
}

sub _to_hash_value {
    my $self = shift;
    my $value = shift;
    return unless defined $value;

    if(ref $value eq 'ARRAY') {
        return [ map { $self->_to_hash_value($_) } @$value ];
    } elsif(blessed $value) {
        return $value->to_hash;
    } else {
        return $value;
    }
}



# Methods for objects duplication

=head2 clone

    Возвращает копию текущего объекта по всем полями, кроме id, который удаяется

    my $model = Model::Something->new(id=>123, name=>'somename');
    $model->id; # 123
    my $model_clone = $model->clone;
    $model_clone->id; # die
    $model_clone->has_id; # false

=cut

sub clone {
    my $self = shift;
    return $self->new($self->_clone_drop_data('id'));
}

=head2 alter(%new_values)

    Аналогично clone возвращает копию объекта, но включая поле id, и меняя 
    значения полей указанные в %new_values = ($field => $value).
    При пустом $new_vaules получаем точную копию объекта

    __drop - специальный ключ, ссылается на массив имен полей которые не будут
    включены в результирующий ответ

=cut

sub alter {
    my ($self, %new_values) = @_;
    my $to_drop = $self->_cut_to_drop(\%new_values);
    return $self->new(
        $self->_clone_drop_data(@$to_drop),
        %new_values
    );
}

=head2 alter_clone(%new_values)

    Аналог alter, но из результата вырезается id
    __drop - специальный ключ, ссылается на массив имен полей которые не будут
    включены в результирующий ответ

=cut

sub alter_clone {
    my ($self, %new_values) = @_;
    my $to_drop = $self->_cut_to_drop(\%new_values);
    return $self->new(
        $self->_clone_drop_data('id', @$to_drop),
        %new_values
    );
}

sub _clone_drop_data {
    my $self = shift;
    my $to_drop = \@_;
    my %data;
    @data{$self->fields} = @{$self}{$self->fields};
    delete $data{$_} foreach @$to_drop;
    return %data;
}

sub _cut_to_drop {
    my ($self, $data) = @_;
    return delete($data->{__drop}) || [];
}

# Fields methods

=head2 fields

    Список полей текущего объекта, если поле не было задано при создании объекта,
    его не будет в этом списке, даже если оно задано в классе

=cut

sub fields {
    my $self = shift;
    return @{ $self->_fields_aref };
}

=head2 simple_fields

    Список полей текущего объекта без учёта вложенных объектов

=cut

sub simple_fields {
    my $self = shift;
    my @simple_fields;
    foreach my $f ($self->fields) {
        push @simple_fields, $f unless $self->_is_link_field($f)
    }
    return @simple_fields;
}

sub _is_link_field {
    my $self = shift;
    my $field = shift;
    no strict 'refs';
    my $links = ${ref($self) . '::_LINKS'};
    return $links->{$field} ? 1 : 0;
}

=head2 missing_fields(@fields_to_check)

    Возвращает список полей состоящий из полей указанных в @fields_to_check
    но не определенных при созданнии объекта

=cut

sub missing_fields {
    my ($self, @fields) = @_;
    return grep {
        ! $self->_has_field_hashref->{$_}
    } @fields;
}

=head2 has_fields(@fields_to_check)

    Возвращает true если все поля указанные в @fields_to_check определены при создании объекта

=cut

sub has_fields {
    my ($self, @fields_to_check) = @_;
    return not scalar $self->missing_fields(@fields_to_check);
}



# Helpers

sub yesno { return $_[1] ? "Yes" : "No" }

sub trim_timestamp {
    my $self = shift;
    my $timestamp = shift;
    $timestamp =~ s/[^\d]//g;
    return $timestamp;
}

1;
