package Model::DataTree::Compiler;

use strict;
use warnings;
use utf8;

=pod

    $Id$

=head1 NAME

    Model::DataTree::Compiler;

=head1 SYNOPSIS

    my $tree_compiler = Model::DataTree::Compiler->new($data);
    $tree_compiler->build($scheme, $data);

=head1 DESCRIPTION

    Класс для преобразования списка с данными в дерево на основании переданной схемы

=head1 METHODS

=head2 new($scheme, $conversion_map)

    $scheme - объект Model::DataTree::Scheme
    возвращает объект Model::DataTree::Compiler

    Если задана $conversion_map то имена полям рекурсивно
    конверитуются согласно этому кэшу при вызове build

    Сконветировать  имена полей дерева можно отдельным вызовом
    Model::DataTree::Compiler->convert($tree, $conversion_map)

=head2 build($data)

    $data - ссылка на массив строк с данными (результата SQL
    запрос)

    метод возвращает данные в виде древовидного хэша
    сгруппированного согласно схеме

=head2 convert($data_tree, $conversion_map)

    Конвертировует имена полей в дереве согласно карте.
    Пример карты

    my $conversion_map = {
        login => 'Username',
        uid => 'UserId',
        campaigns => [ Campaigns => {
            cid => 'CampaignId',
            clicks => 'Clicks',
            groups => [
                AdGroups => {
                    group_name => 'AdGroupName',
                    pid => 'AdGroupId',
                    banners => [ Banners => {
                        bid => 'Id',
                        body => 'Text',
                        title => 'Title'
                    }],
                }
            ]
        }]
    };

    Карта - хэш полей соятоящий из имени поля и имени в которое
    его нужно преобразовать.

    Для связей (полей которые содержат структуры данных):
    <Имя_Поля> => [ <Новое имя поля>, $submap ]

=cut

use Yandex::HashUtils qw/hash_cut/;

sub new {
    my $class = shift;
    my $scheme = shift or die 'no scheme set';
    my $conversion_map = shift; # optional
    return bless { scheme => $scheme, conversion_map => $conversion_map }, $class;
}

sub cache { shift->{cache} //= {} }

sub flush {
    delete shift->{cache}
}

sub scheme { shift->{scheme} }
sub conversion_map { shift->{conversion_map} }

sub build {
    my $self = shift;
    my $data = shift # []
        or die "no data";

    $self->flush;

    my $results = [];
    foreach my $row (@$data) {
        my ($is_new, $item) = $self->_process_row_by_scheme(
            $row,
            $self->scheme,
            $self->cache,
            $self->conversion_map);
        push @$results, $item if $item && $is_new;
    }

    return $results;
}

# возвращает хэш с данными объекта или undef если данный объект уже
# был добавлен
sub _process_row_by_scheme {
    my $self = shift;
    my $row = shift;
    my $scheme = shift;
    # { id => object, branch_name => sub_tree_cache } }
    # кэш объектов текущего уровня иерархии, нужен чтобы не пересоздавать
    # и не передобавлять уже добавленный в результирующую структуру объект
    my $cache = shift || $self->cache;
    my $cmap = shift; # если надо сконвертировать также имена полей

    my ($key_name, $key_value) = $self->_row_key_value($scheme, $row);
    return unless $key_value;

    my $is_new = 0;
    my $item = $cache->{by_keys}{$key_value};
    if(!$item) {
        $is_new = 1;
        my $converted_key_name = $cmap && $cmap->{$key_name}
            ? $cmap->{$key_name}
            : $key_name;
        $item = $cache->{by_keys}{$key_value} = {
            $converted_key_name => $key_value,
        };
        foreach($scheme->fields) {
            my $f = $cmap && $cmap->{$_} ? $cmap->{$_} : $_;
            $item->{$f} = $row->{$_}
        }

    }

    foreach my $branch ($scheme->branches) {
        my $branch_cache = $cache->{branches}{$branch} //= {};

        # convertMap here
        my $sub_conversion;
        my $converted_branch_name;
        if($cmap && $cmap->{$branch}) {
            $converted_branch_name = $cmap->{$branch};
            if(ref $converted_branch_name eq 'ARRAY') {
                $sub_conversion = $converted_branch_name->[1];
                $converted_branch_name = $converted_branch_name->[0];
            }
        }
        my $sub_scheme = $scheme->branch($branch);
        my ($is_first_occurence, $sub_data) = $self->_process_row_by_scheme(
            $row, $sub_scheme, $branch_cache, $sub_conversion
        );
        $branch = $converted_branch_name if $converted_branch_name;
        if($sub_scheme->is_array) {
            $item->{$branch} //= [];
            push @{ $item->{$branch} }, $sub_data if $is_first_occurence;
        } else {
            $item->{$branch} = $sub_data;
        }
    }

    return ($is_new, $item);
}

sub _row_key_value {
    my $self = shift;
    my $scheme = shift;
    my $row = shift;
    my $key_name = $scheme->key;
    return ($key_name, $row->{$key_name});
}

sub convert {
    my $self = shift;
    my $data = shift;

    my $cmap = shift; # conversion map

    foreach my $tree (ref $data eq 'ARRAY' ? @$data : $data ) {
        foreach my $fld (keys %$tree) {
            my $value = delete $tree->{$fld};
            if(ref $cmap->{$fld} eq 'ARRAY') { # isa scheme
                my ($link_name, $sub_scheme) = @{ $cmap->{$fld} };
                $tree->{ $link_name } = $self->convert( $value, $sub_scheme);
            } elsif ($cmap->{$fld}) {
                $tree->{ $cmap->{$fld} } = $value;
            }
        }
    }
    return $data;
}

1;

__END__
