package Model::DataTree::Scheme;

use strict;
use warnings;
use utf8;

=pod

    $Id$

=head1 NAME

    Model::DataTree::Scheme

=head1 SYNOPSIS

    Model::DataTree::Scheme->new(
        uid => [qw/uid login/],
        campaigns => [
            cid => [qw/cid name clicks/],
            groups => [
                pid => [qw/pid group_name/],
                banners => [
                    bid => [qw/bid title body href/]
                ],
                keywords => [
                    id => [qw/id phrase/]
                ],
                minus_words => [
                    [ not_array => 'id' ] => [qw/mw_id mw_text/]
                ]
            ]
        ]
    );

=head1 DESCRIPTION

    Класс для описания преобразования плоского списка перемноженных данных
    (результата sql-запроса по связанным таблицам) в дерево данных

=head1 METHODS

=head2 new($scheme_definition|@scheme_definition)

    \@scheme_definition = $scheme_definition = [
        $key_definition => [ @fields ]
        $sub_scheme_name => $sub_scheme_definition
    ]

    $key_definition либо строка с именем поля содержащего ключ группировки,
    либо массив вида [ \@flags => 'key_field_name' ], если ключ один поддерживается запись [ $flag => 'key_field_name' ]

    Возможные флаги:
        * not_array - без ключа данные вычисленные по подсхеме будут записанны в массив

=cut

sub new {
    my $class = shift;
    my @scheme = ref $_[0] ? @{$_[0]} : @_;

    my $key;
    my $flags;

    if(ref $scheme[0]) {
        $key = $scheme[0][1];
        $flags = { map {
                $_ => 1
            } ( ref $scheme[0][0] ? @{$scheme[0][0]} : $scheme[0][0] )
        };
    } else {
        $key = $scheme[0];
    }

    my $branches = {};

    for(my $i = 2; $i <= $#scheme; $i=$i+2) {
        $branches->{ $scheme[$i] } = $class->new($scheme[$i+1]);
    }

    return bless {
        key => $key,
        fields => $scheme[1],
        flags => $flags,
        branches => $branches
    }
}

sub not_array {
    my $flags = shift->flags;
    return exists $flags->{not_array} && $flags->{not_array}
}

sub is_array { !shift->not_array }

sub key { shift->{key} }
sub fields { @{ shift->{fields} } }

sub branches { keys %{ shift->{branches} } }

sub branch { $_[0]->{branches}->{$_[1]} or die "no branch $_[1]" }

sub flags { shift->{flags} }

1;
