package Model::Mapper::Base;

use strict;
use warnings;
use utf8;

=pod

    $Id$

=head1 NAME

    Model::Mapper::Base;

=head1 SYNOPSIS

    package Model::Mapper::Banner;

    use Model::Mapper::Base;

=head1 DESCRIPTION

    Базовый класс для мапперов моделей директа

=head1 METHODS

=cut

use Yandex::HashUtils qw/hash_cut/;
use Yandex::DBTools;
use Yandex::DBShards;

use Model::Mappers;
use Model::Mapper::SQL;
use Model::Mapper::Base::Exporter;
use Model::RecordSet;

use Model::DataTree::Compiler;

sub new {
    my ($class, $db) = @_;

    my $self = bless {
        limit => 0,
        offset => 0,
        db => $db
    }, $class;

    die "not database specified" unless $self->db;
    return $self;
}

# утилиты, используемые в наследниках

sub mapper {
    my ($self, $mapper_name) = @_;
    $self->{mapper} ||= Model::Mappers->new($self->db);
    return $self->{mapper}->get($mapper_name);
}

sub found_rows { shift->{found_rows} }

sub set_limit {
    my $self = shift;
    $self->{limit} = shift||0;
}

sub set_offset {
    my $self = shift;
    $self->{offset} = shift||0;
}

sub limit { shift->{limit} }
sub offset { shift->{offset} }


sub select {
    my $self = shift;
    return $self->{selector} = Model::Mapper::SQL->new(mapper => $self);
}

sub selector {
    my $self = shift;
    unless($self->{selector}) {
        # initializing default selector if not any
        $self->{selector} = $self->select;
        $self->{selector}->fields( keys %{$self->fields_map} );
    }
    return $self->{selector};
}

sub selector_sql {
    my $self = shift;
    my %params = @_; # as in Model::Mapper::SQL
    my $selector = $self->selector;

    $selector->sql(
        limit => $self->limit,
        offset => $self->offset,
        %params
    );
}

sub fields_map {
    my $self = shift;
    my $class = (ref $self || $self);
    no strict 'refs';
    unless(${"$class\::_FIELD_TO_COLUMN_MAP"}) {
        my $cmap = $self->columns_map;
        ${"$class\::_FIELD_TO_COLUMN_MAP"} = {
            map { $cmap->{$_} => $_ } keys %$cmap
        };
    }
    return ${"$class\::_FIELD_TO_COLUMN_MAP"};
}

sub column_to_field {
    my ($self, $column) = @_;
    return $self->columns_map->{$column};
}

# на входе массив полей модели, на выходе те из них которые мапятся в DB. Нужно
# чтобы использовать произвольные объекты в мапперах - незнакомые мапперу поля
# будут проигнорированы при операциях обновления и сохранения.
sub grep_db_fields {
    my ($self, @fields) = @_;
    return grep { exists $self->fields_map->{$_} } @fields;
}

sub db { return shift->{db} }
sub shard_key { return shift->db->{key} }
sub shard_value { return shift->db->{val} }

sub is_single_sharded { return scalar dbnames(shift->db) == 1 }

sub inflate {
    my $self = shift;

    my %parameters;

    if(ref $_[0] eq 'ARRAY') {
        return [ map { $self->inflate($_) } @{$_[0]} ];
    } elsif( ref $_[0] eq 'HASH') {
        %parameters = %{$_[0]};
    } else {
        %parameters = @_;
    }

    foreach my $field (keys %parameters) {
        if(ref $parameters{$field}) {
            my $link_accessor = "$field\_link";
            my $link = $self->$link_accessor;
            $parameters{$field} = $link->right_mapper->new($self->db)
                ->inflate($parameters{$field})
        }
    }

    return $self->_model_class_name->new(%parameters);
}

sub get_row {
    my $self = shift;
    my $model = shift or return {};
    my $data = $model->to_hash;

    return $self->_hashref_fields_to_columns($model->to_hash);
}

# преобразует список моделей в список строк для БД.
# Все модели должны иметь набор идентичный первой в списке модели
# иначе будет ошибка
sub get_rows {
    my ($self, @models) = @_;
    my @fields = $models[0]->fields;
    my $rows = [];
    foreach my $o (@models) {
        push @$rows,
            $self->_hashref_fields_to_columns(
                $o->fields_to_hash(@fields)
            );
    }
    return $rows;
}

sub _hashref_fields_to_columns {
    my ($self, $hash) = @_;

    my $columns = {};
    foreach( keys %$hash) {
        my $column = $self->fields_map->{$_}
            or next;
        $columns->{ $self->fields_map->{$_} } = $hash->{$_}
    }
    return $columns;
}

# интерфейсная часть, которая может быть переопределена в наследниках

# метод для создания объектов в заданном шарде
# у всех объектов должен быть набор полей аналогичный первому в списке
sub create {
    my $self = shift;
    my @objects = @_;

    die "mapper has to be single sharded to create" unless $self->is_single_sharded;


    my @fields = $self->grep_db_fields($objects[0]->fields);
    my @columns = map { $self->fields_map->{$_} } @fields;

    my $rows = $self->get_rows(@objects);

    my $new_rows = [];
    my $new_ids; # бывает что мы знаем id заранее, напр. для объектов связанных 1к1
    if(grep { $_ eq $self->_table_id } @columns) {
        $new_ids = [ map { $_->id } @objects ];
        $new_rows = [ map { [ @{$_}{@columns} ] } @$rows ];
    } else {
        # для id для которых в метабазе храним только инкримент
        # указывать chain_key => chain_val нельзя
        my $sharded_by = $self->_shard_id_name;
        if (my $chain_key = $Yandex::DBShards::SHARD_KEYS{$sharded_by}->{chain_key}) {
            $new_ids = get_new_id_multi( $sharded_by, $#$rows+1, $self->shard_key => $self->shard_value )
        } else {
            $new_ids = get_new_id_multi( $sharded_by, $#$rows+1 );
        }

        # убедимся что у всех записей один набор полей
        for(my $i=0; $i <= $#$rows; $i++) {
            my $r = $rows->[$i];
            push @$new_rows, [
                $new_ids->[$i],
                @{$r}{@columns}
            ];
        }
        unshift @columns, $self->_table_id;
    }
    my $keys = join( ', ',  @columns);
    do_mass_insert_sql($self->db,
        "insert into " . $self->table_name_quoted .
            " ($keys) values %s",
        $new_rows
    );

    return @$new_ids;
}

# иногда нужно переопределить если имя primary key не совпадает с его
# именем в таблице SHARD_KEYS как например phid для bid.id
sub _shard_id_name { shift->_table_id }

sub update {
    my $self = shift;
    my @objects = @_;
    my $count = 0;
    foreach(@objects) {
        $count += do_update_table(
            $self->db,
            $self->_table_name,
            $self->get_row($_),
            where => { $self->_table_id => $_->id }
        );
    }
    return $count;
}

sub delete_ids {
    my ($self, @ids) = @_;
    return unless scalar @ids;
    return do_delete_from_table(
        $self->db,
        $self->_table_name,
        where => { $self->_table_id => \@ids }
    );
}

sub delete {
    my ($self, @models) = @_;
    my @ids = map { $_->id } @models;
    return $self->delete_ids(@ids);
}

sub get { # id -> Model::Something
    my $self = shift;
    my $id = shift or return;
    return $self->get_by_ids($id)->[0];
}

sub id_field_name {
    my $self = shift;
    my $table_id = $self->_table_id;
    my $id_field = $self->column_to_field($table_id);
    return $id_field;

}

sub where_sql_pk {
    my $self = shift;
    return $self->selector->map_field_to_sql_nq($self->id_field_name);
}

sub get_by_ids { # id -> ( Models )
    my ($self, @ids) = @_;

    return $self->find(
        db => $self->db,
        where => {
            $self->where_sql_pk => \@ids
        },
        extra => 'ORDER BY ' . $self->where_sql_pk . ' ASC'
    );
}

sub field_to_sql {
    my ($self, $table_column) = @_;
    return $self->selector->map_field_to_sql($table_column);
}

sub field_to_where {
    my ($self, $table_column) = @_;
    return $self->selector->map_field_to_sql_nq($table_column);
}

sub find {
    my $self = shift;
    my %params = @_; # as in Model::Mapper::SQL + db
    my $db = (delete $params{db} || $self->db)
        or die "Model::Mapper::Base->find db must be set";

    my $selector = $self->selector;
    my $total_count;
    $self->{found_rows} = undef;

    my $result = get_all_sql(
        $db,
        # если использовать строку не сработает SHARD_IDS в where
        # без arrayref (деблессинга) упадет JSON.pm при логировании
        # иначе надо патчить Yandex::Log->__get_text_message
        # на предмет allow_blessed => 1, convert_blessed => 1
        $self->selector_sql(%params)->arrayref
    );

    if($selector->to_count_rows) {
        $self->{found_rows} = get_one_field_sql($db, "SELECT FOUND_ROWS()");
    }

    return Model::RecordSet->new(
        $self->inflate_results_by_selector($result)
    );
}

sub selector_results {
    my $self = shift;
    my $results = shift;
    return $self->selector_results_compiler->build($results);
}

sub selector_results_compiler {
    my $self = shift;
    return Model::DataTree::Compiler->new( $self->selector->scheme, $self->selector->conversion_map )
}

sub inflate_results_by_selector {
    my $self = shift;
    my $results = shift; # get_all_sql rows []

    return $self->inflate( $self->selector_results($results) );
}

sub yesno { return $_[1] ? "Yes" : "No" }

sub dbh { get_dbh(shift->db) }

sub table_name_quoted {
    my $self = shift;
    return sql_quote_identifier($self->_table_name)
}

sub import {
    my $caller = caller(0);
    return if $caller->isa(__PACKAGE__);

    no strict 'refs';
    # чтобы не вызвать этот import при use потомков
    *{"${caller}::import"} = sub { 1 };

    Model::Mapper::Base::Exporter->do_import(__PACKAGE__, $caller);

    return; # ok
}

1;
