package Model::Mapper::Base::Exporter;

use strict;
use warnings;
use utf8;

use Module::Load;
use Yandex::HashUtils;

use Model::Mapper::Link::Factory;

=pod

    $Id$

=head1 NAME

    Model::Mapper::Base::Exporter

=head1 SYNOPSIS

    use Model::Mapper::Base::Exporter;
    sub import {
        Model::Mapper::Base::Exporter->do_import(__PACKAGE__, $caller);
        return; # ok
    }

=head1 DESCRIPTION

    Выгружает синтаксический сахар в указанный метод, см. использование в Model::Mapper::Base.

=head1 METHODS

=head2 do_import($base_class, $destination_class)

    импортирует синтаксический сахар для описания мапперов и их зависимостей в модуль $destination_class, модуль указанный в $base_class - это модуль заказывающий импорт (чаще всего это __PACKAGE__);

=cut
sub do_import {
    my $class = shift;
    my $base_class = shift;
    my $child_class = shift;

    # Чтобы не пробрасывать вниз по стеку вызовов одни и теже параметры
    # и чтобы не плодить глобальных переменных
    my $self = bless {
        base_class => $base_class,
        child_class => $child_class
    }, $class;

    no strict 'refs';

    push @{"${child_class}::ISA"}, $self->base_class;

    $self->import_model_description_sugar;
    $self->import_relations_sugar;

    return; #ok
}

sub base_class { shift->{base_class} }

sub child_class { shift->{child_class} }

=head1 Методы маппера

=head2 $mapper->columns_map

    Метод возвращающий хэш соответствия колонки БД полю модели

=head1 Сахар для описания маппера

=head2 define_columns { column_name => field_name }

    описывает преобразование колонок базы данных в поля модели 

=head2 model $ModelClass from $table_name by $id_field_name

    $ModelClass - имя класса модели, объекты которой отдает маппер
    $table_name - таблица в БД с данными модели
    $id_field_name - имя поля, соответствующее primary key колонке в таблице

=cut

sub import_model_description_sugar {
    my $self = shift;
    my $child_class = $self->child_class;

    no strict 'refs';

    $self->_add_method(define_columns => sub {
        my $columns = shift;
        ${"$child_class\::COLUMNS"} = {} unless defined ${"$child_class\::COLUMNS"};
        hash_merge ${"$child_class\::COLUMNS"}, $columns;
    });
    $self->_add_method(columns_map => sub {
        ${"$child_class\::COLUMNS"} # Список колонок в таблице
    });

    $self->_add_method('_model_class_name', sub {
        ${"$child_class\::MODEL_CLASS_NAME"} # Имя класса модели, напр. Model::AdGroup
    });

    $self->_add_method('_table_name', sub {
        ${"$child_class\::TABLE_NAME"} # SQL таблица с данными модели
    });

    $self->_add_method('_table_id', sub {
        ${"$child_class\::TABLE_ID"} # столбец с ID
    });

    $self->_add_method('by', sub {
        my %p = ( by => shift() );
        $p{from} = $_[1] if @_;
        return %p;
    });

    $self->_add_method('from', sub {
        my $from = shift;
        my %p = $_[1]
            ? ( from => $from, @_ )
            : ( from => $from );
        return %p;
    });

    $self->_add_method('model', sub {
        if (${"$child_class\::MODEL_CLASS_NAME"}) {
            warn "$child_class model declared several times";
            return;
        }

        my $model = shift;
        my $params = { @_ };

        my $format = "call should be model 'Model::Something' from 'some_table' by 'pk_id'";

        die "No model name specified, $format" unless $model;

        my $from = $params->{from}
            or die "No table name specified as 'from', $format";

        my $by = $params->{by}
            or die "No id column name specified as 'by', $format";

        $self->_add_static_variable('MODEL_CLASS_NAME', $model);
        $self->_add_static_variable('TABLE_NAME', $from);
        $self->_add_static_variable('TABLE_ID', $by);

        # to avoid 'used-once' warnings
        my @a = (${"$child_class\::TABLE_NAME"}, ${"$child_class\::TABLE_ID"});

        load $model;
    });

    return; # ok
}

=head1 Сахара для описания связей

=head2 has_one $link_field_name => by $foreign_key_field from $MapperClassName

=head2 has_many $link_field_name => by $foreign_key_field from $MapperClassName

=head2 may_has_many $link_field_name => by $foreign_key_field from $MapperClassName

    has_one - модели маппера принадлежит один объект модели маппера $MapperClassName
    may_has_one - модели маппера может принадлежать один объект модели маппера
        $MapperClassName (LEFT JOIN)
    $link_field_name - имя поля модели в котором будет содержатся объект зависимой модели
    $foreign_key_field - имя поля в связанном маппере по которому связываем к
        моделям текущего маппера ($foreign_key_field = id текущего маппера)
    $ MapperClassName - имя класса связанного маппера

=head2 belongs_to $link_field_name => from $ParentMapperClass => thru $parent_key

    Каждый объект модели данного маппера принадлежит объекту модели маппера
    $ParentMapperClass по ключу $parent_key

=cut

sub import_relations_sugar {
    my $self = shift;
    my $child_class = $self->child_class;

    $self->_add_static_variable('LINKS', {});
    $self->_add_method('_links', sub {
        no strict 'refs';
        return ${"$child_class\::LINKS"};
    });
    my $links = $child_class->_links;

    $self->_add_method('thru', sub {
        my %p = ( thru => shift() );
        $p{from} = $_[1] if @_;
        return %p;
    });

    my $links_factory = 'Model::Mapper::Link::Factory';

    $self->_add_method('has_many', sub {
        my $field_name = shift;

        $self->_model_has_field_or_die($field_name);

        my $link = $links->{$field_name} = ref $_[0]
            ? $_[0]
            : $links_factory->new(
                    mapper_class => $child_class,
                    field => $field_name,
                    @_
                )->has_many_link;

        $self->_add_link_accessor($field_name, $link);
        return; # ok
    });

    $self->_add_method('may_has_one', sub {
        my $field_name = shift;

        $self->_model_has_field_or_die($field_name);

        my $link = $links->{$field_name} = ref $_[0]
            ? $_[0]
            : $links_factory->new(
                    mapper_class => $child_class,
                    field => $field_name,
                    @_
                )->may_has_one_link;
        $self->_add_link_accessor($field_name, $link);
        return; # ok
    });

    $self->_add_method('has_one', sub {
        my $field_name = shift;

        $self->_model_has_field_or_die($field_name);

        my $link = $links->{$field_name} = ref $_[0]
            ? $_[0]
            : $links_factory->new(
                    mapper_class => $child_class,
                    field => $field_name,
                    @_
                )->has_one_link;
        $self->_add_link_accessor($field_name, $link);
        return; # ok
    });

    $self->_add_method('belongs_to', sub {
        my $field_name = shift;

        $self->_model_has_field_or_die($field_name);

        my $link = $links->{$field_name} = ref $_[0]
            ? $_[0]
            : $links_factory->new(
                    mapper_class => $child_class,
                    field => $field_name,
                    @_
                )->belongs_to_link;

        $self->_add_link_accessor($field_name, $link);
        return; # ok
    });

    $self->_add_method('may_belongs_to', sub {
        my $field_name = shift;

        $self->_model_has_field_or_die($field_name);

        my $link = $links->{$field_name} = ref $_[0]
            ? $_[0]
            : $links_factory->new(
                    mapper_class => $child_class,
                    field => $field_name,
                    @_
                )->may_belongs_to_link;

        $self->_add_link_accessor($field_name, $link);
        return; # ok
    });

}

sub _model_has_field_or_die {
    my $self = shift;
    my $field_name = shift or die "no field_name";

    my $class = $self->child_class->_model_class_name
        or die "class for $field_name not found";

    $class->can($field_name)
        or die $class . " has no $field_name field";
}

sub _add_link_accessor {
    my ($self, $field_name, $link) = @_;
    $self->_add_method("$field_name\_link", sub {
        return $link
    });
}

sub _add_method {
    my ($self, $name, $sub_code_ref) = @_;
    my $child_class = $self->child_class;

    no strict 'refs';
    *{"${child_class}::$name"} = $sub_code_ref;
}

sub _add_static_variable {
    my ($self, $name, $value) = @_;
    my $child_class = $self->child_class;

    no strict 'refs';
    ${"$child_class\::$name"} = $value;
}

1;
