package Model::Mapper::SQL::SelectFromJoin;

use strict;
use warnings;
use utf8;

use Scalar::Util qw/blessed unweaken/;
use List::MoreUtils qw/pairwise/;

use Mouse;

use Yandex::DBTools;

use Model::DataTree::Scheme;

has mapper      => (is => 'ro', isa => 'Model::Mapper::Base', weak_ref => 1);
has _selectors  => (is => 'rw', isa => 'HashRef', default => sub { {} });
has _fields     => (is => 'rw', isa => 'ArrayRef[Str]', default => sub { [] });

has _links => (
    is => 'rw',
    isa => 'HashRef[Model::Mapper::Link]',
    default => sub { {} },
);

has _links_added => (
    is => 'rw',
    isa => 'ArrayRef',
    auto_deref => 1,
    default => sub { [] }
);

has _table_column_to_field => (
    is => 'rw',
    isa => 'HashRef',
    default => sub { {} }
);

has columns     => (
    is => 'ro',
    lazy_build => 1,
    builder => sub {
        my $self = shift;
        my $columns = [];
        foreach my $f (@{$self->_fields}) {
            my $col = $self->mapper->fields_map->{$f}
                or die "can't find column name for field $f";
            push @$columns, $col;
        }
        return $columns;
    }
);

has columns_map => (
    is => 'ro',
    isa => 'HashRef',
    auto_deref => 1,
    lazy_build => 1,
    builder => sub {
        my $self = shift;
        my @cols = @{$self->columns};
        my @fields = @{$self->_fields};
        return {
            pairwise { $a => $b }
            @cols, @fields
        };
    }
);

sub selector { $_[0]->_selectors->{$_[1]} }

# jquery like interface for configuration
sub fields {
    my $self = shift;
    my @fields = @_;
    my $id_field = $self->mapper->id_field_name;
    unless(grep {  $id_field eq $_ } @fields) {
        push @fields, $id_field;
    }
    $self->_fields([@fields]);
    $self->columns; # to get errors right away if any

    return $self;
}

sub also {
    my $self = shift;
    my $link_field = shift;
    my $link = $self->_add_link_by_field($link_field);

    my $linked_selector;
    my $right_mapper; # don't move into if, you will loose weak ref
    if(blessed $_[0]) {
        $linked_selector = shift;
        $linked_selector->can('mapper') or die "not a selector";
        die 'Mapper of selector not matched by linked one'
            unless ref $linked_selector->mapper eq $link->right_mapper;
    } else {
        my $fields = shift; # []
        $right_mapper = $link->right_mapper->new($self->mapper->db);
        $linked_selector = $right_mapper->select()->fields(@$fields);
    }
    # by default ref to mapper is weak, if not fortify it
    # we will loose mapper, since it's not stored anywhere else
    unweaken( $linked_selector->{mapper} );
    $self->_selectors->{$link_field} = $linked_selector;

    return $self;
}

# / jquery like interface for configuration

# interface
sub sql {
    my $self = shift;

    return "SELECT "
        . join(', ', $self->sql_select_fields )
        . $self->_sql_from
        . $self->sql_join;
}

sub conversion_map {
    my $self = shift;
    my $prefix = shift||'';
    my $table = $prefix || $self->mapper->_table_name;

    my $conversion_map = {
        map {
            "$table." . $self->mapper->fields_map->{$_}
            => $_
        }
        @{$self->_fields}
    };

    foreach my $link_field ($self->_links_added) {
        my $link = $self->link($link_field);
        my $subprefix = $prefix ? "$prefix:" : '';
        $conversion_map->{$link_field} = [
            $link_field,
            $self->selector($link_field)->conversion_map("$subprefix$link_field")
        ];
    }
    return $conversion_map;
}

sub scheme {
    my $self = shift;
    return Model::DataTree::Scheme->new( $self->scheme_data );
}

sub linked_with {
    my $self = shift;
    my $link_field_name = shift;
    return exists $self->_links->{$link_field_name};
}

# / interface

# mostly protected interface

sub scheme_data {
    my $self = shift;
    my $prefix = shift||'';
    my $scheme = [];
    my $table = $prefix || $self->mapper->_table_name;

    my $table_id = $self->mapper->_table_id;

    $scheme = [
        "$table.$table_id"  => [
            map { "$table.$_" }
            @{ $self->columns }
        ]
    ];

    foreach my $link_field ($self->_links_added) {
        my $subprefix = $prefix ? "$prefix:" : '';
        my $subscheme = $self->selector($link_field)
            ->scheme_data("$subprefix$link_field");
        if(!$self->link($link_field)->is_multi) {
            $subscheme->[0] = [ not_array => $subscheme->[0] ];
        }
        push @$scheme, ($link_field, $subscheme);
    }

    return $scheme
}

sub link {
    my $self = shift;
    my $field_name = shift;
    return $self->_links->{$field_name};
}

sub sql_select_fields {
    my $self = shift;
    my $prefix = shift||'';
    my @sql_fields;
    my $table = $prefix || $self->mapper->_table_name;

    push @sql_fields,
        map { "`$table`.$_ as `$table.$_`" }
        @{ $self->columns };

    foreach my $link_field ($self->_links_added) {
        my $subprefix = $prefix ? "$prefix:" : '';
        push @sql_fields, $self->selector($link_field)
            ->sql_select_fields("$subprefix$link_field");
    }
    return @sql_fields;
}

sub sql_join {
    my $self = shift;
    my $prefix = shift||'';
    my @join;
    foreach my $link_field ($self->_links_added) {
        my $subprefix = $prefix ? "$prefix:" : '';
        push @join, $self->link($link_field)->sql_join($prefix),
            $self->selector($link_field)->sql_join( "$subprefix$link_field" );
    }
    return join(' ', @join);
}

# / protected interface

sub _add_link_by_field {
    my $self = shift;
    my $link_field = shift;
    unless($self->link($link_field)) {
        push @{$self->_links_added}, $link_field; # to preserve order on cloning
        my $link_getter = "$link_field\_link";
        my $mapper = $self->mapper;

        die "$link_field ain't defined in " . $mapper
            unless $mapper->can($link_getter);
        $self->_links->{$link_field} = $mapper->$link_getter;
    }
    return $self->link($link_field);
}

sub _sql_from {
    my $self = shift;
    return ' FROM ' . $self->_quote_identifier($self->mapper->_table_name);
}

sub _quote_identifier {
    my $self = shift;
    return sql_quote_identifier(@_);
}

__PACKAGE__->meta->make_immutable();
