use strict;
use warnings;

# Создаем класс модели
package Model::Keyword;
use Model::Base;

define_fields qw/id group_id/   => 'UnsignedInt';
define_fields price             => 'Number';
define_fields modified_time     => 'Timestamp';
define_fields normalized_text   => 'Varchar255';

# другой способ задавать поля
define_types_fields(
    String => [
        'text'
        , 'title'
    ],
    Varchar32 => [
        'name'
        , 'text_hash'
    ]
);


done; # окончание определения модели

package main;

# Используем класс модели

use Test::More tests => 19;
use Test::Exception;;
use DDP;

# use Model::Keyword;
my $keyword;

dies_ok(sub { $keyword = Model::Keyword->new(id => 'some id')});

$keyword = Model::Keyword->new(id => 123, price => 10.1123);
is_deeply([ $keyword->fields ],  [ 'id', 'price' ]);
is_deeply($keyword->to_hash, { id => 123, price => 10.1123});

ok(!$keyword->has_group_id);
dies_ok(sub { $keyword->group_id }, "not defined in new()");

dies_ok(sub {my $keyword = Model::Keyword->new('wrong_field' => 10)});

ok($keyword->has_fields(qw/id price/));
ok(!$keyword->has_fields(qw/id price group_id/));
use Carp::Always;
is_deeply(
    [ $keyword->missing_fields(qw/id price group_id text/) ]
    , [ qw/group_id text/ ]
);

my $clone = $keyword->clone;
is_deeply($clone->to_hash, { price => 10.1123 } );

my $price_update = $keyword->alter(price => 10.2);
is_deeply($price_update->to_hash, { id => 123, price => 10.2 } );

my $new_keyword = $keyword->alter_clone(price => 11);
is_deeply($new_keyword->to_hash, { price => 11 } );

$keyword = Model::Keyword->new(
    id => 1,
    group_id => 11,
    price => 42,
    text => 'some keyword and stuff',
    normalized_text => 'keyword some stuff',
    modified_time => '2013-07-15 12:00:00'
);

is_deeply($keyword->fields_to_hash(qw/id group_id price/), { id => 1, group_id => 11, price => 42 });
is_deeply(
    $keyword->alter(price => 12, __drop => [qw/group_id/]), {
        id => 1,
        price => 12,
        text => 'some keyword and stuff',
        normalized_text => 'keyword some stuff',
        modified_time => '2013-07-15 12:00:00'
});

is_deeply(
    $keyword->alter_clone(price => 12, __drop => [qw/group_id/]), {
        price => 12,
        text => 'some keyword and stuff',
        normalized_text => 'keyword some stuff',
        modified_time => '2013-07-15 12:00:00'
});


ok($keyword->modified_time_short eq '20130715120000', 'modified time short version is 20130715120000');

my $to_validate = {
    id => 'wrong id',
    name => 'aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaab', # 33 chars
};

dies_ok(sub { Model::Keyword->new($to_validate) }, 'fails to init model with non-validated data');

is_deeply(
    Model::Keyword->shove(
        %$to_validate
    )->to_hash, $to_validate,
    'shoving non-validated hash to model'
);

is_deeply(
    Model::Keyword->shove(
        $to_validate
    )->to_hash, $to_validate,
    'shoving non-validated hashref to model'
);

