#!/usr/bin/perl

use strict;
use warnings;
use utf8;

use Test::More tests => 1;
use Test::Deep;

use Yandex::DBTools;
use Yandex::DBShards;

use lib '/var/www/ppc.yandex.ru/protected';

use Settings;

use Model::DataTree::Compiler;
use Model::DataTree::Scheme;

my $data = data();

my $scheme = Model::DataTree::Scheme->new(
    uid => [qw/uid login/],
    campaigns => [
        cid => [qw/cid name clicks/],
        groups => [
            pid => [qw/pid group_name/],
            banners => [
                bid => [qw/bid title body href/]
            ],
            keywords => [
                id => [qw/id phrase/]
            ],
            minus_words => [
                [ not_array => 'id' ] => [qw/mw_id mw_text/]
            ]
        ]
    ]
);


my $conversion = {
    login => 'Username',
    uid => 'UserId',
    campaigns => [ Campaigns => {
        cid => 'CampaignId',
        clicks => 'Clicks',
        groups => [
            AdGroups => {
                group_name => 'AdGroupName',
                pid => 'AdGroupId',
                banners => [ Banners => {
                    bid => 'Id',
                    body => 'Text',
                    title => 'Title'
                }],
            }
        ]
    }]
};

my $tree_compiler = Model::DataTree::Compiler->new($scheme);
my $users_tree = $tree_compiler->build($data);
my $users_converted = $tree_compiler->convert($users_tree, $conversion);

is_deeply([
  {
    'Id' => '433928274',
    'Title' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{434}\x{436}\x{438}\x{43d}\x{441}\x{43e}\x{432}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}!",
    'Text' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{434}\x{436}\x{438}\x{43d}\x{441}\x{43e}\x{432}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!"
  }
], $users_tree->[0]->{Campaigns}->[2]->{AdGroups}->[3]->{Banners},
'converted results');

=off

# select to get data

my $uid = 132521783; # lamodasem

my $cids = [qw/9493283 9493282 9493281/];
my $shard = PPC(uid => $uid);

my $data = get_all_sql($shard, ['
    SELECT
        u.uid, u.login,
        c.cid, c.name, c.clicks,
        p.pid, p.group_name,
        b.bid, b.title, b.body, b.href,
        bi.id, bi.phrase
    FROM users u
    LEFT JOIN campaigns c USING(uid)
    LEFT JOIN phrases p USING(cid)
    LEFT JOIN minus_words mw USING(mw_id)
    LEFT JOIN banners b USING(bid)
    LEFT JOIN bids bi ON bi.pid = p.pid
', where => { 'u.uid' => SHARD_IDS, 'c.cid' => $cids} ]);
=cut

sub data {
[
  {
    'bid' => '433928110',
    'group_name' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{440}\x{435}\x{43c}\x{43d}\x{438}_\x{430}\x{43a}\x{43a}\x{441}\x{435}\x{441}\x{443}\x{430}\x{440}\x{44b}",
    'clicks' => '1412',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{421}\x{443}\x{43c}\x{43a}\x{438} \x{41e}\x{447}\x{43a}\x{438} \x{420}\x{435}\x{43c}\x{43d}\x{438} \x{417}\x{43e}\x{43d}\x{442}\x{44b} \x{41a}\x{43e}\x{448}\x{435}\x{43b}\x{44c}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{440}\x{435}\x{43c}\x{43d}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305026',
    'href' => 'www.lamoda.ru/c/703/accs-muzhskie-remni/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%A1%D1%83%D0%BC%D0%BA%D0%B8%20%D0%9E%D1%87%D0%BA%D0%B8%20%D0%A0%D0%B5%D0%BC%D0%BD%D0%B8%20%D0%97%D0%BE%D0%BD%D1%82%D1%8B%20%D0%9A%D0%BE%D1%88%D0%B5%D0%BB%D1%8C%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834367',
    'title' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{440}\x{435}\x{43c}\x{43d}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493281',
    'phrase' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{440}\x{435}\x{43c}\x{43d}\x{438}"
  },
  {
    'bid' => '433928121',
    'group_name' => "\x{41b}\x{435}\x{442}\x{43d}\x{438}\x{435} \x{441}\x{443}\x{43c}\x{43a}\x{438}_\x{430}\x{43a}\x{43a}\x{441}\x{435}\x{441}\x{443}\x{430}\x{440}\x{44b}",
    'clicks' => '1412',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{421}\x{443}\x{43c}\x{43a}\x{438} \x{41e}\x{447}\x{43a}\x{438} \x{420}\x{435}\x{43c}\x{43d}\x{438} \x{417}\x{43e}\x{43d}\x{442}\x{44b} \x{41a}\x{43e}\x{448}\x{435}\x{43b}\x{44c}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{41b}\x{435}\x{442}\x{43d}\x{438}\x{435} \x{441}\x{443}\x{43c}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305033',
    'href' => 'www.lamoda.ru/c/2782/accs-letnie-sumki/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%A1%D1%83%D0%BC%D0%BA%D0%B8%20%D0%9E%D1%87%D0%BA%D0%B8%20%D0%A0%D0%B5%D0%BC%D0%BD%D0%B8%20%D0%97%D0%BE%D0%BD%D1%82%D1%8B%20%D0%9A%D0%BE%D1%88%D0%B5%D0%BB%D1%8C%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834374',
    'title' => "\x{41b}\x{435}\x{442}\x{43d}\x{438}\x{435} \x{441}\x{443}\x{43c}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493281',
    'phrase' => "\x{41b}\x{435}\x{442}\x{43d}\x{438}\x{435} \x{441}\x{443}\x{43c}\x{43a}\x{438}"
  },
  {
    'bid' => '433928136',
    'group_name' => "\x{421}\x{443}\x{43c}\x{43a}\x{438}_\x{430}\x{43a}\x{43a}\x{441}\x{435}\x{441}\x{443}\x{430}\x{440}\x{44b}",
    'clicks' => '1412',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{421}\x{443}\x{43c}\x{43a}\x{438} \x{41e}\x{447}\x{43a}\x{438} \x{420}\x{435}\x{43c}\x{43d}\x{438} \x{417}\x{43e}\x{43d}\x{442}\x{44b} \x{41a}\x{43e}\x{448}\x{435}\x{43b}\x{44c}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{421}\x{443}\x{43c}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305040',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D1%81%D1%83%D0%BC%D0%BA%D0%B8&utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%A1%D1%83%D0%BC%D0%BA%D0%B8%20%D0%9E%D1%87%D0%BA%D0%B8%20%D0%A0%D0%B5%D0%BC%D0%BD%D0%B8%20%D0%97%D0%BE%D0%BD%D1%82%D1%8B%20%D0%9A%D0%BE%D1%88%D0%B5%D0%BB%D1%8C%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834391',
    'title' => "\x{421}\x{443}\x{43c}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493281',
    'phrase' => "\x{421}\x{443}\x{43c}\x{43a}\x{438}"
  },
  {
    'bid' => '433928150',
    'group_name' => "\x{417}\x{43e}\x{43d}\x{442}\x{44b}_\x{430}\x{43a}\x{43a}\x{441}\x{435}\x{441}\x{443}\x{430}\x{440}\x{44b}",
    'clicks' => '1412',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{421}\x{443}\x{43c}\x{43a}\x{438} \x{41e}\x{447}\x{43a}\x{438} \x{420}\x{435}\x{43c}\x{43d}\x{438} \x{417}\x{43e}\x{43d}\x{442}\x{44b} \x{41a}\x{43e}\x{448}\x{435}\x{43b}\x{44c}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{417}\x{43e}\x{43d}\x{442}\x{44b} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305046',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D0%B7%D0%BE%D0%BD%D1%82&utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%A1%D1%83%D0%BC%D0%BA%D0%B8%20%D0%9E%D1%87%D0%BA%D0%B8%20%D0%A0%D0%B5%D0%BC%D0%BD%D0%B8%20%D0%97%D0%BE%D0%BD%D1%82%D1%8B%20%D0%9A%D0%BE%D1%88%D0%B5%D0%BB%D1%8C%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834415',
    'title' => "\x{417}\x{43e}\x{43d}\x{442}\x{44b}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493281',
    'phrase' => "\x{417}\x{43e}\x{43d}\x{442}\x{44b}"
  },
  {
    'bid' => '433928165',
    'group_name' => "\x{41a}\x{43e}\x{436}\x{430}\x{43d}\x{44b}\x{435} \x{441}\x{443}\x{43c}\x{43a}\x{438}_\x{430}\x{43a}\x{43a}\x{441}\x{435}\x{441}\x{443}\x{430}\x{440}\x{44b}",
    'clicks' => '1412',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{421}\x{443}\x{43c}\x{43a}\x{438} \x{41e}\x{447}\x{43a}\x{438} \x{420}\x{435}\x{43c}\x{43d}\x{438} \x{417}\x{43e}\x{43d}\x{442}\x{44b} \x{41a}\x{43e}\x{448}\x{435}\x{43b}\x{44c}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{41a}\x{43e}\x{436}\x{430}\x{43d}\x{44b}\x{435} \x{441}\x{443}\x{43c}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305055',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D0%BA%D0%BE%D0%B6%D0%B0%D0%BD%D1%8B%D0%B5+%D1%81%D1%83%D0%BC%D0%BA%D0%B8&utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%A1%D1%83%D0%BC%D0%BA%D0%B8%20%D0%9E%D1%87%D0%BA%D0%B8%20%D0%A0%D0%B5%D0%BC%D0%BD%D0%B8%20%D0%97%D0%BE%D0%BD%D1%82%D1%8B%20%D0%9A%D0%BE%D1%88%D0%B5%D0%BB%D1%8C%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834426',
    'title' => "\x{41a}\x{43e}\x{436}\x{430}\x{43d}\x{44b}\x{435} \x{441}\x{443}\x{43c}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493281',
    'phrase' => "\x{41a}\x{43e}\x{436}\x{430}\x{43d}\x{44b}\x{435} \x{441}\x{443}\x{43c}\x{43a}\x{438}"
  },
  {
    'bid' => '433928184',
    'group_name' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{43e}\x{447}\x{43a}\x{438}_\x{430}\x{43a}\x{43a}\x{441}\x{435}\x{441}\x{443}\x{430}\x{440}\x{44b}",
    'clicks' => '1412',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{421}\x{443}\x{43c}\x{43a}\x{438} \x{41e}\x{447}\x{43a}\x{438} \x{420}\x{435}\x{43c}\x{43d}\x{438} \x{417}\x{43e}\x{43d}\x{442}\x{44b} \x{41a}\x{43e}\x{448}\x{435}\x{43b}\x{44c}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{43e}\x{447}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305066',
    'href' => 'www.lamoda.ru/c/2396/accs_ns-ochki-myzskie/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%A1%D1%83%D0%BC%D0%BA%D0%B8%20%D0%9E%D1%87%D0%BA%D0%B8%20%D0%A0%D0%B5%D0%BC%D0%BD%D0%B8%20%D0%97%D0%BE%D0%BD%D1%82%D1%8B%20%D0%9A%D0%BE%D1%88%D0%B5%D0%BB%D1%8C%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834437',
    'title' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{43e}\x{447}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493281',
    'phrase' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{43e}\x{447}\x{43a}\x{438}"
  },
  {
    'bid' => '433928201',
    'group_name' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43e}\x{447}\x{43a}\x{438}_\x{430}\x{43a}\x{43a}\x{441}\x{435}\x{441}\x{443}\x{430}\x{440}\x{44b}",
    'clicks' => '1412',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{421}\x{443}\x{43c}\x{43a}\x{438} \x{41e}\x{447}\x{43a}\x{438} \x{420}\x{435}\x{43c}\x{43d}\x{438} \x{417}\x{43e}\x{43d}\x{442}\x{44b} \x{41a}\x{43e}\x{448}\x{435}\x{43b}\x{44c}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43e}\x{447}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305076',
    'href' => 'www.lamoda.ru/c/2395/accs_ns-ochki/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%A1%D1%83%D0%BC%D0%BA%D0%B8%20%D0%9E%D1%87%D0%BA%D0%B8%20%D0%A0%D0%B5%D0%BC%D0%BD%D0%B8%20%D0%97%D0%BE%D0%BD%D1%82%D1%8B%20%D0%9A%D0%BE%D1%88%D0%B5%D0%BB%D1%8C%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834486',
    'title' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43e}\x{447}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493281',
    'phrase' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43e}\x{447}\x{43a}\x{438}"
  },
  {
    'bid' => '433928219',
    'group_name' => "\x{421}\x{43e}\x{43b}\x{43d}\x{446}\x{435}\x{437}\x{430}\x{449}\x{438}\x{442}\x{43d}\x{44b}\x{435} \x{43e}\x{447}\x{43a}\x{438}_\x{430}\x{43a}\x{43a}\x{441}\x{435}\x{441}\x{443}\x{430}\x{440}\x{44b}",
    'clicks' => '1412',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{421}\x{443}\x{43c}\x{43a}\x{438} \x{41e}\x{447}\x{43a}\x{438} \x{420}\x{435}\x{43c}\x{43d}\x{438} \x{417}\x{43e}\x{43d}\x{442}\x{44b} \x{41a}\x{43e}\x{448}\x{435}\x{43b}\x{44c}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{421}\x{43e}\x{43b}\x{43d}\x{446}\x{435}\x{437}\x{430}\x{449}\x{438}\x{442}\x{43d}\x{44b}\x{435} \x{43e}\x{447}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305085',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D0%A1%D0%BE%D0%BB%D0%BD%D1%86%D0%B5%D0%B7%D0%B0%D1%89%D0%B8%D1%82%D0%BD%D1%8B%D0%B5+%D0%BE%D1%87%D0%BA%D0%B8&utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%A1%D1%83%D0%BC%D0%BA%D0%B8%20%D0%9E%D1%87%D0%BA%D0%B8%20%D0%A0%D0%B5%D0%BC%D0%BD%D0%B8%20%D0%97%D0%BE%D0%BD%D1%82%D1%8B%20%D0%9A%D0%BE%D1%88%D0%B5%D0%BB%D1%8C%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834526',
    'title' => "\x{421}\x{43e}\x{43b}\x{43d}\x{446}\x{435}\x{437}\x{430}\x{449}\x{438}\x{442}\x{43d}\x{44b}\x{435} \x{43e}\x{447}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493281',
    'phrase' => "\x{421}\x{43e}\x{43b}\x{43d}\x{446}\x{435}\x{437}\x{430}\x{449}\x{438}\x{442}\x{43d}\x{44b}\x{435} \x{43e}\x{447}\x{43a}\x{438}"
  },
  {
    'bid' => '433928232',
    'group_name' => "\x{421}\x{43f}\x{43e}\x{440}\x{442}\x{438}\x{432}\x{43d}\x{44b}\x{435} \x{441}\x{443}\x{43c}\x{43a}\x{438}_\x{430}\x{43a}\x{43a}\x{441}\x{435}\x{441}\x{443}\x{430}\x{440}\x{44b}",
    'clicks' => '1412',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{421}\x{443}\x{43c}\x{43a}\x{438} \x{41e}\x{447}\x{43a}\x{438} \x{420}\x{435}\x{43c}\x{43d}\x{438} \x{417}\x{43e}\x{43d}\x{442}\x{44b} \x{41a}\x{43e}\x{448}\x{435}\x{43b}\x{44c}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{421}\x{43f}\x{43e}\x{440}\x{442}\x{438}\x{432}\x{43d}\x{44b}\x{435} \x{441}\x{443}\x{43c}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305091',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D0%A1%D0%BF%D0%BE%D1%80%D1%82%D0%B8%D0%B2%D0%BD%D1%8B%D0%B5+%D1%81%D1%83%D0%BC%D0%BA%D0%B8&utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%A1%D1%83%D0%BC%D0%BA%D0%B8%20%D0%9E%D1%87%D0%BA%D0%B8%20%D0%A0%D0%B5%D0%BC%D0%BD%D0%B8%20%D0%97%D0%BE%D0%BD%D1%82%D1%8B%20%D0%9A%D0%BE%D1%88%D0%B5%D0%BB%D1%8C%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834534',
    'title' => "\x{421}\x{43f}\x{43e}\x{440}\x{442}\x{438}\x{432}\x{43d}\x{44b}\x{435} \x{441}\x{443}\x{43c}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493281',
    'phrase' => "\x{421}\x{43f}\x{43e}\x{440}\x{442}\x{438}\x{432}\x{43d}\x{44b}\x{435} \x{441}\x{443}\x{43c}\x{43a}\x{438}"
  },
  {
    'bid' => '433928239',
    'group_name' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{440}\x{435}\x{43c}\x{43d}\x{438}_\x{430}\x{43a}\x{43a}\x{441}\x{435}\x{441}\x{443}\x{430}\x{440}\x{44b}",
    'clicks' => '1412',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{421}\x{443}\x{43c}\x{43a}\x{438} \x{41e}\x{447}\x{43a}\x{438} \x{420}\x{435}\x{43c}\x{43d}\x{438} \x{417}\x{43e}\x{43d}\x{442}\x{44b} \x{41a}\x{43e}\x{448}\x{435}\x{43b}\x{44c}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{440}\x{435}\x{43c}\x{43d}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305098',
    'href' => 'www.lamoda.ru/c/701/accs-remni/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%A1%D1%83%D0%BC%D0%BA%D0%B8%20%D0%9E%D1%87%D0%BA%D0%B8%20%D0%A0%D0%B5%D0%BC%D0%BD%D0%B8%20%D0%97%D0%BE%D0%BD%D1%82%D1%8B%20%D0%9A%D0%BE%D1%88%D0%B5%D0%BB%D1%8C%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834552',
    'title' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{440}\x{435}\x{43c}\x{43d}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493281',
    'phrase' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{440}\x{435}\x{43c}\x{43d}\x{438}"
  },
  {
    'bid' => '433928245',
    'group_name' => "\x{420}\x{435}\x{43c}\x{43d}\x{438}_\x{430}\x{43a}\x{43a}\x{441}\x{435}\x{441}\x{443}\x{430}\x{440}\x{44b}",
    'clicks' => '1412',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{421}\x{443}\x{43c}\x{43a}\x{438} \x{41e}\x{447}\x{43a}\x{438} \x{420}\x{435}\x{43c}\x{43d}\x{438} \x{417}\x{43e}\x{43d}\x{442}\x{44b} \x{41a}\x{43e}\x{448}\x{435}\x{43b}\x{44c}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{420}\x{435}\x{43c}\x{43d}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305103',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D1%80%D0%B5%D0%BC%D0%BD%D0%B8&utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%A1%D1%83%D0%BC%D0%BA%D0%B8%20%D0%9E%D1%87%D0%BA%D0%B8%20%D0%A0%D0%B5%D0%BC%D0%BD%D0%B8%20%D0%97%D0%BE%D0%BD%D1%82%D1%8B%20%D0%9A%D0%BE%D1%88%D0%B5%D0%BB%D1%8C%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834600',
    'title' => "\x{420}\x{435}\x{43c}\x{43d}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493281',
    'phrase' => "\x{420}\x{435}\x{43c}\x{43d}\x{438}"
  },
  {
    'bid' => '433928248',
    'group_name' => "\x{411}\x{43e}\x{43b}\x{44c}\x{448}\x{438}\x{435} \x{441}\x{443}\x{43c}\x{43a}\x{438}_\x{430}\x{43a}\x{43a}\x{441}\x{435}\x{441}\x{443}\x{430}\x{440}\x{44b}",
    'clicks' => '1412',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{421}\x{443}\x{43c}\x{43a}\x{438} \x{41e}\x{447}\x{43a}\x{438} \x{420}\x{435}\x{43c}\x{43d}\x{438} \x{417}\x{43e}\x{43d}\x{442}\x{44b} \x{41a}\x{43e}\x{448}\x{435}\x{43b}\x{44c}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{411}\x{43e}\x{43b}\x{44c}\x{448}\x{438}\x{435} \x{441}\x{443}\x{43c}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305105',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D0%91%D0%BE%D0%BB%D1%8C%D1%88%D0%B8%D0%B5+%D1%81%D1%83%D0%BC%D0%BA%D0%B8&utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%A1%D1%83%D0%BC%D0%BA%D0%B8%20%D0%9E%D1%87%D0%BA%D0%B8%20%D0%A0%D0%B5%D0%BC%D0%BD%D0%B8%20%D0%97%D0%BE%D0%BD%D1%82%D1%8B%20%D0%9A%D0%BE%D1%88%D0%B5%D0%BB%D1%8C%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834644',
    'title' => "\x{411}\x{43e}\x{43b}\x{44c}\x{448}\x{438}\x{435} \x{441}\x{443}\x{43c}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493281',
    'phrase' => "\x{411}\x{43e}\x{43b}\x{44c}\x{448}\x{438}\x{435} \x{441}\x{443}\x{43c}\x{43a}\x{438}"
  },
  {
    'bid' => '433928250',
    'group_name' => "\x{41a}\x{43e}\x{448}\x{435}\x{43b}\x{44c}\x{43a}\x{438}_\x{430}\x{43a}\x{43a}\x{441}\x{435}\x{441}\x{443}\x{430}\x{440}\x{44b}",
    'clicks' => '1412',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{421}\x{443}\x{43c}\x{43a}\x{438} \x{41e}\x{447}\x{43a}\x{438} \x{420}\x{435}\x{43c}\x{43d}\x{438} \x{417}\x{43e}\x{43d}\x{442}\x{44b} \x{41a}\x{43e}\x{448}\x{435}\x{43b}\x{44c}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{41a}\x{43e}\x{448}\x{435}\x{43b}\x{44c}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305107',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D0%BA%D0%BE%D1%88%D0%B5%D0%BB%D0%B5%D0%BA&utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%A1%D1%83%D0%BC%D0%BA%D0%B8%20%D0%9E%D1%87%D0%BA%D0%B8%20%D0%A0%D0%B5%D0%BC%D0%BD%D0%B8%20%D0%97%D0%BE%D0%BD%D1%82%D1%8B%20%D0%9A%D0%BE%D1%88%D0%B5%D0%BB%D1%8C%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834645',
    'title' => "\x{41a}\x{43e}\x{448}\x{435}\x{43b}\x{44c}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493281',
    'phrase' => "\x{41a}\x{43e}\x{448}\x{435}\x{43b}\x{44c}\x{43a}\x{438}"
  },
  {
    'bid' => '433928252',
    'group_name' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{441}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b}_\x{43e}\x{431}\x{443}\x{432}\x{44c}",
    'clicks' => '4071',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{41c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b} \x{42d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438} \x{422}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b} \x{421}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b} \x{411}\x{430}\x{43b}\x{435}\x{442}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{441}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305108',
    'href' => 'www.lamoda.ru/c/3222/shoes-slipony-muzhskie/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%9C%D0%BE%D0%BA%D0%B0%D1%81%D0%B8%D0%BD%D1%8B%20%D0%AD%D1%81%D0%BF%D0%B0%D0%B4%D1%80%D0%B8%D0%BB%D1%8C%D0%B8%20%D0%A2%D0%BE%D0%BF%D1%81%D0%B0%D0%B9%D0%B4%D0%B5%D1%80%D1%8B%20%D0%A1%D0%BB%D0%B8%D0%BF%D0%BE%D0%BD%D1%8B%20%D0%91%D0%B0%D0%BB%D0%B5%D1%82%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834673',
    'title' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{441}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493282',
    'phrase' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{441}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b}"
  },
  {
    'bid' => '433928253',
    'group_name' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{442}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b}_\x{43e}\x{431}\x{443}\x{432}\x{44c}",
    'clicks' => '4071',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{41c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b} \x{42d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438} \x{422}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b} \x{421}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b} \x{411}\x{430}\x{43b}\x{435}\x{442}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{442}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305109',
    'href' => 'www.lamoda.ru/c/2456/shoes-myzskie-topsaydery/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%9C%D0%BE%D0%BA%D0%B0%D1%81%D0%B8%D0%BD%D1%8B%20%D0%AD%D1%81%D0%BF%D0%B0%D0%B4%D1%80%D0%B8%D0%BB%D1%8C%D0%B8%20%D0%A2%D0%BE%D0%BF%D1%81%D0%B0%D0%B9%D0%B4%D0%B5%D1%80%D1%8B%20%D0%A1%D0%BB%D0%B8%D0%BF%D0%BE%D0%BD%D1%8B%20%D0%91%D0%B0%D0%BB%D0%B5%D1%82%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834674',
    'title' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{442}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493282',
    'phrase' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{442}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b}"
  },
  {
    'bid' => '433928255',
    'group_name' => "\x{421}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b}_\x{43e}\x{431}\x{443}\x{432}\x{44c}",
    'clicks' => '4071',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{41c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b} \x{42d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438} \x{422}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b} \x{421}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b} \x{411}\x{430}\x{43b}\x{435}\x{442}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{421}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305110',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D1%81%D0%BB%D0%B8%D0%BF%D0%BE%D0%BD%D1%8B?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%9C%D0%BE%D0%BA%D0%B0%D1%81%D0%B8%D0%BD%D1%8B%20%D0%AD%D1%81%D0%BF%D0%B0%D0%B4%D1%80%D0%B8%D0%BB%D1%8C%D0%B8%20%D0%A2%D0%BE%D0%BF%D1%81%D0%B0%D0%B9%D0%B4%D0%B5%D1%80%D1%8B%20%D0%A1%D0%BB%D0%B8%D0%BF%D0%BE%D0%BD%D1%8B%20%D0%91%D0%B0%D0%BB%D0%B5%D1%82%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834675',
    'title' => "\x{421}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493282',
    'phrase' => "\x{421}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b}"
  },
  {
    'bid' => '433928256',
    'group_name' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{441}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b}_\x{43e}\x{431}\x{443}\x{432}\x{44c}",
    'clicks' => '4071',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{41c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b} \x{42d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438} \x{422}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b} \x{421}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b} \x{411}\x{430}\x{43b}\x{435}\x{442}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{441}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305112',
    'href' => 'www.lamoda.ru/c/3219/shoes-slipony/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%9C%D0%BE%D0%BA%D0%B0%D1%81%D0%B8%D0%BD%D1%8B%20%D0%AD%D1%81%D0%BF%D0%B0%D0%B4%D1%80%D0%B8%D0%BB%D1%8C%D0%B8%20%D0%A2%D0%BE%D0%BF%D1%81%D0%B0%D0%B9%D0%B4%D0%B5%D1%80%D1%8B%20%D0%A1%D0%BB%D0%B8%D0%BF%D0%BE%D0%BD%D1%8B%20%D0%91%D0%B0%D0%BB%D0%B5%D1%82%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834702',
    'title' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{441}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493282',
    'phrase' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{441}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b}"
  },
  {
    'bid' => '433928257',
    'group_name' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b}_\x{43e}\x{431}\x{443}\x{432}\x{44c}",
    'clicks' => '4071',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{41c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b} \x{42d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438} \x{422}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b} \x{421}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b} \x{411}\x{430}\x{43b}\x{435}\x{442}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305113',
    'href' => 'www.lamoda.ru/c/2439/shoes-mokasini/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%9C%D0%BE%D0%BA%D0%B0%D1%81%D0%B8%D0%BD%D1%8B%20%D0%AD%D1%81%D0%BF%D0%B0%D0%B4%D1%80%D0%B8%D0%BB%D1%8C%D0%B8%20%D0%A2%D0%BE%D0%BF%D1%81%D0%B0%D0%B9%D0%B4%D0%B5%D1%80%D1%8B%20%D0%A1%D0%BB%D0%B8%D0%BF%D0%BE%D0%BD%D1%8B%20%D0%91%D0%B0%D0%BB%D0%B5%D1%82%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834704',
    'title' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493282',
    'phrase' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b}"
  },
  {
    'bid' => '433928259',
    'group_name' => "\x{41c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b}_\x{43e}\x{431}\x{443}\x{432}\x{44c}",
    'clicks' => '4071',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{41c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b} \x{42d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438} \x{422}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b} \x{421}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b} \x{411}\x{430}\x{43b}\x{435}\x{442}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{41c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305114',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D0%BC%D0%BE%D0%BA%D0%B0%D1%81%D0%B8%D0%BD%D1%8B&utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%9C%D0%BE%D0%BA%D0%B0%D1%81%D0%B8%D0%BD%D1%8B%20%D0%AD%D1%81%D0%BF%D0%B0%D0%B4%D1%80%D0%B8%D0%BB%D1%8C%D0%B8%20%D0%A2%D0%BE%D0%BF%D1%81%D0%B0%D0%B9%D0%B4%D0%B5%D1%80%D1%8B%20%D0%A1%D0%BB%D0%B8%D0%BF%D0%BE%D0%BD%D1%8B%20%D0%91%D0%B0%D0%BB%D0%B5%D1%82%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834705',
    'title' => "\x{41c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493282',
    'phrase' => "\x{41c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b}"
  },
  {
    'bid' => '433928262',
    'group_name' => "\x{42d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438}_\x{43e}\x{431}\x{443}\x{432}\x{44c}",
    'clicks' => '4071',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{41c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b} \x{42d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438} \x{422}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b} \x{421}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b} \x{411}\x{430}\x{43b}\x{435}\x{442}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{42d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305117',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D1%8D%D1%81%D0%BF%D0%B0%D0%B4%D1%80%D0%B8%D0%BB%D1%8C%D0%B8&utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%9C%D0%BE%D0%BA%D0%B0%D1%81%D0%B8%D0%BD%D1%8B%20%D0%AD%D1%81%D0%BF%D0%B0%D0%B4%D1%80%D0%B8%D0%BB%D1%8C%D0%B8%20%D0%A2%D0%BE%D0%BF%D1%81%D0%B0%D0%B9%D0%B4%D0%B5%D1%80%D1%8B%20%D0%A1%D0%BB%D0%B8%D0%BF%D0%BE%D0%BD%D1%8B%20%D0%91%D0%B0%D0%BB%D0%B5%D1%82%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834739',
    'title' => "\x{42d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493282',
    'phrase' => "\x{42d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438}"
  },
  {
    'bid' => '433928263',
    'group_name' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{442}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b}_\x{43e}\x{431}\x{443}\x{432}\x{44c}",
    'clicks' => '4071',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{41c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b} \x{42d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438} \x{422}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b} \x{421}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b} \x{411}\x{430}\x{43b}\x{435}\x{442}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{442}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305118',
    'href' => 'www.lamoda.ru/c/2441/shoes-topsaydery/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%9C%D0%BE%D0%BA%D0%B0%D1%81%D0%B8%D0%BD%D1%8B%20%D0%AD%D1%81%D0%BF%D0%B0%D0%B4%D1%80%D0%B8%D0%BB%D1%8C%D0%B8%20%D0%A2%D0%BE%D0%BF%D1%81%D0%B0%D0%B9%D0%B4%D0%B5%D1%80%D1%8B%20%D0%A1%D0%BB%D0%B8%D0%BF%D0%BE%D0%BD%D1%8B%20%D0%91%D0%B0%D0%BB%D0%B5%D1%82%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834742',
    'title' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{442}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493282',
    'phrase' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{442}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b}"
  },
  {
    'bid' => '433928265',
    'group_name' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{44d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438}_\x{43e}\x{431}\x{443}\x{432}\x{44c}",
    'clicks' => '4071',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{41c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b} \x{42d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438} \x{422}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b} \x{421}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b} \x{411}\x{430}\x{43b}\x{435}\x{442}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{44d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305120',
    'href' => 'www.lamoda.ru/c/2457/shoes-myzskie-espadrily/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%9C%D0%BE%D0%BA%D0%B0%D1%81%D0%B8%D0%BD%D1%8B%20%D0%AD%D1%81%D0%BF%D0%B0%D0%B4%D1%80%D0%B8%D0%BB%D1%8C%D0%B8%20%D0%A2%D0%BE%D0%BF%D1%81%D0%B0%D0%B9%D0%B4%D0%B5%D1%80%D1%8B%20%D0%A1%D0%BB%D0%B8%D0%BF%D0%BE%D0%BD%D1%8B%20%D0%91%D0%B0%D0%BB%D0%B5%D1%82%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834743',
    'title' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{44d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493282',
    'phrase' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{44d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438}"
  },
  {
    'bid' => '433928266',
    'group_name' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{43c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b}_\x{43e}\x{431}\x{443}\x{432}\x{44c}",
    'clicks' => '4071',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{41c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b} \x{42d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438} \x{422}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b} \x{421}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b} \x{411}\x{430}\x{43b}\x{435}\x{442}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{43c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305121',
    'href' => 'www.lamoda.ru/c/2455/shoes-muzhskie-mokasini/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%9C%D0%BE%D0%BA%D0%B0%D1%81%D0%B8%D0%BD%D1%8B%20%D0%AD%D1%81%D0%BF%D0%B0%D0%B4%D1%80%D0%B8%D0%BB%D1%8C%D0%B8%20%D0%A2%D0%BE%D0%BF%D1%81%D0%B0%D0%B9%D0%B4%D0%B5%D1%80%D1%8B%20%D0%A1%D0%BB%D0%B8%D0%BF%D0%BE%D0%BD%D1%8B%20%D0%91%D0%B0%D0%BB%D0%B5%D1%82%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834766',
    'title' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{43c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493282',
    'phrase' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{43c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b}"
  },
  {
    'bid' => '433928267',
    'group_name' => "\x{422}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b}_\x{43e}\x{431}\x{443}\x{432}\x{44c}",
    'clicks' => '4071',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{41c}\x{43e}\x{43a}\x{430}\x{441}\x{438}\x{43d}\x{44b} \x{42d}\x{441}\x{43f}\x{430}\x{434}\x{440}\x{438}\x{43b}\x{44c}\x{438} \x{422}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b} \x{421}\x{43b}\x{438}\x{43f}\x{43e}\x{43d}\x{44b} \x{411}\x{430}\x{43b}\x{435}\x{442}\x{43a}\x{438}",
    'login' => 'lamodasem',
    'body' => "\x{422}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305122',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D1%82%D0%BE%D0%BF%D1%81%D0%B0%D0%B9%D0%B4%D0%B5%D1%80%D1%8B&utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%9C%D0%BE%D0%BA%D0%B0%D1%81%D0%B8%D0%BD%D1%8B%20%D0%AD%D1%81%D0%BF%D0%B0%D0%B4%D1%80%D0%B8%D0%BB%D1%8C%D0%B8%20%D0%A2%D0%BE%D0%BF%D1%81%D0%B0%D0%B9%D0%B4%D0%B5%D1%80%D1%8B%20%D0%A1%D0%BB%D0%B8%D0%BF%D0%BE%D0%BD%D1%8B%20%D0%91%D0%B0%D0%BB%D0%B5%D1%82%D0%BA%D0%B8&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834778',
    'title' => "\x{422}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493282',
    'phrase' => "\x{422}\x{43e}\x{43f}\x{441}\x{430}\x{439}\x{434}\x{435}\x{440}\x{44b}"
  },
  {
    'bid' => '433928269',
    'group_name' => "\x{41f}\x{43b}\x{430}\x{449}\x{438}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{41f}\x{43b}\x{430}\x{449}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305123',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D0%BF%D0%BB%D0%B0%D1%89&utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834779',
    'title' => "\x{41f}\x{43b}\x{430}\x{449}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{41f}\x{43b}\x{430}\x{449}\x{438}"
  },
  {
    'bid' => '433928271',
    'group_name' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{41f}\x{443}\x{445}\x{43e}\x{432}\x{438}\x{43a}\x{438}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{41f}\x{443}\x{445}\x{43e}\x{432}\x{438}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305126',
    'href' => 'www.lamoda.ru/c/491/clothes-muzhskie-puhoviki/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834809',
    'title' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{41f}\x{443}\x{445}\x{43e}\x{432}\x{438}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{41f}\x{443}\x{445}\x{43e}\x{432}\x{438}\x{43a}\x{438}"
  },
  {
    'bid' => '433928272',
    'group_name' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43f}\x{430}\x{43b}\x{44c}\x{442}\x{43e}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43f}\x{430}\x{43b}\x{44c}\x{442}\x{43e} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305127',
    'href' => 'www.lamoda.ru/c/361/clothes-palto/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834810',
    'title' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43f}\x{430}\x{43b}\x{44c}\x{442}\x{43e}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43f}\x{430}\x{43b}\x{44c}\x{442}\x{43e}"
  },
  {
    'bid' => '433928274',
    'group_name' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{434}\x{436}\x{438}\x{43d}\x{441}\x{43e}\x{432}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{434}\x{436}\x{438}\x{43d}\x{441}\x{43e}\x{432}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305128',
    'href' => 'www.lamoda.ru/c/2504/clothes-jeans-kurtki/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834811',
    'title' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{434}\x{436}\x{438}\x{43d}\x{441}\x{43e}\x{432}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}!",
    'cid' => '9493283',
    'phrase' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{434}\x{436}\x{438}\x{43d}\x{441}\x{43e}\x{432}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}"
  },
  {
    'bid' => '433928276',
    'group_name' => "\x{41a}\x{443}\x{440}\x{442}\x{43a}\x{438}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{41a}\x{443}\x{440}\x{442}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305131',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D0%BA%D1%83%D1%80%D1%82%D0%BA%D0%B8&utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834958',
    'title' => "\x{41a}\x{443}\x{440}\x{442}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{41a}\x{443}\x{440}\x{442}\x{43a}\x{438}"
  },
  {
    'bid' => '433928277',
    'group_name' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43f}\x{443}\x{445}\x{43e}\x{432}\x{438}\x{43a}\x{438}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43f}\x{443}\x{445}\x{43e}\x{432}\x{438}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305132',
    'href' => 'www.lamoda.ru/c/365/clothes-puhoviki/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834966',
    'title' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43f}\x{443}\x{445}\x{43e}\x{432}\x{438}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43f}\x{443}\x{445}\x{43e}\x{432}\x{438}\x{43a}\x{438}"
  },
  {
    'bid' => '433928279',
    'group_name' => "\x{41a}\x{43e}\x{436}\x{430}\x{43d}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{41a}\x{43e}\x{436}\x{430}\x{43d}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305133',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D0%BA%D0%BE%D0%B6%D0%B0%D0%BD%D1%8B%D0%B5+%D0%BA%D1%83%D1%80%D1%82%D0%BA%D0%B8&utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834967',
    'title' => "\x{41a}\x{43e}\x{436}\x{430}\x{43d}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{41a}\x{43e}\x{436}\x{430}\x{43d}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}"
  },
  {
    'bid' => '433928280',
    'group_name' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{43a}\x{43e}\x{436}\x{430}\x{43d}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{43a}\x{43e}\x{436}\x{430}\x{43d}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305135',
    'href' => 'www.lamoda.ru/c/2503/clothes-kozhanye-kurtki/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834992',
    'title' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{43a}\x{43e}\x{436}\x{430}\x{43d}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}!",
    'cid' => '9493283',
    'phrase' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{43a}\x{43e}\x{436}\x{430}\x{43d}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}"
  },
  {
    'bid' => '433928283',
    'group_name' => "\x{41f}\x{443}\x{445}\x{43e}\x{432}\x{438}\x{43a}\x{438}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{41f}\x{443}\x{445}\x{43e}\x{432}\x{438}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305136',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D0%BF%D1%83%D1%85%D0%BE%D0%B2%D0%B8%D0%BA%D0%B8&utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834995',
    'title' => "\x{41f}\x{443}\x{445}\x{43e}\x{432}\x{438}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{41f}\x{443}\x{445}\x{43e}\x{432}\x{438}\x{43a}\x{438}"
  },
  {
    'bid' => '433928287',
    'group_name' => "\x{41f}\x{430}\x{43b}\x{44c}\x{442}\x{43e}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{41f}\x{430}\x{43b}\x{44c}\x{442}\x{43e} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305137',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D0%BF%D0%B0%D0%BB%D1%8C%D1%82%D0%BE&utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344834996',
    'title' => "\x{41f}\x{430}\x{43b}\x{44c}\x{442}\x{43e}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{41f}\x{430}\x{43b}\x{44c}\x{442}\x{43e}"
  },
  {
    'bid' => '433928292',
    'group_name' => "\x{422}\x{440}\x{435}\x{43d}\x{447}\x{43a}\x{43e}\x{442}\x{44b}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{422}\x{440}\x{435}\x{43d}\x{447}\x{43a}\x{43e}\x{442}\x{44b} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305139',
    'href' => 'www.lamoda.ru/c/363/clothes-zenskie-plaschi/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344835000',
    'title' => "\x{422}\x{440}\x{435}\x{43d}\x{447}\x{43a}\x{43e}\x{442}\x{44b}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{422}\x{440}\x{435}\x{43d}\x{447}\x{43a}\x{43e}\x{442}\x{44b}"
  },
  {
    'bid' => '433928295',
    'group_name' => "\x{414}\x{436}\x{438}\x{43d}\x{441}\x{43e}\x{432}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{414}\x{436}\x{438}\x{43d}\x{441}\x{43e}\x{432}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305140',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D0%B4%D0%B6%D0%B8%D0%BD%D1%81%D0%BE%D0%B2%D1%8B%D0%B5+%D0%BA%D1%83%D1%80%D1%82%D0%BA%D0%B8&utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344835025',
    'title' => "\x{414}\x{436}\x{438}\x{43d}\x{441}\x{43e}\x{432}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{414}\x{436}\x{438}\x{43d}\x{441}\x{43e}\x{432}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}"
  },
  {
    'bid' => '433928296',
    'group_name' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{430}\x{44f} \x{43f}\x{430}\x{440}\x{43a}\x{430}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{430}\x{44f} \x{43f}\x{430}\x{440}\x{43a}\x{430} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305141',
    'href' => 'www.lamoda.ru/c/3562/clothes-parki-jen/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344835026',
    'title' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{430}\x{44f} \x{43f}\x{430}\x{440}\x{43a}\x{430}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{430}\x{44f} \x{43f}\x{430}\x{440}\x{43a}\x{430}"
  },
  {
    'bid' => '433928300',
    'group_name' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{43e}\x{435} \x{43f}\x{430}\x{43b}\x{44c}\x{442}\x{43e}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{43e}\x{435} \x{43f}\x{430}\x{43b}\x{44c}\x{442}\x{43e} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305143',
    'href' => 'www.lamoda.ru/c/487/clothes-muzhskie-palto/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344835035',
    'title' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{43e}\x{435} \x{43f}\x{430}\x{43b}\x{44c}\x{442}\x{43e}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{43e}\x{435} \x{43f}\x{430}\x{43b}\x{44c}\x{442}\x{43e}"
  },
  {
    'bid' => '433928301',
    'group_name' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305145',
    'href' => 'www.lamoda.ru/c/479/clothes-muzhskaya-verkhnyaya-odezhda/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344835050',
    'title' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}"
  },
  {
    'bid' => '433928302',
    'group_name' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43a}\x{43e}\x{436}\x{430}\x{43d}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43a}\x{43e}\x{436}\x{430}\x{43d}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305146',
    'href' => 'www.lamoda.ru/c/2461/clothes-kozhanie-kurtki/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344835063',
    'title' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43a}\x{43e}\x{436}\x{430}\x{43d}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}!",
    'cid' => '9493283',
    'phrase' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43a}\x{43e}\x{436}\x{430}\x{43d}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}"
  },
  {
    'bid' => '433928303',
    'group_name' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{432}\x{435}\x{442}\x{440}\x{43e}\x{432}\x{43a}\x{438}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{432}\x{435}\x{442}\x{440}\x{43e}\x{432}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305147',
    'href' => 'www.lamoda.ru/c/2505/clothes-muzhskie-vetrovki/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344835105',
    'title' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{432}\x{435}\x{442}\x{440}\x{43e}\x{432}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{438}\x{435} \x{432}\x{435}\x{442}\x{440}\x{43e}\x{432}\x{43a}\x{438}"
  },
  {
    'bid' => '433928305',
    'group_name' => "\x{412}\x{435}\x{442}\x{440}\x{43e}\x{432}\x{43a}\x{438}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{412}\x{435}\x{442}\x{440}\x{43e}\x{432}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305149',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D0%B2%D0%B5%D1%82%D1%80%D0%BE%D0%B2%D0%BA%D0%B8&utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344835157',
    'title' => "\x{412}\x{435}\x{442}\x{440}\x{43e}\x{432}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{412}\x{435}\x{442}\x{440}\x{43e}\x{432}\x{43a}\x{438}"
  },
  {
    'bid' => '433928307',
    'group_name' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{432}\x{435}\x{442}\x{440}\x{43e}\x{432}\x{43a}\x{438}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{432}\x{435}\x{442}\x{440}\x{43e}\x{432}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305151',
    'href' => 'www.lamoda.ru/c/2463/clothes-zenskie-vetrovki/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344835194',
    'title' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{432}\x{435}\x{442}\x{440}\x{43e}\x{432}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{432}\x{435}\x{442}\x{440}\x{43e}\x{432}\x{43a}\x{438}"
  },
  {
    'bid' => '433928308',
    'group_name' => "\x{41f}\x{430}\x{440}\x{43a}\x{430}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{41f}\x{430}\x{440}\x{43a}\x{430} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305153',
    'href' => 'www.lamoda.ru/catalogsearch/result/?q=%D0%BF%D0%B0%D1%80%D0%BA%D0%B0?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344835223',
    'title' => "\x{41f}\x{430}\x{440}\x{43a}\x{430}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{41f}\x{430}\x{440}\x{43a}\x{430}"
  },
  {
    'bid' => '433928314',
    'group_name' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{434}\x{436}\x{438}\x{43d}\x{441}\x{43e}\x{432}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{434}\x{436}\x{438}\x{43d}\x{441}\x{43e}\x{432}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305155',
    'href' => 'www.lamoda.ru/c/2462/clothes-jeansovie-kurtki/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344835243',
    'title' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{434}\x{436}\x{438}\x{43d}\x{441}\x{43e}\x{432}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}!",
    'cid' => '9493283',
    'phrase' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{434}\x{436}\x{438}\x{43d}\x{441}\x{43e}\x{432}\x{44b}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}"
  },
  {
    'bid' => '433928317',
    'group_name' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{430}\x{44f} \x{43f}\x{430}\x{440}\x{43a}\x{430}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{430}\x{44f} \x{43f}\x{430}\x{440}\x{43a}\x{430} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305158',
    'href' => 'www.lamoda.ru/c/3561/clothes-parki-muj/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344835273',
    'title' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{430}\x{44f} \x{43f}\x{430}\x{440}\x{43a}\x{430}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{41c}\x{443}\x{436}\x{441}\x{43a}\x{430}\x{44f} \x{43f}\x{430}\x{440}\x{43a}\x{430}"
  },
  {
    'bid' => '433928320',
    'group_name' => "\x{41f}\x{43e}\x{441}\x{435}\x{442}\x{438}\x{43b}\x{438} \x{436}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}_\x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'clicks' => '6151',
    'uid' => '132521783',
    'name' => "Display_Re_Cat:\x{412}\x{435}\x{440}\x{445}\x{43d}\x{44f}\x{44f} \x{43e}\x{434}\x{435}\x{436}\x{434}\x{430}",
    'login' => 'lamodasem',
    'body' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438} \x{43d}\x{430} Lamoda! \x{421}\x{43a}\x{438}\x{434}\x{43a}\x{438}! \x{411}\x{435}\x{441}\x{43f}\x{43b}\x{430}\x{442}\x{43d}\x{430}\x{44f} \x{434}\x{43e}\x{441}\x{442}\x{430}\x{432}\x{43a}\x{430} \x{43f}\x{43e} \x{420}\x{424}!",
    'pid' => '375305160',
    'href' => 'www.lamoda.ru/c/357/clothes-verkhnyaya-odezhda/?utm_campaign={campaign_id}.Display_Re_Cat%3A%D0%92%D0%B5%D1%80%D1%85%D0%BD%D1%8F%D1%8F%20%D0%BE%D0%B4%D0%B5%D0%B6%D0%B4%D0%B0&utm_term=PHR.{phrase_id}.{banner_id}&utm_content={position_type}_{position}_{keyword}_1&utm_source=YDirect&utm_medium=cpc',
    'id' => '2344835310',
    'title' => "\x{416}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}! \x{420}\x{430}\x{441}\x{43f}\x{440}\x{43e}\x{434}\x{430}\x{436}\x{430}!",
    'cid' => '9493283',
    'phrase' => "\x{41f}\x{43e}\x{441}\x{435}\x{442}\x{438}\x{43b}\x{438} \x{436}\x{435}\x{43d}\x{441}\x{43a}\x{438}\x{435} \x{43a}\x{443}\x{440}\x{442}\x{43a}\x{438}"
  }
];
}
