package Direct::PhraseTools;

=head1 NAME

Direct::PhraseTools

=head1 DESCRIPTION

    Данный модуль предназначен для выполнения разнообразных преобразований/вычислений на основе текстов ключевых фраз (или минус-фраз) объявлений Директа.
    В частности:
    - вычисление фразы и ее нормальной фраз по заданному тексту поискового запроса или АДФ
    - вычисление минус-фразы и ее нормальной формы по заданному тексту поискового запроса или АДФ

    На данный момент используется в мастере отчетов со стороны Директа и БК, для вычисления статусов ПЗ/АДФ

=head1 SYNOPSYS

    Пример использования в БК для сопоставления ПЗ и фразы клиента:
    my $norm_phrase_from_search_query = get_norm_phrase( unquoted_key_phrase_from_text( $search_query) ) );
    my $norm_phrase_from_bm_phrase = get_norm_phrase( unquoted_key_phrase_from_text( quoted_phrases_bs2direct( $bm_phrase ) ) );
    my $norm_phrase_from_client_phrase = get_norm_phrase( unquoted_key_phrase_from_key_phrase( quoted_phrases_bs2direct( $client_phrase ) ) );
    my $status = $norm_phrase_from_search_query eq $norm_phrase_from_client_phrase ? 'added' : 'none';
    С минус-фразами - аналогично.

=cut

use Direct::Modern;

use List::MoreUtils qw/all/;

use Yandex::MyGoodWords;
use Yandex::StopWords;

use parent 'Exporter';
our @EXPORT_OK = qw/key_phrase_from_text
                    minus_phrase_from_text
                    unquoted_key_phrase_from_text
                    unquoted_key_phrase_from_key_phrase
                    polish_phrase_text
                    get_norm_phrase
                    unquote_phrase
                    quoted_phrases_bs2direct
                    stop_word_add_plus
                   /;

=head2 key_phrase_from_text

    по произвольному тексту (ПЗ/АДФ) возвращает соответсвующую тексту форму фразы, готовой к добавлению в ключевые фразы клиента
    (не гарантируется, что результат пройдет валидацию ключевых фраз при попытке добавления фразы)

=cut

sub key_phrase_from_text {
    my $text = shift;

    if (defined $text) {
        $text = polish_phrase_text($text);
        unless (is_phrase_quoted($text)) {
            $text = stop_word_add_plus($text, skip_minus_words => 1);
        }
        $text = clean_quoted_pluses($text);
    }

    return $text;
}

=head2 minus_phrase_from_text

    по произвольному тексту (ПЗ/АДФ) возвращает соответсвующую тексту форму фразы, готовую к добавлению в минус-фразы клиента
    (результат не обязательно пройдет валидацию минус-фраз при попытке добавления минус-фразы)

    ВАЖНО: результат должен отличаться от key_phrase_from_text только наличием кавычек, иначе Директ разойдется с БК в определении статуса ПЗ/АДФ

=cut

sub minus_phrase_from_text {
    my $text = shift;

    $text = key_phrase_from_text($text);
    if (defined $text) {
        if (is_phrase_quotable($text)) {
            $text = qq{"$text"};
            $text = clean_quoted_pluses($text);
        }
    }

    return $text;   
}

=head2 unquoted_key_phrase_from_text

    по произвольному тексту (ПЗ/АДФ) возвращает соответсвующую тексту форму фразы, при этом, если фраза закавычена - убираем кавычки, и фиксируем стоп-слова
    (поскольку роль фиксатора стоп-слов до этого играли кавычки)

    Используется БК для заполнения справочника соответствия ПЗ <=> Нормализованный текст клиентской фразы

=cut

sub unquoted_key_phrase_from_text {
    my $text = shift;

    if (defined $text) {
        $text = key_phrase_from_text($text);
        $text = unquoted_key_phrase_from_key_phrase($text);
    }

    return $text;
}

=head2 unquoted_key_phrase_from_key_phrase

    по тексту ключевой фразы (т.е. она уже в нужной форме, не требует никаких преобразований) возвращает форму фразы без кавычек, 
    а именно: если фраза закавычена - убираем кавычки, и фиксируем стоп-слова
    (поскольку роль фиксатора стоп-слов до этого играли кавычки)

    Используется БК для заполнения справочника соответствия Фраза клиента <=> Нормализованный текст клиентской фразы, для сравнения с ПЗ/ДРФ

=cut

sub unquoted_key_phrase_from_key_phrase {
    my $text = shift;

    if (defined $text) {
        if (is_phrase_quoted($text)) {
            $text = unquote_phrase($text);
            $text = stop_word_add_plus($text);
        }
    }

    return $text;
}

=head2 get_norm_phrase

    По фразе получить норм. форму
    Параметры:
        текст фразы
    Результат:
        norm_phrase - нормальная форма
        если фраза undef или пустая - возвращаем undef

=cut

sub get_norm_phrase {
    my $phrase = shift;
    return undef if !defined $phrase;

    $phrase =~ s/\\*//g;
    $phrase =~ s/\-\s+/-/g;
    $phrase =~ s/^\s+//g;
    $phrase =~ s/\s+$//g;

    my $norm_phrase = Yandex::MyGoodWords::norm_words($phrase);
    $norm_phrase = undef if defined $norm_phrase && $norm_phrase eq '';

    return $norm_phrase;
}

=head2 polish_phrase_text

приведение ключевых слов к общему виду:
    - нормализуем пробелы
    - если фраза в кавычках - удаляем плюсы
    - если во фразе из одних чисел присутствуют "плохие" числа, и нет препятствий для добавления кавычек - они добавляются.

=cut
sub polish_phrase_text {
    my ($phrase, $has_implied_minuswords) = @_;

    $phrase =~ s/\s+/ /g;
    $phrase =~ s/^\s+|\s+$//g;
    $phrase =~ s/(\[|")\s+/$1/g;
    $phrase =~ s/\s+(\]|")/$1/g;

    if (not $has_implied_minuswords and is_phrase_quotable($phrase) and all { /^\+?\d+$/ } split(/\s+/, $phrase)) {
        my @digit_words = $phrase =~ /(?:^|(?<=\s|\+))([\d]+)(?=$|\s)/g;
        if (not all {is_good_digits($_)} @digit_words) {
            $phrase = qq{"$phrase"};
        }
    }

    $phrase = clean_quoted_pluses($phrase);
    return $phrase;
}

=head2 clean_quoted_pluses
    
для фразы в кавычках удаляем плюсы - иначе будут проблемы с обратным преобразование фраз в статистике

=cut
sub clean_quoted_pluses {
    my $phrase = shift;

    if (is_phrase_quoted($phrase)) {
        $phrase =~ s/\+//g;
    }
    return $phrase;
}

=head2 quoted_phrases_bs2direct(phrase)

    Перевод фразы из внутреннего представления БК в формат Директа -
        обратная функции Yandex::MyGoodWords::process_quoted_phrases

    Заменяет ~0(новый формат), или ~\d(старый формат) на двойные кавычки вокруг фразы

=cut

sub quoted_phrases_bs2direct {
    
    my $phrase = shift;

    if (defined($phrase) && $phrase =~ s/\s+\~[0-9]+//g) {
        $phrase =~ s/\+//g;
        $phrase = qq["$phrase"];
    }
    return $phrase;
}

=head2 stop_word_add_plus (str)

    В строке str добавить перед каждым стоп-словом знак "+" 
    %O - именованные параметры
        skip_minus_words - минус слова оставляем "как есть"

=cut
sub stop_word_add_plus
{
    my ($str, %O) = @_;
    my @str = split / +/, $str;
    my @char_order = ('+'); # если перед словом были какие-то спец символы - мы должны сохранить их
    for my $s (@str) {
        my %chars;
        my $s0 = $s;
        if ($s =~ s!^(\W+)!!) {
            %chars = map { $_ => 1 } split //, $1;
        }
        # пропускаем слова, которые не нужно фиксировать - с ! или в [] ""
        if ($chars{'!'} or $chars{'"'} or $chars{'['} or 
            ($chars{'-'} && $O{skip_minus_words})) {
            $s = $s0;
            next;
        }
        if (Yandex::StopWords::is_stopword($s)) {
            $chars{'+'}=1;
        }
        $s = (join '', map { defined $chars{$_} ? $_ : '' } @char_order).$s;
    }
    return join ' ', @str;
}

=head2 is_phrase_quotable

Можно ли взять указанную фразу в кавычки (не все спецсимволы допустимы внутри кавычек).

=cut

sub is_phrase_quotable($) {
    $_[0] !~ m( \[
              | \]
              | \s-
              | "
              )x;
}

=head2 is_phrase_quoted

Фраза заключена в кавычки

=cut
sub is_phrase_quoted($) {
    if (defined $_[0] && $_[0] =~ m/^".*"$/) {
        return 1;
    }
    return 0;
}

=head2 unquote_phrase

Очистить фразу от кавычек

=cut

sub unquote_phrase($) {
    $_[0] =~ s/"//g;
    return $_[0];
}


=head2 is_good_digits

    my $dont_need_to_quote = is_good_digits($digits_only_keyword);

=cut

sub is_good_digits($) {
    my ($d) = @_;

    die "Digit-only words could be checked" if $d =~ /\D/;

    my $dcnt= keys %{{ map {$_=>1} split //, $d }};
    return 1 if $d =~ /\d{5}/ && ($dcnt > 3);
    return 1 if $d =~ /\d{6}/;
    return 1 if $d =~ /^(666|585|999)$/;
    return 1 if $d =~ /^(206|21(0[4789]|31|20|13|15))$/;
    return 1 if $d =~ /^(8600|6300|6110|7390|6290|3307|3309|9100|2752|2347|7511|8130|3613|2217|3501|3308|4361|3205|8092|8306|3720|8061|8734|2080|6271|7562|7506|3725|6516|1682|4320|6288|1589|6267|8110|4308|8310|1795|8102|6129|2428|6210|6520|2710|2630|2612|6501|1812|2515|8734|2101|8732|65115|22171|65116|10880|77100|54115|20880|21214|6802|5871|3182|7669|2206|7505|2595|88358|8204|4513|9446|3505|11200|7201|8421|6270|7224|8504|8725|7223|4713|6007|7994|6180|50560|2711|4088|3961|56686|7197|5008|
                        5427|784|55944|53453|3716|8405|3438|5022|9860|6304|6387|66062|8812|81611|8601|2274|6317|2793|6702|7438|7857|8972|11554|2641|2825|6467|1665|4151|1652|56996|3964|9730|43114|54452|21721|30130|4403|1612|6022|9262|8162|7660|4706|22270|33314|4014|2414|4061|6461|9780|70002|6337|5460|4556|2523|8140|8389|60260|6500|6303|2717|2123|2512|2106|2126|5080)$/x;
    return 1 if $d =~ /^(59|926|903|499|86|814|428|808)$/;
    return 1 if $d =~ /\d{4}/ && ($d > 2030) && $d =~ /[2-9]/ && ($dcnt == 2);
    return 0;
}



1;
