package Yandex::Golem;

=head1 NAME

    Yandex::Golem - работа с АПИ golem.yandex-team.ru

=head1 DESCRIPTION

    Golem — это внутренний сервис, собирающий информацию
    об инфраструктуре компании из различных источников
    и представляющий её в наглядном виде пользователям.

    Описание АПИ здесь:
    https://golem.yandex-team.ru/api/

=cut

use strict;
use warnings;

use LWP::UserAgent;
use URI::Escape qw/uri_escape_utf8/;

use Yandex::Trace;
use Log::Any '$log';

use base qw/Exporter/;
our @EXPORT_OK = qw/
    downtime_check
    firewall_expand_macro
    sms_send
/;

use utf8;

=head2 $API_BASE_URL

    Начало url-а, к которому идут все запросы

=cut
our $API_BASE_RW_URL ||= "https://golem.yandex-team.ru/api/";
our $API_BASE_RO_URL ||= "http://ro.admin.yandex-team.ru/api/";

our %API_BY_REQUEST = (
    'sms/send'              => \$API_BASE_RW_URL,
    'downtime/check'        => \$API_BASE_RO_URL,
    'firewall/expand_macro' => \$API_BASE_RO_URL,
);

# default
our $API_BASE_URL ||= $API_BASE_RO_URL;

=head2 @TIMEOUTS

    Таймауты запросов голема

=cut
our @TIMEOUTS = (1, 4);

=head1 METHODS

=head2 downtime_check

    Проверить, есть ли (либо планируется) downtime для указанного хоста
    Возвращает ссылку на массив [start, end] либо undef, если downtime не выставлен

=cut
sub downtime_check($) {
    my ($host) = @_;
    my $profile = Yandex::Trace::new_profile('golem:downtime:check');
    my $ret = _http_get("downtime/check", {object => $host, downtime_type => 'current_future'});
    chomp $ret;
    if ($ret =~ /^ok\s+(.*)$/s) {
        return [split /\s+/, $1, 2];
    } elsif ($ret eq 'ok') {
        return undef;
    } else {
        die "API call failed: $ret";
    }
}

=head2 firewall_expand_macro

    Раскрыть макрос фаервола
    Принимает один параметр - имя макроса
    Возвращает ссылку на массив и ip и сетей
    В случае ошибки - die

=cut
sub firewall_expand_macro($) {
    my ($macro) = @_;
    my $profile = Yandex::Trace::new_profile('golem:firewall:expand_macro');
    my $ret = _http_get("firewall/expand_macro", {macro => $macro});
    if ($ret =~ /^ok\s+(.*)/s) {
        return [map {s/\s//g; $_} split ' or ', $1];
    } else {
        die "API call failed: $ret";
    }
}

=head2 sms_send

    Отправить sms на несколько доменных логинов
    Принимает два параметра - массив логинов и текст сообщения

=cut
sub sms_send($$) {
    my ($resps, $text) = @_;
    my $profile = Yandex::Trace::new_profile('golem:sms:send');
    my $ret = _http_get("sms/send", {
        resps => join(',', @$resps),
        msg => $text
    });
    if ($ret =~ /^ok\s+(.*)/s) {
        return [map {s/\s//g; $_} split ' or ', $1];
    } else {
        die "API call failed: $ret";
    }
}

sub _http_get {
    my ($service, $params) = @_;

    my $api = ${ $API_BY_REQUEST{$service} || \$API_BASE_URL };

    my $url = $api."$service.sbml";
    if ($params && ref $params eq 'HASH' && keys %$params) {
        $url .= "?".join("&", map {uri_escape_utf8($_)."=".uri_escape_utf8($params->{$_})} sort keys %$params);
    } elsif ($params && ref $params eq 'ARRAY' && @$params) {
        $url .= "?".join("&", map {uri_escape_utf8($_->[0])."=".uri_escape_utf8($_->[1])} @$params);        
    } elsif (!ref $params) {
        $url .= "?$params" if defined $params;
    } else {
        die "Incorrect params: $params";
    }

    my @errors;
    for my $timeout (@TIMEOUTS) {
        my $ua = LWP::UserAgent->new(timeout => $timeout);
        # после переезда на precise eval можно удалить
        eval {
            $ua->ssl_opts(verify_hostname => 0);
        };
        $log->trace("GET $url");
        my $resp = $ua->get($url);
        if ($resp->is_success()) {
            return $resp->content();
        } else {
            push @errors, "Can't get $url: ".$resp->status_line;
        }
    }
    
    die join(", ", @errors);
}

1;
