package Yandex::HighlightWords;

=head1 NAME

    Yandex::HighlightWords
    Модуль для поиска в plain text слов, включенных в словари.
    Словари  могут хранится в базе данных, в файлах (одна строка должна содержать одну фразу)
    или словарь может передоваться в конструктор в виде ссылке на массив фраз.

=head1 DESCRIPTION

=cut

use utf8;
use strict;
use warnings;
use List::MoreUtils qw/ uniq /;
use Yandex::DBTools qw/get_dbh get_one_field_sql get_all_sql get_one_line_array_sql/;
use Yandex::MyGoodWords;
use Yandex::Trace;

=head2

    Таблицы, из которых умеет читать модуль, должны иметь следущую структуру:

    +-------------+--------------+------+-----+---------------------+----------------+
    | Field       | Type         | Null | Key | Default             | Extra          |
    +-------------+--------------+------+-----+---------------------+----------------+
    | phrase      | varchar(255) | NO   |     |                     |                |
    | marker      | varchar(20)  | NO   |     | red                 |                |
    | comment     | text         | YES  |     | NULL                |                |
    | timechanged | datetime     | NO   |     | 0000-00-00 00:00:00 |                |
    | id          | int(11)      | NO   | PRI | NULL                | auto_increment |
    | uid         | int(11)      | NO   |     | 0                   |                |
    | timecreated | datetime     | NO   |     | 0000-00-00 00:00:00 |                |
    +-------------+--------------+------+-----+---------------------+----------------+
    или
    +-------------+--------------+------+-----+---------------------+----------------+
    | Field       | Type         | Null | Key | Default             | Extra          |
    +-------------+--------------+------+-----+---------------------+----------------+
    | phrase      | varchar(255) | NO   |     |                     |                |
    | timechanged | datetime     | NO   |     | 0000-00-00 00:00:00 |                |
    +-------------+--------------+------+-----+---------------------+----------------+

=cut


our $WORD_DELIMETERS ||= qr/[^\w\d\%\$]+/;
our $DEFAULT_COMMENT ||= 'Обратите внимание! Фраза содержится в дополнительных списках для подсветки.';


=item new

    Конструктор.

    Конструктору необходимо передавать фразы или файлы со словарями и путь к ним (если он не включен в имя файлов).
    Также можно передать дескриптор для подключения к базе данных и имена таблиц содержащих словари или любое сочетание перечисленных вариантов.

    my $highlighter = Yandex::HighlightWords->new( {

        'phrases' => ['негодяй', 'редиска'],

        # список файлов со словарями
        'files' => ['last_names.txt', 'first_names.txt'],

        # путь к файлам со словарями, не должен заканчиваться слэшом "/", его подставит модуль
        'filespath' => '/etc/porno',

        # хендлер базы данных
        'dbh' => $dbh,

        # функция которая отвечает за "подсветку" найденных слов,
        # можно не передавать, тогда будет использоваться функция по умолчанию
        'regexp_func' => sub {
            my ( $h, $v ) = @_;
            return( '<b style="color: ' . $h->{'marker'} . '">' . $v . '</b>' );
        },

        # проверять обновления словарей каждый раз при вызове метода highlight
        'check_changes' => 0|1,

        # пауза в секундах перед повторной проверкой на изменения в словарях (по умолчанию 0 сек)
        'check_changes_wait_time' => 30,

        # читает базу из файлов только один раз при создании объекта
        'dont_update_from_files' => 1

    } );

=cut

sub new {

    my ( $this, $params ) = @_;
    my $class = ref( $this ) || $this;

    if ( ! $params || ref( $params ) ne "HASH" || (
         ! exists( $params->{'files'} ) && ! exists( $params->{'phrases'} ) &&
         ! exists( $params->{'tables'} ) && ! exists( $params->{'short_tables'} ) ) ) {
        die( 'You must specify one of mandatory options: files, phrases, tables or short_tables.' );
    }

    my $self = {
        'params'                  => {},
        'words_to_phrases_map'    => {}, # структура для подсветки "без учета порядка слов"
        'hilight_inited'          => 0,
        'phrases_qty_by_storage'  => {},
        'regexp_func'             => $params->{'regexp_func'} && ref( $params->{regexp_func} ) eq 'CODE' ? $params->{'regexp_func'} : \&original_regexp,
        'check_changes'           => defined( $params->{'check_changes'} ) ? $params->{'check_changes'} : 1,
        'check_changes_wait_time' => int( $params->{'check_changes_wait_time'} || 0 ),
        'dont_update_from_files'  => $params->{'dont_update_from_files'} || 0,
        'storage'                 => {}, # структура для подсветки "по умолчанию" (где все слова из словарной фразы должны встретиться подряд и в нужном порядке)
        'united_storage'          => {}, # глобальный словарь
        '_update_united_storage'  => 1,  # флаг, указывающий на то, что надо пересобрать глобальный словарь
        '_loaded_from_files'      => 0   # флаг, указывающий на то, что были загруженны данные из файла
    };

    bless( $self, $class );

    foreach my $tbl_param ( qw/tables short_tables/ ) {
        next if ( ! exists( $params->{ $tbl_param } ) );
        die( 'Database handle wasn not defined.' ) if ( ! $params->{'dbh'} );
        $self->{'params'}{'dbh'} = $params->{'dbh'};
        $self->{'params'}{'type'} = $params->{'type'};
        $self->{'params'}{ $tbl_param } = $params->{ $tbl_param };
    }

    if ( exists( $params->{'files'} ) ) { # определим списки файлов для подстветки

        $self->{'params'}{'highlight_filelist'} = $params->{'files'};

        if ( defined( $params->{'filespath'} ) ) {
            $self->{'params'}{'highlight_filelist'} = [ map { $params->{'filespath'} .'/' .$_ } @{ $self->{'params'}{'highlight_filelist'} } ];
        }
    }

    if ( exists( $params->{'phrases'} ) ) {
        $self->{'params'}{'phrases'} = $params->{'phrases'};
    }

    $self->reinitialize();
    return( $self );
}

=item _update_from_files_lists

    Перечитывает фразы для подстветки из файлов.
    Если перечитывать не откуда (не были указаны файлы), то вернёт 0.
    Иначе вернёт 1.

=cut

sub _update_from_files_lists {

    my ( $this ) = @_;

    if ( ! defined( $this->{'params'}{'highlight_filelist'} ) || ! scalar( @{ $this->{'params'}{'highlight_filelist'} } )
        || ( $this->{'_loaded_from_files'} && $this->{'dont_update_from_files'} ) ) {
        return( 0 );
    }

    foreach my $fname ( @{ $this->{'params'}{'highlight_filelist'} } ) {

        next if ( ! $fname || $fname =~ m/^\s*$/ );

        my ( $inode_number_old, $mtime_old ) = @{ $this->{'files_stat'}{ $fname } || [ 0, 0 ] };
        my ( $inode_number_new, $mtime_new ) = ( stat( $fname ) )[ 1, 9 ];

        my $need_update_from_file = ! $inode_number_old || ! $mtime_old || ! $inode_number_new || ! $mtime_new ||
                                    $inode_number_old != $inode_number_new ||
                                    $mtime_old != $mtime_new ||
                                    $mtime_new >= $this->{'hilight_inited'};

        next if ( ! $need_update_from_file ); # если файл не менялся, то нечего и обновлять

        $this->{'files_stat'}{ $fname } = [ $inode_number_new, $mtime_new ];
        $this->clean_storage($fname);

        if ( open( LIST_FH, "<:encoding(UTF-8)", $fname ) ) {

            $this->{'_update_united_storage'} = 1;

            my ( $file_name ) = $fname =~ m/([^\/\\]+)(?:\.\w+)?$/;

            while ( my $line = <LIST_FH> ) {

                chomp( $line );

                ( my $hilight_phrase = $line ) =~ s/^\s+|\s+$//g;

                next if ( $hilight_phrase  eq '' );

                $this->add_phrase( $hilight_phrase, {
                    'list'        => 1,
                    'comment'     => $DEFAULT_COMMENT,
                    'id'          => 0,
                    'marker'      => '#ff6633',
                    'source'      => $file_name,
                    'storage_key' => $fname
                } );
            }

            close( LIST_FH ) or warn "Can't close file($fname): $!";
        } else {
            warn "Can't open file($fname): $!";
        }
    }

    return( $this->{'_loaded_from_files'} = 1 );
}


=item _update_from_db_tables

    Перечитывает фразы для подстветки из таблиц в БД.
    Если перечитывать не откуда (не были указаны таблицы), то вернёт 0.
    Иначе вернёт 1.

=cut

sub _update_from_db_tables {

    my ( $this ) = @_;

    return( 0 ) if ( ! exists( $this->{'params'}{'dbh'} ) || ! ( exists( $this->{'params'}{'tables'} ) || exists( $this->{'params'}{'short_tables'} ) ) );

    my $dbh = $this->{'params'}{'dbh'};
    my $quoted_type = '';

    if ( $this->{'params'}{'type'} ) {
        my $dbh_object = ref( $dbh ) eq 'Yandex::DBTools::db' ? $dbh : get_dbh( $dbh );
        $quoted_type = $dbh_object->quote( $this->{'params'}{'type'} );
    }

    foreach my $tbl_param ( qw/tables short_tables/ ) {

        next if ( ! defined( $this->{'params'}{ $tbl_param } ) );

        foreach my $table_name ( @{ $this->{'params'}{ $tbl_param } } ) { # загружаем данные из БД

            next if ( $table_name !~ m/^[\w\-\_\d]+$/ );

            my $sql_modificate = "SELECT UNIX_TIMESTAMP(MAX(timechanged)), count(*) as qty FROM ${table_name}";
            my ($m_time, $phrases_qty) = get_one_line_array_sql( $dbh, $sql_modificate );

            next if ( $m_time <= $this->{'hilight_inited'} &&
                      ($this->{phrases_qty_by_storage}->{$table_name} // 0) == $phrases_qty ); # если данные в таблице не менялись, то нечего и обновлять
            
            $this->clean_storage($table_name);

            my ( $where_cond, $select_fields ) = ( '', '' );

            if ( $tbl_param eq 'tables' ) {
                $where_cond = $quoted_type ? 'where type = '. $quoted_type : '';
                $select_fields = 'id, phrase, marker, comment, unix_timestamp(timechanged) as timechange';
            } else {
                $select_fields = 'phrase, unix_timestamp(timechanged) as timechange';
            }

            my $sql = "select ${select_fields} from ${table_name} ${where_cond} order by timechange desc";
            my $hilight_phrases = get_all_sql( $dbh, $sql ) || [];

            $this->{'_update_united_storage'} = 1 if ( @{ $hilight_phrases } );

            # подготавливаем быстрый хэших для подсветки
            foreach my $hilight_phrase ( @{ $hilight_phrases } ) {
                $this->add_phrase( $hilight_phrase->{'phrase'}, {
                    'marker'      => $hilight_phrase->{'marker'} || 'red',
                    'comment'     => $hilight_phrase->{'comment'} || '',
                    'id'          => $hilight_phrase->{'id'} || 0,
                    'source'      => $table_name,
                    'storage_key' => $table_name
                } );
            }

            $this->{phrases_qty_by_storage}->{$table_name} = $phrases_qty;
        }
    }

    return( 1 );
}

=item _update_from_params_phrases

    Перечитывает фразы для подстветки из параметра "phrases".
    Если перечитывать не откуда (не был указан параметр), то вернёт 0.
    Иначе вернёт 1.

=cut

sub _update_from_params_phrases {

    my ( $this ) = @_;

    return( 0 ) if ( ! defined( $this->{'params'}{'phrases'} ) );
    return( 0 ) if ( scalar( keys( %{ $this->{'storage'}{'hardcoded'} || {} } ) ) );

    $this->{'_update_united_storage'} = 1;

    foreach my $phrase ( @{ $this->{'params'}{'phrases'} } ) {
        $this->add_phrase( $phrase, {
            'marker'      => 'red',
            'comment'     => $DEFAULT_COMMENT,
            'id'          => 0,
            'source'      => 'hardcoded',
            'storage_key' => 'hardcoded'
        } );
    }

    return( 1 );
}

=item reinitialize

    Проверяет необходимость обновления слов для подстветки и если надо, то обновляет их.
    Возвращает 0, если обновление словарей не требуется. иначе вернёт 1 (словари были обнволены).

=cut

sub reinitialize {

    my ( $self ) = @_;

    if ( $self->{'hilight_inited'} && $self->{'check_changes_wait_time'} &&
        ( time() - $self->{'hilight_inited'} ) < $self->{'check_changes_wait_time'} ) {
        return 0;
    }

    $self->_update_from_files_lists();
    $self->_update_from_db_tables();
    $self->_update_from_params_phrases();
    $self->_build_united_storage();

    $self->{'hilight_inited'} = time();

    return( 1 );
}


=item _build_united_storage

    Пересобирает глобальный словарь, если что-то менялось в локальных.

=cut

sub _build_united_storage {

    my ( $this ) = @_;

    if ( $this->{ '_update_united_storage' } ) {
        $this->{ 'united_storage' } = {};
        for my $storage (keys( %{ $this->{'storage'} // {}} )){
            my $storage_hash = $this->{'storage'}{ $storage } // {};
            for my $word (keys %$storage_hash){
                $this->{ 'united_storage' }{$word} //= [];
                push @{ $this->{ 'united_storage' }{$word} }, @{$storage_hash->{$word}};
            }
        }
        $this->{ '_update_united_storage' } = 0;
    }

    return( $this->{ 'united_storage' } || {} );
}


sub add_phrase {

    my ($self, $phrase, $params) = @_;
    my $only_this = $phrase =~ s/^\^//;
    my $ignore_word_position = $phrase =~ s/^\~//;
    my $storage_key = $params->{'storage_key'} || 'default';
    my $storage = $self->{'storage'}{ $storage_key } ||= {};
    my $tmp = [];
    my $i = 0;

    foreach my $orig_word ( split( /[^\w\d\!]+/, $phrase ) ) {

        my $equal_flag = ( $orig_word =~ s/^\!// );
        my $lc_word = lc( $orig_word );
        my $word = undef;

        $word = Yandex::MyGoodWords::norm_words( $orig_word ) if ( ! $equal_flag );
        $word = $lc_word if ( ! $word );

        push( @{ $tmp }, {
            'word' => $word,
            'orig_word' => $lc_word,
            'equal_flag' => $equal_flag,
            $i++ ? () : (
                'ignore_word_position' => $ignore_word_position ? 1 : 0,
                'only_this' => $only_this ? 1 : 0
            ),
            %{ $params },
        } );
    }

    if ( my $key = $tmp->[0]{'word'} ) {
        $storage->{ $key } ||= [];
        push( @{ $storage->{ $key } }, $tmp );
    }

    $self->{words_to_phrases_map}->{words} ||= {};
    $self->{words_to_phrases_map}->{phrases} ||= [];

    push @{$self->{'words_to_phrases_map'}->{'phrases'}}, {pattern => $tmp, storage => $storage_key};

    for (my $i = 0; $i < @$tmp; $i++) {

        next if ( ! defined( $tmp->[$i]{'word'} ) );

        my $words_map = $self->{'words_to_phrases_map'};
        push( @{ $words_map->{'words'}{ $tmp->[$i]{'word'} } }, { 'phrase_ind' => $#{$words_map->{'phrases'}}, 'word_pos' => $i, 
                                                                   equal_flag => $tmp->[$i]{equal_flag} } );
    }
}

sub clean_storage {
    my ($self, $storage_key) = @_;

    return unless $storage_key;

    $self->{'storage'}{ $storage_key } = {};
    delete $self->{phrases_qty_by_storage}->{$storage_key};

    # чистим в словах ссылки на фразы обновляемого источника, и удаляем слова, которые не ссылаются ни на одну фразу
    foreach my $word (keys %{$self->{words_to_phrases_map}->{words}}) {
        my $word_phrases = $self->{words_to_phrases_map}->{words}->{$word};
        $word_phrases = [grep { $self->{words_to_phrases_map}->{phrases}->[$_->{phrase_ind}]->{storage} ne $storage_key  } @$word_phrases];
        delete $self->{words_to_phrases_map}->{words}->{$word} unless @$word_phrases;
    }

    # вычисляем смещение индексов фраз после удаления
    my %phrases_ind_shift = ();
    foreach my $i (0 .. $#{$self->{words_to_phrases_map}->{phrases}}) {
        if ($self->{words_to_phrases_map}->{phrases}->[$i]->{storage} eq $storage_key) {
            foreach my $j ($i+1 .. $#{$self->{words_to_phrases_map}->{phrases}}) {
                $phrases_ind_shift{$j}--;
            }
        }
    }

    # удаляем фразы обновляемого источника
    $self->{words_to_phrases_map}->{phrases} = [ grep { $_->{storage} ne $storage_key } @{$self->{words_to_phrases_map}->{phrases}}];

    # применяем смещение индексов в ссылках на фразы
    foreach my $word_phrases ( values %{$self->{words_to_phrases_map}->{words}}) {
        foreach (@$word_phrases) {
            if (exists $phrases_ind_shift{$_->{phrase_ind}}) {
                $_->{phrase_ind} += $phrases_ind_shift{$_->{phrase_ind}};
            }
        }
    }
}

sub phrases_list {

    my $self = shift(@_);

    return(
        sort(
            map { map {
                ( $_ -> [ 0 ] -> { 'only_this' } ? '^' : '' )
                . ( $_ -> [ 0 ] -> { 'ignore_word_position' } ? '~' : '' )
                . join( ' ', map {
                    ( $_ -> { 'equal_flag' } ? '!' : '' )
                    . $_->{'orig_word'}
                } @$_ )
            } @$_ }
            map { values( $self->{'storage'}{ $_ } ) } keys( %{ $self->{'storage'} } )
        )
    );
}

#------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

=head2 highlight

    обертка к функциям подсветки словарных фраз в тексте

    my $options = {};   # по умолчанию
    my $comments = {};

    my ($highlighted_text, $highlight_flag) = $highlighter->highlight($text, $comments, $options);
        $highlighted_text - измененный(=подсвченный) текст, если не указано в options->{not_change_phrase}(иначе исходная фраза)
        $highlight_flag - флаг, сработала ли подсветка на данной фразе
        $comments - ссылка на хэш, куда будет сохранены комментарии к найденным фразам, если они есть

    Флаги $options:

            not_change_phrase - не подсвечивать текст, только считать факт подсветки.
                            Применяется для ускорения проверки т.к. не выполняется regexp, подсвечивающий текст.

            un_check - ссылка на массив фраз, которые не нужно учитывать при подсветке.
                            Из-за того, что не требуется изменять списки и создавать новый объект класса ускоряет работу.

            ignore_word_position - словарная фраза подсвечивается в тексте, если все слова из словарной фразы входят в текст (в любом порядке)

            debug - вместо подсветки фразы в тексте - печатает в STDOUT подсвечиваемые фразы из текста и списков, а также название списка в csv формате.
                            Иногда удобно применяется при отладке, а также при построении различных отчетов.

   Флаги в подсвечиваемой строке
        - если в начале слова стоит восклицательный знак - ищется точное соотвествие (без учета словоформ)
        - если в начале строки стоит крышка (^) - ищется точная копия строки (но каждое отдельное слово может быть в
                                                            разных словоформах - зависит от наличия \!)

=cut

sub highlight {

    my ( $self, $text, $link_comments, $options ) = @_;
    my $res = { 'text' => $text };
    
    my $profile = Yandex::Trace::new_profile('highlight_words:highlight');

    $link_comments ||= {};
    $options ||= {};

    if ( $text && $text !~ m/^\s+$/s ) {

        $self->reinitialize() if ( $self->{'check_changes'} );

        my $highlight_method = $options->{'ignore_word_position'} ? '_hilight_phrase_ignore_word_position' : '_hilight_phrase';

        $res = $self->$highlight_method( $text, $options );

        foreach my $key ( keys( %{ $res->{'comments'} } ) ) {
            $link_comments->{ $key } = $res->{'comments'}{ $key };
        }
    }

    return( $res ) if ( ! wantarray() );
    return( map { $res->{$_} } qw/text flag highlighted_words sources splitted_text/ );
}

#------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

=head2 _hilight_phrase

    метод по подсветке словарных фраз в заданном тексте, алгоритмом "по-умолчанию":
        - словарная фраза считается входящей в текст если все слова входят в текст подряд и в неизменном порядке

=cut

sub _hilight_phrase {

    my ( $self, $text, $options ) = @_;
    my @mass = _norm_words_mass( $text ); # массив нормализованного текста для поиска
    my @text_for_replace = split( /($WORD_DELIMETERS)/, $text );

    # делаем список исключений слов для подсветки
    my %not_highlight = _not_highlight_norm_words($options->{un_check});

    my %comments = ();
    my %response = ( text => $text, comments => {},flag => 0, highlighted_words => {}, names => {}, sources => {} );
    my $knowledge_base = $self->{ 'united_storage' } || {};

    # собственно сверхбыстрая проверка ;-)

    for(my $i = 0; $i <= $#mass; $i++) {

        # выбираем слово за словом
        my $word = $mass[$i];

        my $orig_word = lc($word->{orig_word});
        my $norm_word = $word->{norm_word};

        # Нашли слово с корого начинается фраза в списке подсветки
        if ( (exists $knowledge_base->{$orig_word} || exists $knowledge_base->{$norm_word}) && !defined $not_highlight{$norm_word} ) {

            # найденное слово в списках подсветки
            # если есть нормализованная форма - ищем ее, иначе - оригинальную фразу
            my $find_word = exists $knowledge_base->{$norm_word} ? $norm_word : $orig_word;

            # если фраза в списках многословная - проверяем "правильную" последовательность
            # Проверим все фразы начинающиеся с найденного слова..
            for(my $k = 0; $k < @{$knowledge_base->{$find_word}}; $k++) {

                my $pattern = $knowledge_base->{$find_word}->[$k];

                if( $pattern -> [ 0 ] -> { 'ignore_word_position' } ) {

                    my %words = ();

                    for (my $j = 0; $j < @$pattern; $j++) {

                        push( @{ $words{ $pattern -> [ $j ] -> { 'word' } } }, {
                            phrase_ind => 0,
                            word_pos => $j,
                            equal_flag => $pattern->[ $j ]->{equal_flag},
                        } );
                    }

                    $self -> _hilight_phrase_ignore_word_position_impl( {
                        text => $text,
                        options => $options,
                        mass => \@mass,
                        not_highlight => \%not_highlight,
                        comments => \%comments,
                        response => \%response,
                        text_for_replace => \@text_for_replace,
                        words_to_phrases_map => {
                            phrases => [
                                {
                                    pattern => $pattern,
                                    storage => $pattern -> [ 0 ] -> { 'storage_key' },
                                },
                            ],
                            words => \%words,
                        },
                    } );

                } elsif ($self->_is_pattern_in_text(\@mass, $i, $pattern, \%not_highlight)) {
                    # save comment to spec hash
                    $comments{$pattern->[0]->{id} || 0} = $pattern->[0]->{comment};

                    for (my $j = 0; $j < @$pattern; $j++) {
                        # "подсвечиваем" всю фразу в тексте - чтобы не проверять второй раз
                        if (defined $mass[$i + $j]) {
                           foreach my $key (qw/id marker comment fname source/) {
                               $mass[$i + $j]->{$key} = $pattern->[$j]->{$key};
                           }

                           $mass[$i + $j]->{word_from_list}=$pattern->[$j]->{word};

                           $self->_add_word_to_response($mass[$i + $j], \%response);

                           if ($options->{debug}) {
                               my $word = $mass[$i + $j];
                               warn "$word->{orig_word};$word->{norm_word};".($word->{fname} || $word->{source} ).";\n";
                           }
                        }
                    }

                    $text = $self->_apply_regexp(\@mass, 
                                                 [$i .. $i + (scalar @$pattern) - 1], 
                                                 \@text_for_replace,
                                                 $options->{regexp_argv},
                                                ) if !$options->{not_change_phrase};

                    $self->_add_phrase_to_response(\%response, $pattern);
                }
            }
        }
    }

    $text = join("", @text_for_replace);

    if (wantarray) {
        return ($text, \%comments, $response{flag}, $response{highlighted_words}, $response{sources}, \@text_for_replace);
    }

    $response{comments} = \%comments;
    $response{text} = $text;
    $response{splitted_text} = \@text_for_replace;

    return \%response;
}


=head2 _hilight_phrase_ignore_word_position

    метод по подсветке словарных фраз в заданном тексте, без учета порядка слов:
        - словарная фраза считается входящей в текст если все слова фразы входят в текст, в любом месте

=cut

sub _hilight_phrase_ignore_word_position
{
    my $self = shift;
    my $text = shift;
    my $options = shift;

    # массив нормализованного текста для поиска
    my @mass = _norm_words_mass($text);

    my @text_for_replace = split( /($WORD_DELIMETERS)/, $text );

    # делаем список исключений слов для подсветки
    my %not_highlight = _not_highlight_norm_words($options->{un_check});

    my %comments = ();
    my %response = ( text => $text, comments => {},flag => 0, highlighted_words => {}, names => {}, sources => {} );

    $self -> _hilight_phrase_ignore_word_position_impl( {
        text => $text,
        options => $options,
        mass => \@mass,
        not_highlight => \%not_highlight,
        comments => \%comments,
        response => \%response,
        text_for_replace => \@text_for_replace,
        words_to_phrases_map => $self -> { 'words_to_phrases_map' },
    } );

    $text = join("", @text_for_replace);

    if (wantarray) {
        return ($text, \%comments, $response{flag}, $response{highlighted_words}, $response{sources}, \@text_for_replace);
    }

    $response{comments} = \%comments;
    $response{text} = $text;
    $response{splitted_text} = \@text_for_replace;
    
    return \%response;
}

sub _hilight_phrase_ignore_word_position_impl {

    my ( $self, $context ) = @_;

    my ( $text, $options, $mass, $not_highlight, $comments, $response,
        $text_for_replace, $words_to_phrases_map ) = @$context{ 'text', 'options',
        'mass', 'not_highlight', 'comments', 'response', 'text_for_replace',
        'words_to_phrases_map' };

    my %highlight_phrases = ();

    keys( my %processed_words ) = scalar( keys( %{ $words_to_phrases_map->{'words'} } ) );

    for( my ( $i, $j ) = ( 0, 0 ); $i <= $#$mass; $i++) {

        # выбираем слово за словом
        my $word = $mass->[$i];

        my $orig_word = lc($word->{orig_word});
        my $norm_word = $word->{norm_word};
        
        if ( defined $words_to_phrases_map->{words}{$orig_word} 
            && $words_to_phrases_map->{words}{$orig_word}[0]{equal_flag} ) {
            $norm_word = $orig_word;
        }

        next if ( ! $words_to_phrases_map->{words}{$norm_word} || $not_highlight->{$norm_word} );

        if ( ! $processed_words{ $norm_word } ) {
            $processed_words{ $norm_word } = ++$j;
        }

        foreach my $data ( @{ $words_to_phrases_map->{'words'}{ $norm_word } } ) {
  
            my ( $phrase_index, $word_position ) = @{ $data }{ qw/ phrase_ind word_pos / };

            push( @{ $highlight_phrases{ $phrase_index } }, {
                'word_num' => "$phrase_index $word_position",
                'orig_phrase_word_pos' => $i,
                'dict_phrase_word_pos' => $word_position
            } );
        }
    }

    foreach my $phrase_index ( keys( %highlight_phrases ) ) {

        my $phrase = $words_to_phrases_map->{'phrases'}->[ $phrase_index ];

        # это хеши с данными слов из переданной в конструктор фразы
        my $pattern = $phrase->{'pattern'} // [];
        my $words_cnt = scalar( uniq( map { $_->{'word_num'} } @{ $highlight_phrases{ $phrase_index } }));

        # если кол-во слов в шаблоне меньше, чем в текущей фразе, то ищем дальше
        next if ( $words_cnt != scalar( @{ $pattern } ) );

        $comments->{ $pattern->[0]->{'id'} || 0 } = $pattern->[0]->{'comment'};

        foreach my $word_intersect ( @{ $highlight_phrases{ $phrase_index } } ) {

            # "подсвечиваем" всю фразу в тексте - чтобы не проверять второй раз
            my ( $orig_phrase_wp, $dict_phrase_wp ) = @{ $word_intersect }{ qw/ orig_phrase_word_pos dict_phrase_word_pos / };

            foreach my $key (qw/id marker comment fname source/) {
                $mass -> [$orig_phrase_wp]->{$key} = $pattern->[$dict_phrase_wp]->{$key};
            }

            $mass->[$orig_phrase_wp]->{word_from_list}=$pattern->[$dict_phrase_wp]->{word};

            $self->_add_word_to_response($mass -> [$orig_phrase_wp], $response);

            if ($options->{debug}) {
               my $word = $mass -> [$orig_phrase_wp];
               warn "$word->{orig_word};$word->{norm_word};".($word->{fname} || $word->{source} ).";\n";
            }
        }

        $text = $self->_apply_regexp($mass,
                                     [ map { $_->{orig_phrase_word_pos} }
                                         @{$highlight_phrases{$phrase_index}} ],
                                     $text_for_replace,
                                     $options->{regexp_argv}) if !$options->{not_change_phrase};
        $self->_add_phrase_to_response($response, $pattern);
    }

    return;
}

sub _norm_words_mass {

    my $text = shift;
    my @result = ();

    foreach my $word ( grep { defined( $_ ) && $_ ne '' } split( /$WORD_DELIMETERS/, $text ) ) {
        push( @result, {
            'orig_word'     => $word,
            'norm_word'     => Yandex::MyGoodWords::norm_words( $word ) || lc( $word ),
            'highlite_flag' => 0,
        } );
    }

    return( @result );
}

sub _not_highlight_norm_words {
    my $words_list = shift || [];
    my %not_highlight = ();
    foreach my $w (@$words_list) {
        my $norm_word = Yandex::MyGoodWords::norm_words($w) || lc($w);
        $not_highlight{$norm_word} = 1;
    }
    return %not_highlight;
}

sub _is_pattern_in_text
{
    my ($self, $text, $start, $pattern, $not_highlight) = @_;

    #Если мы ищем точный набор слов (ни больше ни меньше), то проверим что нет ничего лишнего.
    if ( $pattern->[0]->{only_this}
         && scalar @$text != scalar @$pattern ) {
        return 0;
    }

    for(my $j = 1; $j < @$pattern; $j++) {
        if (defined $text->[$start + $j] && $text->[$start + $j] ne '') {

            # подготавливаем слова для сравнения
            my $orig_text_word = lc($text->[$start + $j]->{orig_word});
            my $norm_text_word = lc($text->[$start + $j]->{norm_word});
            my $word_from_pattern = lc($pattern->[$j]->{word});

            if (($word_from_pattern ne $orig_text_word
                 && $word_from_pattern ne $norm_text_word) || defined $not_highlight->{$norm_text_word} ) {
                return 0;
            }

        } else {
           return 0;
        }
    }

    return 1;
}

sub _add_word_to_response
{
    my ($self, $word, $response) = @_;

    $response->{flag} = 1;

    push @{ $response->{highlighted_words}{ $word->{orig_word} } }, ( $word->{fname} || '');
    $response->{names}{ $word->{fname} || $word->{source} } = 1 if defined $word->{fname} || defined $word->{source} ;
    $response->{sources}{ $word->{orig_word}.'_'.$word->{word_from_list} } = $word->{source} if exists $word->{source};
}

sub _apply_regexp
{
    my ($self, $mass, $index_list, $text, $argv) = @_;

    my $p = 0;

    #Текст разбит с помощью split(/(spaces)/) и если первый символ строки это spaces, то первый элемент списка будет пустой.
    if ($text->[0] eq '') {
        $p = 2;
    }

    foreach my $i ( @$index_list ) {
        my $real_position = $i * 2 + $p; #вычисляем индекс слова в исходном тексте с учетом  разделителей между словами

        #если еще не обрабатывали
        if ( defined( $mass->[$i]{'orig_word'} ) && defined( $text->[ $real_position ] ) && $mass->[$i]{'orig_word'} eq $text->[ $real_position ] )  {
            $text->[$real_position] = $self->{regexp_func}->($mass->[$i], $text->[$real_position], $text, $real_position, $argv);
        }
    }
}

sub _add_phrase_to_response
{
    my ($self, $response, $phrase) = @_;
    my @phrase = map { $_->{orig_word} } @$phrase;
    push @{ $response->{highlighted_phrases} }, \@phrase;
}

#------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

sub get_files_list{
    my $self = shift;
    return join(', ', map {/([^\/]+)$/} @{$self->{params}->{highlight_filelist}});
}

#------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

sub original_regexp {
    my ($h, $v) = @_;
    my $marker = $h->{marker} || 'red';
    my $id = $h->{id} || 0;
    return '<b style="color: ' . $marker . '" onMouseOver="show_comment(' . $id . ', this)" onMouseOut="hide_comment(' . $id . ');"\>' . $v . '</b>';
}

#------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

1;
__END__
