#!/usr/bin/perl -w

use strict;
use List::Util qw/max/;
use Test::More;
use Test::Deep;

use Yandex::DBTools;

use Yandex::HighlightWords;
use Encode;

use utf8;
use open ':std' => ':utf8';


my %dict = (
        'отслеживание' => undef,
        'отслеживание посылок почты россии' => 'отслеживание посылок [почты] [россии]',
        'отправить посылку почтой россии' => 'отправить посылку [почтой] [россии]',
        'отслеживание посылок почты' => undef,
        'Lg bp 450 в наличии Всего за 5490 руб. Самовывоз бесплатно. Доставка 350 руб!' => undef,
        'отправить посылку почтой в россии' => 'отправить посылку [почтой] в [россии]',
        'отправить посылку "почтой в россии"' => 'отправить посылку "[почтой] в [россии]"',
        'посылка с зеленым кофе' => 'посылка с [зеленым] [кофе]',
        'россия отправить посылку почтой' => '[россия] отправить посылку [почтой]',
        'отправка зеленого почтой кофе россии' => 'отправка [зеленого] [почтой] [кофе] [россии]',
        'отправка почтой зеленого россии кофе' => 'отправка [почтой] [зеленого] [россии] [кофе]',
);

my @dict2 = (
    'доставка бананов и доставка огурцов на слонах'
);
Test::More::plan(tests => ( 3 * scalar( keys %dict ) + scalar( @dict2 ) )+1 );

my $regexp_func = sub { return "[".$_[1]."]" };


sub hi {
    my ($h_obj, $word) = @_;
    my $response =  $h_obj->highlight($word, undef, {ignore_word_position => 1});
    return undef unless $response->{flag};
    return $response;
};

sub hi2 {
    my ($h_obj, $word) = @_;
    my $response =  $h_obj->highlight($word, undef, {});
    return undef unless $response->{flag};
    return $response;
};


my $h_obj = Yandex::HighlightWords->new({ phrases => ['почта россии', 'зеленый кофе'],
                                          regexp_func => $regexp_func });

my $h_obj2 = Yandex::HighlightWords->new({ phrases => ['~почта россии', '~зеленый кофе', 
                                                       '',],
                                          regexp_func => $regexp_func });

while (my ($k, $v) = each %dict) {
    my $result = hi($h_obj, $k);
    is (($result//{})->{text}, $v, 'Highlighted phrase: '.Encode::encode_utf8($k));

    my $result2 = hi2($h_obj2, $k);

    # для highlighted_phrases одинаковый порядок гарантировать не получается:
    # при ignore_word_position = 0 "подсвеченные" фразы упорядочены по порядку вхождения первых слов фраз в анализируемый текст, 
    # а при ignore_word_position = 1 -- по перемешанному (итерация по keys) порядку фраз в словаре, 
    # и к словам исходного текста перепривязывать тяжело, да и незачем
    # Поэтому отдельно сравниваем целые результаты без highlighted_phrases, отдельно highlighted_phrases без учета порядка
    my $hph1 = delete $result->{highlighted_phrases} || [];
    my $hph2 = delete $result2->{highlighted_phrases} || [];
    is_deeply( $result, $result2, 'hi() and hi2() result are the same' );
    cmp_bag($hph1, $hph2, "hi() and hi2() give the same highlighted_phrases up to the order of phrases");
}

# проверяем, что корректно работает в случае если хотя бы одно слово из фразы встретилось столько же раз, сколько и слов во фразе
# DIRECTMOD-3472

my $h_obj3 = Yandex::HighlightWords->new( {
    'phrases' => [ '~доставка алкоголя' ],
    'regexp_func' => $regexp_func
} );

foreach my $text ( @dict2 ) {

    my $response = $h_obj3->highlight( $text, {}, {} );

    ok( ! $response->{'flag'}, 'Do not hilight if at least one word met as many times as there are words in the phrase.' );
}

my $h_obj4 = Yandex::HighlightWords->new( {
    'phrases' => [ 'доставка бесплатного вкусного алкоголя домой', 
                   'наличие большого ума защитит сложной'],
    'regexp_func' => $regexp_func
} );

my $result =$h_obj4->highlight( "наличие доставка наличие  бесплатного наличие  вкусного наличие алкоголя наличие" , {}, {ignore_word_position => 1} );
ok( !$result->{'flag'}, 'Do not hilight if at least one word met as many times as there are words in the phrase.' );


1;
__END__
