#!/usr/bin/perl -w

use strict;
use File::Temp qw/tempfile/;
use File::Slurp;
use Time::HiRes qw/usleep/;
use Test::More tests => 40;
use Yandex::HighlightWords;
use List::MoreUtils qw/any/;

use Yandex::DBTools;
use Yandex::DBUnitTest qw/:all/;

=pod
    usleep-ы добавлены, чтобы обойти странное поведение time/stat в perl

    Linux ppcdev.yandex.ru 2.6.24-21-openvz #1 SMP Wed Oct 22 01:43:28 UTC 2008 x86_64 GNU/Linux
    This is perl, v5.8.8 built for x86_64-linux-gnu-thread-multi

    perl -e 'use File::Slurp; for(1..1e5) {my $t = time; write_file("1", 1); my $t1 = (stat(1))[9]; if ($t1 < $t) {die "Shit happens! $t $t1 $_"}}'
=cut

my (undef, $list_a) = tempfile(UNLINK => 1);
my (undef, $list_b) = tempfile(UNLINK => 1);

write_file($list_a, {atomic => 1}, "A");
write_file($list_b, {atomic => 1}, "B");

my $h = Yandex::HighlightWords->new({ files => [$list_a, $list_b], check_changes => 1 });

ok([$h->highlight("A D")]->[1], "Highlighted from A");
ok([$h->highlight("B D")]->[1], "Highlighted from B");
ok(! [$h->highlight("E D")]->[1], "Not highlighted");

write_file($list_a, {atomic => 1}, "C");

ok(! [$h->highlight("A D")]->[1], "C reinitialized - no A");
ok([$h->highlight("C D")]->[1], "C reinitialized - ok C");
ok([$h->highlight("B D")]->[1], "C reinitialized - ok B");

write_file($list_a, {atomic => 1}, "X");

ok(! [$h->highlight("C D")]->[1], "X reinitialized - no C");
ok([$h->highlight("X D")]->[1], "X reinitialized - ok X");
ok([$h->highlight("B D")]->[1], "X reinitialized - ok B");

write_file($list_a, {atomic => 1}, "C");

ok(! [$h->highlight("A D", undef, {ignore_word_position => 1})]->[1], "C reinitialized - no A");
ok([$h->highlight("C D", undef, {ignore_word_position => 1})]->[1], "C reinitialized - ok C");
ok([$h->highlight("B D", undef, {ignore_word_position => 1})]->[1], "C reinitialized - ok B");

# test not check changed
my (undef, $list_x) = tempfile(UNLINK => 1);
write_file($list_x, {atomic => 1}, "X");
my $h2 = Yandex::HighlightWords->new({ files => [$list_b, $list_x], check_changes => 0});
write_file($list_x, {atomic => 1}, "A B C");

ok(! [$h2->highlight("C D")]->[1], "X reinitialized - no C");
ok([$h2->highlight("X D")]->[1], "X reinitialized - ok X");
ok([$h2->highlight("B D")]->[1], "X reinitialized - ok B");

# test check changing

write_file($list_x, "X");
my $h3 = Yandex::HighlightWords->new({ files => [$list_b, $list_x], check_changes => 1});
write_file($list_x, {atomic => 1}, "Y");

ok(! [$h3->highlight("C D")]->[1], "X reinitialized - no C, no D");
ok(! [$h3->highlight("X D")]->[1], "X reinitialized - not X");
ok([$h3->highlight("Y D")]->[1], "X reinitialized - ok Y");

# test wait pause before checking
write_file($list_a, {atomic => 1}, "A");
write_file($list_b, {atomic => 1}, "B");
my $h4 = Yandex::HighlightWords->new({ files => [$list_a, $list_b], check_changes => 1, check_changes_wait_time => 2 });

ok([$h4->highlight("A D")]->[1], "Highlighted from A");
ok(! [$h4->highlight("C D")]->[1], "Not highlighted");
ok(! [$h4->highlight("E D")]->[1], "Not highlighted");

write_file($list_a, {atomic => 1}, "C");

ok([$h4->highlight("A D")]->[1], "Not reinitialized - Highlighted from A");
ok(! [$h4->highlight("C D")]->[1], "Not reinitialized - Not highlighted");
ok(! [$h4->highlight("E D")]->[1], "Not reinitialized - Not highlighted");

sleep(2);
ok(! [$h4->highlight("A D")]->[1], "C reinitialized - no A");
ok([$h4->highlight("C D")]->[1], "C reinitialized - ok C");
ok(! [$h4->highlight("E D")]->[1], "C reinitialized - no E");

# if files weren't changed (inode, mtime), that they shouldn't be updated.

my ( $inode_old, $mtime_old ) = @{ $h4->{'files_stat'}{ $list_a } };

sleep(2); # wait 2 seconds
$h4->reinitialize(); # in case of error it's will be change the access time

my ( $inode_new, $mtime_new ) = @{ $h4->{'files_stat'}{ $list_a } };

# если одно из значений undefined, или они поменялись, то файл обновлялся
ok( ! ( any { ! defined } $inode_new, $mtime_new, $inode_old, $mtime_old ) ||
    $inode_new != $inode_old && $mtime_new != $mtime_old,
    "Ok: files weren't updated because they weren't changed."
);

write_file( $list_a, { 'atomic' => 1 }, 'A' );
sleep(2); # wait 2 seconds
$h4->reinitialize(); # should update from files

( $inode_new, $mtime_new ) = @{ $h4->{'files_stat'}{ $list_a } };

ok( $inode_new != $inode_old || $mtime_new != $mtime_old, "Ok: files were updated." );


# проверяем корректность перечитки данных из БД 
create_table(UT, 'test' => 'phrase:varchar(255):pk:not_null', 'timechanged:timestamp');
insert_phrase('test', $_) foreach (qw/A B/);

my $h5 = new Yandex::HighlightWords({dbh => UT,
                                     short_tables => ['test']});
ok([$h5->highlight("A")]->[1], "Init from DB, ok A");

sleep(1);
insert_phrase('test', 'C');
ok([$h5->highlight("C")]->[1], "Init from DB, C added - ok C");

delete_phrase('test', 'C');
ok(![$h5->highlight("C")]->[1], "Init from DB, C deleted - no C");

sleep(1);
modify_phrase('test', 'A', 'D');
ok([$h5->highlight("D")]->[1], "Init from DB, A => D modified - ok D");

sleep(1);
modify_phrase('test', 'D', 'E', 'dont_change_timestamp');
ok([$h5->highlight("D")]->[1], "Init from DB, D => E modified (ts not changed) - ok D");
ok(![$h5->highlight("E")]->[1], "Init from DB, D => E modified (ts not changed) - no E");

# проверяем корректность работы с несколькими источниками
insert_phrase('test', 'F');
# ошибка может быть нестабильной из-за порядка ключей в keys
for(1 .. 5) {
	my $h6 = new Yandex::HighlightWords({
		phrases	=> ['F A'],
		dbh => UT,
		short_tables => ['test'],
	});

	ok([ $h6->highlight("F G") ]->[1], "Init from phrases+DB, ok: ".$_);
}

sub insert_phrase {
	my ($table, $ph) = @_;
	do_insert_into_table(UT, $table, {phrase => $ph, timechanged__dont_quote => 'CURRENT_TIMESTAMP()'});
}

sub delete_phrase {
	my ($table, $ph) = @_;
	do_delete_from_table(UT, $table, where => {phrase => $ph});
}

sub modify_phrase {
	my ($table, $ph, $ph_new, $dont_change_timestamp) = @_;
	do_update_table(UT, $table, {phrase => $ph_new, 
								 timechanged__dont_quote => $dont_change_timestamp ? 'timechanged' : 'CURRENT_TIMESTAMP()'},
								 where => { phrase => $ph });
}

1;
__END__
