package Yandex::HTTP;

# $Id$

=head1 NAME
    
    Yandex::HTTP
    Модуль для работы с http запросами

=head1 DESCRIPTION

=cut

use 5.010;
use strict;
use warnings;
use utf8;

use Carp;
use Encode;
use Time::HiRes;
use HTTP::Request;
use HTTP::Headers;
use LWP::UserAgent;
use URI::Escape::XS qw/uri_escape/;
use List::MoreUtils qw/ natatime /;
use JSON;

use Yandex::HashUtils qw/ hash_cut hash_copy/;
use Yandex::IDN;

use AnyEvent;
use AnyEvent::HTTP qw//;
use AnyEvent::HTTP::Response;

use vars qw/@ISA @EXPORT/;

=head2 local $Yandex::HTTP::DEFAULT_CONNECT_TIMEOUT = 4.0;

    Таймаут на коннект, значение по-умолчанию - 4 сукунды, в расчёте на один ретрансмит
    Пока работает только для http_parallel_request

=cut
our $DEFAULT_CONNECT_TIMEOUT ||= 4;

=head2 local $Yandex::HTTP::PARALLEL_REQUEST_CHUNK_SIZE = 3_000;

    Количество заданий (url-ов) которые параллельно обрабатываются в http_parallel_request
    Если пришло заданий больше этого количества - они все делятся на части по $PARALLEL_REQUEST_CHUNK_SIZE штук,
    и полученные части обрабатываются последовательно, друг за другом.
    Если $PARALLEL_REQUEST_CHUNK_SIZE = 0 - считаем что ограничения нет.
    Предзаданное значение 3_000 получено экспериментальным путем (при 10_000 происходило переполнение стека).
    
=cut
our $PARALLEL_REQUEST_CHUNK_SIZE;
$PARALLEL_REQUEST_CHUNK_SIZE = 3_000 if !defined $PARALLEL_REQUEST_CHUNK_SIZE;

=head2 $Yandex::HTTP::DISABLE_STDERR_SPAM

Запрет вывода в stderr в http_get и http_post

=cut

our $DISABLE_STDERR_SPAM;

=head2 $Yandex::HTTP::HTTP_FETCH_DEFAULT_TIMEOUT

    Таймаут по умолчанию для http_fetch в секундах

=cut

our $HTTP_FETCH_DEFAULT_TIMEOUT ||= 60;


=head2 $Yandex::HTTP::DEFAULT_CHARSET

Кодировка для ответа по умолчанию

=cut

our $DEFAULT_CHARSET ||= "UTF-8";

require Exporter;
@ISA         = qw(Exporter);
@EXPORT      = qw(
    http_get
    http_post
    submit_form
);

our @EXPORT_OK = qw(
    http_get
    http_post
    submit_form
    http_parallel_request
    http_fetch
);
    
=head2 http_get

  my $content = http_get($url, %lwp_options
=cut

sub http_get($;%)
{
    my ($url, %lwp_options) = @_;
    my $response = submit_form('GET', $url, {}, %lwp_options);

    if ($response->is_success()) {
        return $response->decoded_content();
    } else {
        print STDERR "$0 error \nresponse-code: " . $response->code  if !$DISABLE_STDERR_SPAM;
        return;
    }
}


=head2 http_post

  my $content = http_post($url, \%form, %lwp_options);
  or
  my $content = http_post($url, $post_content, %lwp_options);

=cut

sub http_post
{
    my $response = submit_form('POST', @_);

    if ($response->is_success()) {
        return $response->decoded_content();
    } else {
        print STDERR "$0 error \nresponse-code: " . $response->code . "\ncontent: " . $response->decoded_content()  if !$DISABLE_STDERR_SPAM;
        return;
    }
}


=head2 submit_form

    $resp = submit_form('GET', $url, {uid => $uid, sum => $sum})
    $resp = submit_form('GET', $url, [uid => $uid_1, uid => $uid_2 ])

    $resp = submit_form('POST', $url, {uid => $uid, sum => $sum})
    $resp = submit_form('POST', $url, [uid => $uid_1, uid => $uid_2 ])

    $resp = submit_form('PUT', $url, $content)

    print $resp->content;
    print "OK" if $resp->is_success();
    print $resp->code;

=cut

sub submit_form($$$;%)
{
    my ($method, $url, $form, %options) = @_;
    die "incorrect method name $method" if $method ne uc($method);

    if ($options{headers}) {
        $options{default_headers} //= HTTP::Headers->new();
        $options{default_headers}->header(%{ delete $options{headers} });
    }

    $method = lc($method);
    if ($method eq 'post') {
        return LWP::UserAgent->new(%options)->post($url, Content => $form);
    } elsif ($method eq 'get' || $method eq 'head') {
        my $full_url = make_url($url, $form);
        return LWP::UserAgent->new(%options)->$method($full_url);
    } elsif ($method eq 'put') {
        my $request = HTTP::Request->new(PUT => $url);
        $request->content(encode 'utf8', $form);
        return LWP::UserAgent->new(%options)->request($request);
    } elsif ($method eq 'delete') {
        my $request = HTTP::Request->new(DELETE => $url);
        return LWP::UserAgent->new(%options)->request($request);
    } else {
        die "unsupproted method $method";
    }
}


=head2 make_url

    Собирает url для отправки формы через GET. 
    Параметры позиционные

    make_url('direct.yandex.ru', {login => 'asdf', sum => 3})
    ==> "direct.yandex.ru?login=asdf&sum=3"

    make_url('direct.yandex.ru', [login => 'asdf', login => 'zxcv'])
    ==> "direct.yandex.ru?login=asdf&login=zxcv"

    make_url('direct.yandex.ru?login=asdf', {sum => 3})
    ==> "direct.yandex.ru?login=asdf&sum=3"

    make_url('', {login => 'asdf', sum => 3})
    ==> "login=asdf&sum=3"

=cut 

sub make_url($$)
{
    my ($url, $form) = @_;

    my $paramstr = get_param_string($form);

    return $paramstr        if !$url;
    return $url             if !$paramstr;
    return "$url&$paramstr" if $url =~ /\?/;
    return "$url?$paramstr";
}



=head2 get_param_string( $params, %opt )

    Разворачиваем и сериализуем параметры в строку запроса
    Параметры можно задать:
        * готовая строка - тогда тупо её и возвращаем
        * { param => value, ... }
        * [ param => value, ... ]
        * [ [ param => value ], ... ]
    value может быть скаляр или arrayref
    Опции:
        skip_empty - пропускаем пустые значения

=cut

sub get_param_string
{
    my ($params, %opt) = @_;

    return ( defined $params ? $params : q{} )  unless ref $params;

    # переводим плоский список в список пар
    if ( ref $params eq 'ARRAY' && $params->[0] && ref $params->[0] ne 'ARRAY' ) {
        my @new_params;
        my $iterator = natatime 2, @$params;
        while ( my ($p, $v) = $iterator->() ) {
            push @new_params, [ $p => $v ];
        }
        $params = \@new_params;
    }
    # ... и хеш тоже
    elsif ( ref $params eq 'HASH' ) {
        my @new_params = map {[ $_ => $params->{$_} ]} sort keys %$params;
        $params = \@new_params;
    }

    # собираем строчку
    my $str = join q{&},
        map { uri_escape($_->[0]) . q{=} . uri_escape($_->[1] // q[]) }
        # если просили, отбрасываем пустые параметры
        grep { $opt{skip_empty} ? defined($_->[1]) && length($_->[1]) : 1 }
        # параметр, заданный списком, разворачиваем в список одноимённых параметров
        map { my $p = $_; ref $p->[1] ? ( map {[ $p->[0] => $_ ]} @{$p->[1]} ) : $p }
        ( ref $params eq 'HASH'
            ? ( map {[ $_ => $params->{$_} ]} sort keys %$params )
            : @$params
        );

    return $str;
}


=head2 http_parallel_request

    Делает HTTP запросы в параллельном режиме используя AnyEvent::HTTP и возвращает HTTP ответы, когда они все придут.
    Количество параллельных запросов в один хост ограничивается, см. параметр max_req.

    Может ходить по редиректам, количество редиректов не настраивается и равно $AnyEvent::HTTP::MAX_RECURSE (на момент написания равно 10)

    Параметры позиционные: 
    $method -- 'GET' или 'POST'
    $data -- ссылка на хеш заданий: 
            { 
                id => {
                    url => '...', 
                    ...
                }, 
                ...
            }
        Каждое задание может содержать поля: 
            url 
            body - тело POST запроса, может быть скаляром, ссылкой на скаляр, ссылкоя на структуру для get_param_string
            proxy
            headers
            timeout num_attempts retry_on soft_timeout keepalive persistent

    Далее именованные параметры
    max_req  -- максимальное количество одновременных запросов 
               к одному удаленному хосту. По умолчанию равно $AnyEvent::HTTP::MAX_PER_HOST (на момент написания равно 4)
    timeout -- таймаут на выполнение запроса, в секундах. По умолчанию равно $AnyEvent::HTTP::TIMEOUT (на момент написания равно 300)
    connect_timeout -- таймаут на установку соединения. По умолчанию равно $DEFAULT_CONNECT_TIMEOUT, или значению параметра timeout
                если он явно указан и меньше чем $DEFAULT_CONNECT_TIMEOUT (4 секунды)
    headers -- заголовки для AnyEvent::HTTP::http_request
    log     -- колбек для логирования или объект, умеющий метод out (например, Yandex::Log)
    num_attempts -- максимальное количество попыток для каждого запроса (повтор в случае ошибок и таймаутов)
    max_total_retries -- максимальное количество ретраев на весь вызов
    retry_on     -- колбек для проверки статуса http-ответа. По умолчанию проверяем, что код != 2xx
    soft_timeout -- "мягкий" таймаут: запрос повторяется без обрыва предыдущего запроса
    response_size_limit -- максимальный размер скачиваемого контента (в байтах)
    callback -- опционально, ссылка на колбек для результатов
                в колбек передаётся два параметра: id и результат запроса (как описано ниже)
                При указании callback отключаются keepalive и persistent, т.к. иначе callback может не вызываться
                на каждый запрос
    prepare_callback -- функция для подготовки запроса. Получает ссылку на хеш с параметрами
        запроса, может модифицировать его, например добавить headers или изменить timeout
        Второй параметр - ссылка на хеш с заданием
    ipv6_prefer -- предпочитать ipv6-адреса при резолве доменов (по умолчанию - из v6/v4 будет выбран v4 адрес)
    handle_params — дополнительные параметры для конструктора AnyEvent::Handle
    proxy -- прокси
    content_ref -- вернуть ссылку на контент, а не его копию. в ответе вместо ключа content будет ключ content_ref

    Никакие параметры внутри функции не модифицируются 

    Возвращаемое значение: хеш с ключами из параметра $data

    {
        id => {
            content => ...,    # контент ответа [без content_ref]
            content_ref => \"...", # контент ответа [с content_ref]
            is_success => 0|1, # признак успешности запроса
            headers => {...},  # хеш заголовков и псевдозаголовков из AE::HTTP
            elapsed => 0.0021, # время затраченное на выполнение этого запроса, в секундах
                               # Осторожно! при использовании KeepAlive elapsed многих запросов может быть undef
            callback_error => '...' # $@ после исполнения callback'а
                                    # ключ отсутствует, если коллбек успешно выполнился
        },
    }

    Примеры использования:
    for my $b (@$banners){
        $reqs{$id} = {url => $b->{url},};
    }
    my $bs_resp = http_parallel_request(
        GET => \%reqs, 
        max_req => $Settings::BS_RANK_PARALLEL_LEVEL,
        timeout => $Settings::BS_TIMEOUT,
        headers => { 
            "Host" => $Settings::BSRANK_HOST ,
        },
        log => sub {print STDERR @_, "\n"},
    );

    my $advq_resp = http_parallel_request(
        POST => \%tasks, 
        max_req => $ADVQ_PARALLEL_LEVEL,
        timeout => $ADVQ_TIMEOUT,
    );

=cut
sub http_parallel_request
{
    my ($common_method, $data, %O) = @_;

    local $AnyEvent::HTTP::MAX_PER_HOST = $O{max_req} if $O{max_req};

    local $AnyEvent::PROTOCOL{ipv6} = 2 if $O{ipv6_prefer};
    local $AnyEvent::PROTOCOL{ipv4} = 1 if $O{ipv6_prefer};

    AE::now_update;

    die "unsupported method $common_method" if $common_method !~ /^(?:GET|POST|PUT)$/;

    return {} unless keys %$data;

    my $res = {};

    my $post_like_re = qr/^POST|PUT$/;

    foreach my $task (values %$data) {
        die "missed url" if !$task->{url};
        die "GET method doesn't accept 'body' parameter" if $common_method eq "GET" && defined $task->{body};
        die "POST method requires 'body' parameter" if $common_method =~ $post_like_re && ! defined $task->{body};
    }

    my @ids = keys %$data;
    my $total_retries = 0;
    while (my @chunk = splice @ids, 0, ($PARALLEL_REQUEST_CHUNK_SIZE || scalar @ids)) {
        my $cv = AnyEvent->condvar;
        for my $id (@chunk) {
            my $task = $data->{$id};
            my $method = $common_method; 
            my $url = $task->{url};

            my %params = (total_retries_ref => \$total_retries);
            state $pass_params = [qw/ timeout num_attempts max_total_retries retry_on soft_timeout keepalive persistent handle_params /];
            hash_copy \%params, \%O,   @$pass_params;
            hash_copy \%params, $task, @$pass_params;
            if ($O{prepare_callback}) {
                $O{prepare_callback}->(\%params, $task);
            }

            if ($O{callback}) {
                $params{keepalive} = 0;
                $params{persistent} = 0;
            }

            # NOTE: $headers может быть blessed, если сформирован через HTTP::Request::Common, поэтому hash_merge не использовать
            my $headers = {};
            foreach my $place ( $O{headers} || {}, $task->{headers} || {} ) {
                foreach my $key (keys %{ $place }) {
                    $headers->{$key} = $place->{$key};
                }
            }
            $params{headers} = $headers  if %$headers;

            if ( my $proxy = $task->{proxy} || $O{proxy} ) {
                if (ref $proxy eq 'ARRAY') {
                    $params{proxy} = $proxy;
                }
                elsif ( !ref $proxy && $proxy =~ m[ ^(?: http://)? ([\w\.\-]+) : (\d{1,5}) \b ]xms ) {
                    $params{proxy} = [ $1, $2 ];
                }
                else {
                    croak "Bad proxy: $proxy";
                }
            }

            # засекаем время начала выполнения запроса
            my $req_start;
            $params{on_prepare} = sub {
                $req_start ||= Time::HiRes::time();
                return exists $O{connect_timeout} ? $O{connect_timeout}
                    : $O{timeout} && $O{timeout} < $DEFAULT_CONNECT_TIMEOUT ? $O{timeout}
                    : $DEFAULT_CONNECT_TIMEOUT;
            };

            if ($O{response_size_limit}) {
                $params{on_header} = sub {
                    return ($_[0]{"content-length"} || 0) > $O{response_size_limit} ? 0 : 1;
                };
                $params{on_body} = sub {
                    if ($_[1]->{Status} !~ /3\d{2}/) {
                        $res->{$id}->{content} .= $_[0];
                        return (length $res->{$id}->{content} > $O{response_size_limit}) ? 0 : 1;
                    }
                    else {
                        return 1;
                    }
                }
            }

            my $body_ref;
            if ($method =~ $post_like_re) {
                $body_ref = ref $task->{body} eq 'SCALAR'  ? $task->{body}
                    : ref $task->{body}                    ? \Yandex::HTTP::make_url('', $task->{body}) 
                    : \$task->{body};

                if (Encode::is_utf8($$body_ref)) {
                    $body_ref = \Encode::encode_utf8($$body_ref);
                }
            }

            $cv->begin;
            _ae_http_request($method, Encode::encode_utf8(Yandex::IDN::idn_to_ascii($url)),
                %params,
                ($method =~ $post_like_re ? (body_ref => $body_ref) : ()),
                sub {
                    my ($can_retry, $content, $headers) = @_;
                    my $ok = eval {
                        # для keep-alive соединений мы не можем нормально засечь время обработки ни у одного запроса, кроме первого
                        my $elapsed = defined $req_start ? Time::HiRes::time() - $req_start : undef;
                        if ($O{log}) {
                            my $log_str = '';
                            {   no warnings 'uninitialized';
                                my $reason = $headers->{Reason};
                                utf8::decode($reason);
                                $log_str = "$method $url -> elapsed: $elapsed, res: $headers->{Status} $reason";
                            }
                            _log($O{log}, $log_str);
                        }

                        my $decoded_content_ref;
                        if ($O{response_size_limit}) {
                            if ($res->{$id}) {
                                $decoded_content_ref = _get_decoded_content_ref(\$res->{$id}->{content}, $headers);
                            }
                        } else {
                            $decoded_content_ref = _get_decoded_content_ref(\$content, $headers);
                        }
                        $res->{$id} = {
                            is_success => ($headers->{Status} =~ /^2/ ? 1 : 0),
                            headers => $headers, 
                            elapsed => $elapsed,
                        };
                        if ($O{content_ref}) {
                            $res->{$id}->{content_ref} = $decoded_content_ref;
                        } else {
                            $res->{$id}->{content} = (defined $decoded_content_ref) ? $$decoded_content_ref : undef;
                        }

                        if ($O{callback}) {
                            my $ret = eval {
                                $O{callback}->($id, $res->{$id});
                                1;
                            };
                            unless ($ret) {
                                $res->{$id}->{callback_error} = $@;
                                _log($O{log}, $@);
                                return 0;
                            }
                        }
                        return 1;
                    };
                    # будет ретрай - после него и случится end
                    $cv->end if $ok || !$can_retry;
                    return $ok;
                }
            );
        }

        # EV почему-то глючит с переопределённым __DIE__
        # недоосознанный workaround DIRECT-93398
        local $SIG{__DIE__};

        $cv->recv;
    }

    return $res;
}



=head3 _get_decoded_content_ref

    $decoded_content_ref = _get_decoded_content_ref($content_ref, $headers);

=cut

sub _get_decoded_content_ref {
    my ($content_ref, $headers) = @_;

    my $message = AnyEvent::HTTP::Response->new(undef, $headers)->to_http_message();
    # грязный хак против ещё одного копирования полученного контента
    $message->{_content} = $content_ref;
    $message->{_content_ref} = 1;
    $message->decoded_content(ref => 1, raise_error => 1, default_charset => $DEFAULT_CHARSET);
}


=head2 _ae_http_request

Обёртка вокруг AnyEvent::HTTP::http_request, с повторами и перезапросами.

Параметры совпадают с AnyEvent::HTTP::http_request, плюс:

    num_attempts -- максимальное количество попыток для каждого запроса (повтор в случае ошибок и таймаутов)
    soft_timeout -- "мягкий" таймаут: запрос повторяется без обрыва предыдущего запроса
        NB: При задании soft_timeout отключается KeepAlive
    retry_on     -- колбек для проверки статуса http-ответа. По умолчанию проверяем, что код != 2xx

=cut

sub _ae_http_request {
    my $cb = pop @_;
    my ($method, $url, %opt) = @_;

    my $body_ref = delete $opt{body_ref};

    my $num_attempts = delete $opt{num_attempts} || 1;
    my $soft_timeout = delete $opt{soft_timeout};
    my $reqs_in_fly = 0;

    my @requests;
    my $soft_timer;

    my $shedule_sub;
    my $destroy_sub;

    if ( $soft_timeout ) {
        # отключаем keep-alive: с ним колбек on_prepare не всегда вызывается
        $opt{persistent} = 0;
        $opt{keepalive} = 0;
        my $parent_on_prepare = $opt{on_prepare};
        $opt{on_prepare} = sub {
            if ( $num_attempts > 0 ) {
                $soft_timer = AnyEvent->timer(
                    after => $soft_timeout,
                    cb => sub { $shedule_sub->(); },
                );
            }
            return $parent_on_prepare->(@_)  if $parent_on_prepare;
            return;
        };
    }

    my $retry_checker = $opt{retry_on} || \&_is_request_failed;
    $shedule_sub = sub {
        return if $num_attempts < 1;
        $num_attempts --;
        $reqs_in_fly++;
        push @requests, AnyEvent::HTTP::http_request($method, $url,
            %opt,
            ($body_ref ? (body => $$body_ref) : ()),
            sub {
                my ($content, $headers) = @_;
                $reqs_in_fly--;
                my $can_retry = $num_attempts > 0 
                    && (!defined $opt{max_total_retries} || ${$opt{total_retries_ref}} < $opt{max_total_retries});
                my $need_retry = $retry_checker->($content, $headers);
                if ($need_retry) {
                    if ($reqs_in_fly > 0) {
                        return;
                    } elsif ($can_retry) {
                        ${$opt{total_retries_ref}}++;
                        $shedule_sub->();
                        return;
                    }
                }

                # для последнего ретрая вызываем колбек не смотря ни на что
                my $cb_ok = $cb->($can_retry, @_);
                if ($can_retry && !$cb_ok) {
                    ${$opt{total_retries_ref}}++;
                    $shedule_sub->();
                    return;
                }

                $destroy_sub->();
            }
        );
    };

    $destroy_sub = sub {
        $soft_timer = undef;
        $_ = undef  for @requests;
        $shedule_sub = undef;
        $destroy_sub = undef;
        %opt = ();
    };

    $shedule_sub->();
}


sub _is_request_failed {
    my ($content, $headers) = @_;
    return $headers->{Status} !~ /^2/;
}


=head3 _log($logger, $log_str)

    Функция-обертка для работы с разными логгерами

=cut
sub _log {
    my ($logger, $log_str) = @_;
    return unless $logger;

    if (ref $logger eq 'CODE') {
        $logger->($log_str);
    } else {
        $logger->out($log_str);
    }
}

=head2 http_fetch

    Обёртка над http_parallel_request.
    Возвращает декодированный контент в виде перловой строки по указанному URL.
    Если не смогло или получило какую-то ошибку — умирает. Если при этом передан заголовок  X-Ya-Service-Ticket, то
    в логе он обзерается до TVM_TICKET_MAX_LOG_LENGHT символов.
    ВАЖНОЕ ЗАМЕЧАНИЕ: заголовок обрезается in-place, поэтому для повтора запроса  $headers нужно генерировать ЗАНОВО

    $content = http_fetch($method => $url, $request, %options);

    Принимает параметры:
        $method — [обязательный] HTTP-метод: GET, POST
        $url — [обязательный] ссылка в виде строки
        $request — параметры запроса (query string для GET и body для POST/PUT). Может быть:
            * строкой с уже подготовленными параметрами;
            * ссылкой на хеш вида параметр => значение
            * ссылкой на массив вида параметр, значение
        %options с ключами:
            log — ссылка на sub'у или объект с методом out (Yandex::Log, например) для записи полезных данных о запросе
            timeout — таймаут запроса в секундах [по умолчанию $HTTP_FETCH_DEFAULT_TIMEOUT];
            connect_timeout — таймаут на соединение в секундах [по умолчанию $DEFAULT_CONNECT_TIMEOUT];
            num_attempts — максимальное количество попыток для каждого запроса (повтор в случае ошибок и таймаутов)
            soft_timeout — "мягкий" таймаут: запрос повторяется без обрыва предыдущего запроса
            response_size_limit — максимальный размер скачиваемого контента (в байтах)
            headers — ссылка на хеш с заголовками для AnyEvent::HTTP::http_request
            proxy — [$host, $port[, $scheme]] or undef
            ipv6_prefer - флаг для предпочтения v6 адресов при резолве доменов (умолчание - предпочитать v4)
            handle_params — дополнительные параметры для конструктора AnyEvent::Handle
            response_headers_ref - опциональная ссылка на скаляр, куда будет сохранена ссылка на заголовки ответа
            response_status_stat_ref - опциональная ссылка на скаляр, куда будет сохранена ссылка на статистику по кодам ответов

    $log = sub { my($str_for_log) = @_; print STDERR $str_for_log; };
    $log = Yandex::Log->new(...);

    $content = http_fetch('GET', $url);
    $content = http_fetch('GET', $url, undef, timeout => 10);
    $content = http_fetch('GET', $url, undef, timeout => 10, log => $log);
    $content = http_fetch('GET', $url, undef, timeout => 10, log => $log);
    $content = http_fetch('GET', $url, undef, timeout => 10, log => $log, soft_timeout => 5, num_attempts => 2);
    $content = http_fetch('GET', $url, undef, timeout => 10, log => $log, proxy => $Settings::SSRF_PROXY, headers => $Settings::SSRF_PROXY_HEADERS);

    $content = http_fetch('GET', $url, {param1 => $val1, ...});
    $content = http_fetch('GET', $url, {param1 => $val1, ...}, timeout => 10);
    $content = http_fetch('GET', $url, {param1 => $val1, ...}, timeout => 10, log => $log);
    $content = http_fetch('GET', $url, {param1 => $val1, ...}, timeout => 10, log => $log, soft_timeout => 5, num_attempts => 2);
    $content = http_fetch('GET', $url, {param1 => $val1, ...}, timeout => 10, log => $log, proxy => $Settings::SSRF_PROXY, headers => $Settings::SSRF_PROXY_HEADERS);

    $content = http_fetch('GET', $url, [param1 => $val1, ...]);
    $content = http_fetch('GET', $url, [param1 => $val1, ...], timeout => 10);
    $content = http_fetch('GET', $url, [param1 => $val1, ...], timeout => 10, log => $log);
    $content = http_fetch('GET', $url, [param1 => $val1, ...], timeout => 10, log => $log, soft_timeout => 5, num_attempts => 2);
    $content = http_fetch('GET', $url, [param1 => $val1, ...], timeout => 10, log => $log, proxy => $Settings::SSRF_PROXY, headers => $Settings::SSRF_PROXY_HEADERS);

    $content = http_fetch('POST', $url);
    $content = http_fetch('POST', $url, undef, timeout => 10);
    $content = http_fetch('POST', $url, undef, timeout => 10, log => $log);
    $content = http_fetch('POST', $url, undef, timeout => 10, log => $log, soft_timeout => 5, num_attempts => 2);
    $content = http_fetch('POST', $url, undef, timeout => 10, log => $log, proxy => $Settings::SSRF_PROXY, headers => $Settings::SSRF_PROXY_HEADERS);

    $content = http_fetch('POST', $url, $content);
    $content = http_fetch('POST', $url, $content, timeout => 10);
    $content = http_fetch('POST', $url, $content, timeout => 10, log => $log);
    $content = http_fetch('POST', $url, $content, timeout => 10, log => $log, soft_timeout => 5, num_attempts => 2);
    $content = http_fetch('POST', $url, $content, timeout => 10, log => $log, proxy => $Settings::SSRF_PROXY, headers => $Settings::SSRF_PROXY_HEADERS);

    $content = http_fetch('POST', $url, {param1 => $val1, ...});
    $content = http_fetch('POST', $url, {param1 => $val1, ...}, timeout => 10);
    $content = http_fetch('POST', $url, {param1 => $val1, ...}, timeout => 10, log => $log);
    $content = http_fetch('POST', $url, {param1 => $val1, ...}, timeout => 10, log => $log, soft_timeout => 5, num_attempts => 2);
    $content = http_fetch('POST', $url, {param1 => $val1, ...}, timeout => 10, log => $log, proxy => $Settings::SSRF_PROXY, headers => $Settings::SSRF_PROXY_HEADERS);

    $content = http_fetch('POST', $url, [param1 => $val1, ...]);
    $content = http_fetch('POST', $url, [param1 => $val1, ...], timeout => 10);
    $content = http_fetch('POST', $url, [param1 => $val1, ...], timeout => 10, log => $log);
    $content = http_fetch('POST', $url, [param1 => $val1, ...], timeout => 10, log => $log, soft_timeout => 5, num_attempts => 2);
    $content = http_fetch('POST', $url, [param1 => $val1, ...], timeout => 10, log => $log, proxy => $Settings::SSRF_PROXY, headers => $Settings::SSRF_PROXY_HEADERS);

    $content = http_fetch('GET', $url, \%params, handle_params => {keepalive => 1});

    $content = http_fetch('GET', $url, \%params, response_headers_ref => \$headers);

=cut

use constant HTTP_FETCH_REQUEST_ID => 123;
use constant HTTP_FETCH_MAX_FAILED_RESPONSE_CONTENT_LENGTH => 10_000;
# максимальное количество логируемых символов TVM тикета из заголовка X-Ya-Service-Ticket
use constant TVM_TICKET_MAX_LOG_LENGHT => 50;

sub http_fetch {
    my ($method, $url, $request, %options) = @_;

    my $log = $options{log};

    confess 'method not given' unless $method;
    confess 'url not given' unless $url;
    confess 'url must be string' if ref($url) ne '';

    my %task;
    if ($method eq 'GET') {
        $task{url} = make_url($url, $request);
    } else {
        $task{url} = $url;
        $task{body} = $request;
    }

    my %req = (HTTP_FETCH_REQUEST_ID() => \%task);
    my @common_param_names = qw(log timeout connect_timeout num_attempts soft_timeout response_size_limit headers proxy ipv6_prefer handle_params);
    my $common_params = hash_cut(\%options, @common_param_names);
    $common_params->{timeout} //= $HTTP_FETCH_DEFAULT_TIMEOUT;
    $common_params->{content_ref} = 1;
    my %stat;
    my $retry_on = sub {
        my ($content, $headers) = @_;
        $stat{ $headers->{Status} }++;
        return _is_request_failed(@_);
    };

    my $responses = http_parallel_request($method => \%req, %$common_params, retry_on => $retry_on);
    my $response = $responses->{HTTP_FETCH_REQUEST_ID()};
    my $content_ref = $response->{content_ref};

    if ($response && defined $options{response_headers_ref}) {
        croak 'invalid response_headers_ref given' unless ref($options{response_headers_ref}) eq 'SCALAR';
        ${$options{response_headers_ref}} = $response->{headers};
    }
    if ($response && defined $options{response_status_stat_ref}) {
        croak 'invalid response_status_stat_ref given' unless ref($options{response_status_stat_ref}) eq 'SCALAR';
        ${$options{response_status_stat_ref}} = \%stat;
    }

    if (!$response || !$response->{is_success}) {
        my $reason = $response->{headers}->{Reason} // 'reason unknown';
        my $status = $response->{headers}->{Status} // 'status unknown';
        my $elapsed = $response->{elapsed} // 'unknown';
        utf8::decode($reason);
        utf8::decode($status);
        my $content_cut = substr($$content_ref // 'undef', 0, HTTP_FETCH_MAX_FAILED_RESPONSE_CONTENT_LENGTH);
        # нельзя логировать TVM тикеты - это небезопасно
        # инвариант: проверить весь вызывающий код, что он не логирует заголовки запроса и здесь поменять на копию $options
        if (exists $options{headers} && $options{headers}->{'X-Ya-Service-Ticket'}) {
            # оставляем только первые 50 символов тикета - этого достаточно для определения источника и приемника,
            # но недостаточно чтобы успешно пройти валидацию
            $options{headers}->{'X-Ya-Service-Ticket'} = join('', substr($options{headers}->{'X-Ya-Service-Ticket'}, 0, TVM_TICKET_MAX_LOG_LENGHT), '---');
        }
        my $options_json = to_json(\%options, {allow_blessed => 1, allow_unknown => 1});
        my $msg = "http_fetch($method => $url, $options_json) failed: $status $reason after $elapsed seconds. Content got was:\n$content_cut";
        my $msg_stacktrace = Carp::longmess($msg);
        _log($log, $msg_stacktrace);
        confess $msg;
    }

    return $$content_ref;
}

1;
