#!/usr/bin/perl -w

use strict;
use utf8;

use Test::More;
use Test::Warnings;
use List::MoreUtils qw/ any /;


use Yandex::HTTP;

my @tests = (
    {
        data => undef, 
        result => '',
        name => 'undef',
    },
    {
        data => 'aa=bb',
        result => 'aa=bb',
        name => 'string',
    },
    {
        data => { aa => 'bb', cc => 'dd' },
        result => [ 'aa=bb&cc=dd', 'cc=dd&aa=bb' ],
        name => 'hash'
    },
    {
        data => [ aa => 'bb', cc => 'dd' ],
        result => 'aa=bb&cc=dd',
        name => 'plain list'
    },
    {
        data => [ [aa => 'bb'], [cc => 'dd'] ],
        result => 'aa=bb&cc=dd',
        name => 'paired list'
    },
    {
        data => [ aa => 'bb', cc => ['dd', 'ee'] ],
        result => 'aa=bb&cc=dd&cc=ee',
        name => 'param as list'
    },
    {
        data => [ aa => 'bb', zz => '', cc => [0, undef, 'dd'] ],
        result => 'aa=bb&cc=0&cc=dd',
        extra => [ skip_empty => 1 ],
        name => 'skip empty'
    },
    {
        data => [ aa => "\0", cc => '&=' ],
        result => 'aa=%00&cc=%26%3D',
        name => 'base quoting'
    },
    {
        data => [ aa => "\N{U+400}", cc => 'ЭЮЯ' ],
        result => 'aa=%D0%80&cc=%D0%AD%D0%AE%D0%AF',
        name => 'utf8 quoting'
    },
    {
        data => [ aa => "\xFF\xFE" ],
        result => 'aa=%FF%FE',
        name => 'byte quoting'
    },
    {
        data => { aa => undef },
        result => 'aa=',
        name => 'undef param value',
    },
);


for my $t ( @tests ) {
    my $res = Yandex::HTTP::get_param_string($t->{data}, @{ $t->{extra} || [] });
    if ( ref $t->{result} ) {
        # точный результат неопределён, варианты
        ok( (any {$res eq $_} @{$t->{result}}), $t->{name});
    } else {
        is($res, $t->{result}, $t->{name});
    }
}


done_testing();
