#! /usr/bin/perl

=head1 http_fetch.t

    Тест проверяет работоспособность опции connect_timeout

=cut

use strict;
use warnings;

use Time::HiRes qw/sleep/;

use Test::More;
use Test::Exception;
use Test::HTTP::Server;

use Yandex::HTTP qw/http_fetch/;
use Yandex::Log;

sub Test::HTTP::Server::Request::AUTOLOAD {
    my ($self, $params) = @_;
    my ($method, $uri) = @{$self->{request}};
    my $body = $self->{body} // '';

    if ($uri =~ /^\/code\/(\d+)/) {
        $self->{out_code} = "$1 Reason";
    } elsif ($uri =~ /^\/sleep\/([\d\.]+)/) {
        sleep $1;
    } elsif ($uri =~ m!^/headers!) {
        $self->{out_headers}->{'Test-Header'} = 'got it';
    }

    return "$method $uri $body";
}

# Запускаем тестовый http-сервер
my $server = Test::HTTP::Server->new();

is(http_fetch(GET => $server->uri."test"), "GET /test ");
is(http_fetch(GET => $server->uri."test", "xxx"), "GET /test?xxx ");
is(http_fetch(GET => $server->uri."test1", {x => "qwe"}), "GET /test1?x=qwe ");
is(http_fetch(GET => $server->uri."test2", [x => "%"]), "GET /test2?x=%25 ");

throws_ok { http_fetch(GET => $server->uri.'code/500') } qr/failed: 500/, '500 error';

# TODO: написать тестовую версию Yandex::Log с интерфейсом аля Log::Any::Test
my $log = Yandex::Log->new(no_log => 1);
is($log->{_records_count} // 0, 0, 'no messages in log before');
throws_ok { http_fetch(GET => $server->uri.'code/500', undef, log => $log) } qr/failed: 500/, '500 error with log object given';
is($log->{_records_count} // 0, 2, 'got messages in log after 500 error with log object given');

my $headers;
is(http_fetch(GET => $server->uri.'headers', undef, response_headers_ref => \$headers), 'GET /headers ');
ok($headers && ref($headers) eq 'HASH', 'headers is hash');
# "All the headers in that hash are lowercased" (c) https://metacpan.org/pod/AnyEvent::HTTP
is($headers->{'test-header'}, 'got it', 'Test-Header content');

# скрываем TVM тикеты в логах
my $ticket = '3:serv:CPg6EOb83OMFIggIg496EJ6Qeg:GXRtFsNKRepTMifxK6pFx8UbEyBKiKRMRzYyRtO64GzEyx4yEYOytwZU3jGSaGn2MwpEWdTDNxxAcP1gqcK_hjslI22yBG8ROdQlJdU9eBMg9ttNamOk';
throws_ok {
    http_fetch(GET => $server->uri.'code/501', undef, headers => {'X-Ya-Service-Ticket' => $ticket})
    }  qr/"X-Ya-Service-Ticket":"3:serv:CPg6EOb83OMFIggIg496EJ6Qeg:GXRtFsNKRepTMifx---"/, 'hide X-Ya-Service-Ticket when dies';

$headers = undef;
throws_ok { 
    http_fetch(GET => $server->uri.'code/501', undef, response_headers_ref => \$headers) 
    }  qr/failed: 501/,         'dies when response code is 501';
ok(ref($headers) eq 'HASH',     'response with error has headers, and they are a hashref');
is($headers->{'Status'}, '501', '  and returned header has correct error code');

is(http_fetch(POST => $server->uri."test", "xxx"), "POST /test xxx");

is(http_fetch(POST => $server->uri."test", \("xxxx")), "POST /test xxxx");

throws_ok {
    http_fetch(POST => $server->uri."code/404", "xxx")
} qr/failed: 404/;

lives_ok {
    http_fetch(POST => $server->uri."sleep/0.1", "xxx", timeout => 0.2);
};
throws_ok {
    http_fetch(POST => $server->uri."sleep/0.1", "xxx", timeout => 0.05);
} qr/failed: 596 Connection timed out/;

done_testing();
