#! /usr/bin/perl

use strict;
use warnings;

use Test::More;

use Yandex::HTTP qw/http_parallel_request/;
use Test::HTTP::Server;

use File::Temp;
use Fcntl ':flock';

use JSON;
use Time::HiRes;
use List::MoreUtils qw/ any /;

our $DEBUG = 0;

our $filename = tmpnam();

my $task_cnt = 30;

my $num_parallel = 5;
my $timeout = 3;
my $resend_timeout = 1,
my $attempts = 2;


our %DELAY = (
    1 => [-2, 1],
    2 => [-2, -1],
    6 => [4, 0],
    7 => [4, 4],
    8 => [-2, -2],

    28 => [2, 4],
    29 => [2, 0],
    30 => [2, 1],
);


# Запускаем тестовый http-сервер
sub Test::HTTP::Server::Request::test_request
{
    my ($self, @args) = @_;
    my $req_id = $args[0];
    my $attempt = _get_update_attempt_num($req_id);

    my $response = "response for $req_id, attempt $attempt [$$]\x0D\x0A";
    my $delays = $DELAY{$req_id};
    if ( my $delay = $delays && $delays->[$attempt] ) {
        if ( $delay < 0 ) {
            print STDERR "[$$] emulating error for $req_id (attempt $attempt)\n"  if $DEBUG;
            return;
        }
        else {
            print STDERR "[$$] delaying $req_id (attempt $attempt) for $delay secs\n"  if $DEBUG;
            sleep $delay;
        }
    }

    print STDERR "[$$] responding for $req_id (attempt $attempt)\n"  if $DEBUG;
    return $response;
}

my $server = Test::HTTP::Server->new();

# Собираем задания для загрузки
my $tasks = {};
for my $k (1 .. $task_cnt){
    $tasks->{$k} = { url => $server->uri() . "test_request/$k" };
}

my $result = http_parallel_request("GET", $tasks,
    max_req => $num_parallel,
    timeout => $timeout,
    num_attempts => $attempts,
    soft_timeout => $resend_timeout,
);


for my $req_id (1 .. $task_cnt) {
    my $req = "request $req_id";
    ok( exists $result->{$req_id}, "$req: result exists" );

    my $d = $DELAY{$req_id} || [];
    my @succeed_atts = grep { my $t=$d->[$_]; !$t || ( $t>0 && $t < $timeout ) } ( 0 .. $attempts -1 );

    my $should_fail = !@succeed_atts;

    my $res = $result->{$req_id};
    if ( $should_fail ) {
        is( $res->{is_success}, 0, "$req: failed" );
    }
    else {
        is( $res->{is_success}, 1, "$req: succeed" );
        my ($was_id, $att) = $res->{content} =~ /response for (\d+), attempt (\d+)/;
        is( $was_id, $req_id, "$req: right response" );
        ok( ( any {$att==$_} @succeed_atts ), "$req: possible attempt" );
    }
}




done_testing();


sub _get_update_attempt_num {
    my ($id) = @_;

    open my $file, '+>>', $filename;
    flock $file, LOCK_EX;
    sysseek $file, 0, 0;
    sysread $file, my $buf, 1000;

    my $values = decode_json( $buf || '[]' );
    my $result = $values->[$id] || 0;
    $values->[$id] = $result + 1;
    $buf = encode_json( $values );

    sysseek $file, 0, 0;
    truncate $file, 0;
    syswrite $file, $buf;

    close $file;

    return $result;
}



