#######################################################################
#
#  Direct.Yandex.ru
#
#  Internationalization support
#
#  authors:
#    Maxim Kuzmin <kuzmin@yandex-team.ru>
#    Sergey Mudrik <msa@yandex-team.ru>
#    Sergey Zhuravlev <zhur@yandex-team.ru>
#
#  $Id$
#
#  (c) 2004-2006 Yandex
#
#######################################################################

=head1 NAME

I18n

=head1 DESCRIPTION

Internationalization support

=cut

package Yandex::I18n;
use base qw/Exporter/;

use warnings;
use strict;
use Carp;

use Locale::Messages qw/:locale_h :libintl_h/;
use POSIX qw//;

use Yandex::Trace;
use Yandex::ListUtils qw/xsort/;
use utf8;

our $VERSION = '0.29';
our @EXPORT = qw/iget iget_noop piget_array piget is_valid_lang get_sql_collate_by_lang/;

our $cur_lang;  # 'ru'  either 'ua' or 'en'

our $DEFAULT_LANG ||= 'ru';

our %LOCALES;
unless (%LOCALES) {
    %LOCALES = (
        ru => 'ru_RU', 
        en => 'en_US', 
        ua => 'ua_UA',
        uk => 'uk_UA',
        tr => {locale => 'tr_TR', base => 'en'}
    );
}

our %LANG_ALIAS;
#if (!%LANG_ALIAS) {
#    %LANG_ALIAS = (
#        ua => 'uk',
#    );
#}

our $LANG_SQL_COLLATE_MAP ||= {tr => 'utf8_turkish_ci'};
our $DEFAULT_SQL_COLLATE  ||= 'utf8_unicode_ci';

our $LOCALE_PATH;
our $PROJECT_ID ||= 'direct';

=head2 init_i18n

    Инициализация с выбором языка

    Опции:
        check_file - проверка наличия файла с переводами

=cut

sub init_i18n {
    my ($lang, %opt) = @_;
    my $profile = Yandex::Trace::new_profile('i18n:init_i18n');
    die "The variable LOCALE_PATH is undefined" if (!defined($LOCALE_PATH));
    $cur_lang = resolve_lang_alias($lang);
    if (!is_valid_lang($cur_lang)) {
        $cur_lang = $DEFAULT_LANG;
    } 

    my $base_locale = get_locale($cur_lang);
    my $locale = "$base_locale.utf8";

    if ($opt{check_file} && $cur_lang ne $DEFAULT_LANG) {
        my $mo_file = "$LOCALE_PATH/$base_locale/LC_MESSAGES/$PROJECT_ID.mo";
        croak "No translations for $base_locale found, check $mo_file"  if !-f $mo_file;
    }

    POSIX::setlocale(POSIX::LC_ALL(), 'en_US.utf8'); # LC_NUMERIC и пр. (кроме LC_MESSAGES) выставляем en_US.utf8
    if (POSIX::setlocale(POSIX::LC_MESSAGES(), $locale)) {
        POSIX::setlocale(POSIX::LC_COLLATE(), $locale); # для use locale
        Locale::Messages->select_package('gettext_xs');
        Locale::Messages::nl_putenv("LANGUAGE=");
    } else {
        # не все наши локали существуют в рельных системах (например, ua_UA)
        # для PP-реализации gettext есть возможность не смотреть на системные локали
        # и принудительно указать в файлах для какой локали надо искать перевод
        # а вот XS-версия всегда оперирует системной локалью

        # предупрждение мешало, временно убрано. когда перейдём на нормальную украинскую локаль (uk_UK), нужно будет вернуть
        #warn "Couldn't set system locale $locale. Falling back to PP-implementation of gettext with hardcoded language.";
        Locale::Messages->select_package('gettext_pp');
        Locale::Messages::nl_putenv("LANGUAGE=$locale");
    }

    textdomain($PROJECT_ID);
    bindtextdomain($PROJECT_ID => $LOCALE_PATH);
}

=head2 init_i18n_guard

    Локально переопределяет язык на время существования возвращаемого объекта.

    $guard = init_i18n_guard($lang)

=cut

sub init_i18n_guard {
    my $lang = resolve_lang_alias(shift);

    return Yandex::I18n::I18nGuard->new($lang);
}

=head2 iget

    get translate

    use: [% iget("text message") %]
         [% iget("text message %s", 123) %]
    or
         error(iget("Ошибка базы данных. Менеджер %s не найден.", 'mlogin'));

    One-liner для отладки на бете:
    perl -Mmy_inc=for,protected -MSettings -MYandex::I18n -Mutf8 -le 'Yandex::I18n::init_i18n("en"); print iget("Открыть в новом окне");'

=cut

sub iget
{
    my ($text, @params) = @_;
    utf8::encode($text);
    my $msg = Locale::Messages::turn_utf_8_on(Locale::Messages::gettext($text));
    return @params ? sprintf($msg, @params) : $msg;
}


=head2 iget_noop
    
    Маркер для xiget.pl, ничего не делает

=cut

sub iget_noop
{
    if (scalar @_ == 1) {
        return $_[0];
    } else {
        return @_;
    }
}

=head2 piget_array

    Интерфейс, через который удобно оборачивать части переведённых фраз в некоторую разметку.
    Возвращает массив с кусками фраз. В остальном аналогичен piget'у.
    Позволяет работать с объектами в качестве значений параметров. 
    В качестве значения параметра можно передавать ссылку на функцию. На вход ей поступит 
    замещающий текст и название параметра, а результат будет подставлен в выходной массив.

    $arr = piget_array('Пожалуйста, %{on#активируйте} таймер самоуничтожения', { on => $some_obj });
    $arr = ['Пожалуйста, ', $some_obj, ' таймер самоуничтожения'];
    $arr = piget_array('Пожалуйста, %{on#активируйте} таймер самоуничтожения', { on => sub {my ($text, $name) = @_; return "$name: $text"} });
    $arr = ['Пожалуйста, ', 'on: активируйте', ' таймер самоуничтожения'];

=cut

sub piget_array {
    my ($text, $params) = @_;

    $text = iget($text);
    $params ||= {};

    my @arr;
    for my $el (split /(%\{\s*.+?\s*(?:#.*?)?})/, $text) {
        if ($el =~ /%\{\s*(.+?)\s*(?:#(.*?))?}/) {
            my ($param_name, $param_text) = ($1, $2);
            my $value = $params->{$param_name};
            if (ref($value) eq 'CODE') {
                push @arr, $value->($param_text, $param_name);
            } else {
                push @arr, $value;
            }
        } else {
            push @arr, $el;
        }
    }

    return \@arr;
}

=head2 piget

    Интерфейс, через который удобно оборачивать части переведённых фраз в некоторую разметку.
    Возвращает строку.

    $text = piget('Пожалуйста, %{on#активируйте} таймер самоуничтожения', { on => '<a href="/doom_timer">$on$</a>' });
    $text => 'Пожалуйста, <a href="/doom_timer">активируйте</a> таймер самоуничтожения';

=cut

sub piget {
    my ($text, $params) = @_;

    my %new_params;
    if ($params) {
        while (my($name, $value) = each %$params) {
            if (ref($value) eq 'CODE') {
                # переданные функции никак не изменяем
                $new_params{$name} = $value;
            } else {
                $new_params{$name} = sub {
                    my ($text, $name) = @_;
                    my $param_value = $value;
                    $param_value =~ s/\$$name\$/$text/g;
                    return $param_value;
                };
            }
        }
    }
    my $arr = piget_array($text, \%new_params);
    return join '', @$arr;
}

=head2 my $lang = Yandex::I18n::current_lang()

    Получить код текущего языка

=cut

sub current_lang 
{
    return $cur_lang || $DEFAULT_LANG;
}


sub default_lang {
    return $DEFAULT_LANG;
}


=head2 is_valid_lang

    if (is_valid_lang('ru')) {...}

=cut

sub is_valid_lang
{
    my $lang = shift;
    return exists $LOCALES{$lang} || exists $LANG_ALIAS{$lang};
}


=head2 resolve_lang_alias

    Возвращает базовый язык для алиаса, или сам язык

=cut

sub resolve_lang_alias {
    my $lang = shift;

    return $LANG_ALIAS{$lang} || $lang;
}


=head2 get_other_langs

    Возвращаем массив языков без $DEFAULT_LANG

=cut

sub get_other_langs {
    
    return xsort {get_base_locale($_) || ''}
            grep {$_ ne $DEFAULT_LANG} keys %LOCALES;
}


=head2 get_base_locale($locale)

    Получить базовую локаль от $locale.
    Используется в случае отсутствия оригинального перевода.

=cut

sub get_base_locale {    
    my $lang = resolve_lang_alias(shift);

    ref $LOCALES{$lang} eq 'HASH'
        ? $LOCALES{$lang}->{base}
        : undef
}


=head2 get_locale($lang)

    Получить локаль по языку

=cut

sub get_locale {
    my $lang = resolve_lang_alias(shift);

    ref $LOCALES{$lang} eq 'HASH'
        ? $LOCALES{$lang}->{locale}
        : $LOCALES{$lang}
}


=head2 get_sql_collate_by_lang($lang)

    Возвращаем значение collate по переданному языку

=cut

sub get_sql_collate_by_lang
{
    my $lang = resolve_lang_alias(shift);

    return $LANG_SQL_COLLATE_MAP->{$lang} || $DEFAULT_SQL_COLLATE;
}



=head1 Yandex::I18n::I18nGuard

    Пакет для использования из init_i18n_guard. Нужен для локальных переопределений языковых настроек
    и возвращения их в исходные значения.

=cut

package Yandex::I18n::I18nGuard;

=head2 new

    Устанавливает на время существования объекта указанный язык.

    $lang_guard = Yandex::I18n::I18nGuard($lang);

=cut

sub new {
    my (undef, $lang) = @_;

    my $current_lang = Yandex::I18n::current_lang();
    Yandex::I18n::init_i18n($lang);

    bless {revert_to_lang => $current_lang};
}

sub DESTROY {
    my ($self) = @_;

    if ($self->{revert_to_lang}) {
        Yandex::I18n::init_i18n($self->{revert_to_lang});
    }
}

1;
