package Yandex::I18nTools;

# $Id$

=head1 NAME

Yandex::I18nTools

=head1 DESCRIPTION

Tools for internationalization support

=cut

use warnings;
use strict;

use Encode;
use File::Slurp;
use File::Temp qw/tempfile/;
use HTTP::Request::Common;
use JSON;
use List::Util qw/first/;
use LWP::UserAgent;
use URI::Escape qw/uri_escape_utf8/;

use Yandex::HTTP;
use Yandex::Shell qw/yash_qx/;
use Yandex::Tanker;
use Yandex::I18n;
use Yandex::MailTemplate;

use utf8;

our $PROJECT_ID_VERSION ||= "";
our $AUTHORIZATION ||= "";
our $HINTS_SEPARATOR = ', ';

# $SURROGATE_EMAILS_KEYS -- если 1, то вместо русских текстов для писем отправлять "синтетитеческие" ключи: <имя письма>-<номер абзаца>
our $SURROGATE_EMAILS_KEYS ||= 0;
# название кейсета с устаревшими фразами
our $OUTDATED_KEYSET = 'Outdated';
our $TRIES_TO_UPLOAD = 2;


=head2 get_translated_stats
    
    Возвращаем данные о полноте перевода по срезам
    (
        total => N,
        hints => [],
        en => {
            translated => M,
            hints => { 
                hint1 => {total => N1, translated => M1}, 
                ... 
            }
        },
        ua => {
            ...
        },
        ...
    )

=cut

sub get_translated_stats
{
    my %stats = map {$_ => _get_stats_by_lang($_)} Yandex::I18n::get_other_langs();
    my @messages = _extract_messages(get_pot_filename());
    my %hints = ();
    foreach my $message (@messages) {
        my $hint = _get_best_hint($message, \%hints);
        $hints{$hint}->{number} += 1;
    }
    $stats{total} = scalar @messages;
    $stats{hints} = \%hints;
    return %stats;
}


=head2 _get_stats_by_lang

    Возвращаем срезы для одного файла
    {
        translated => M,
        hints => { 
            hint1 => {total => N1, translated => M1}, 
            ... 
        }
    }

=cut

sub _get_stats_by_lang
{
    my $lang = shift;
    my @messages = _extract_messages(get_po_filename($lang));

    my %stats = (
        translated => 0,
        hints => {},
    );
    foreach my $message (@messages) {
        # перевод может занимать несколько строк
        my ($translation) = $message =~ /^msgstr(?:\[\d+\])? "(.+)"\s*\z/ms;
        if ($translation) {
            $stats{translated} += 1;
        }
        my $hint = _get_best_hint($message, $stats{hints});
        $stats{hints}->{$hint} ||= {total => 0, translated => 0, missing => []};
        $stats{hints}->{$hint}->{total} += 1;
        if ($translation) {
            $stats{hints}->{$hint}->{translated} += 1;
        } else {
            push @{$stats{hints}->{$hint}->{missing}}, $message;
        }
    }
    return \%stats;
}


=head2 _get_best_hint($message, $existing_hints)

    Чтобы не пересекать сообщения по хинтам в интерфейсе, определяем лучший хинт из возможных
    (в файле, все равно, останутся все)

=cut

sub _get_best_hint
{
    my ( $message, $existing_hints ) = @_;
    my $hint = '';

    if ( my ( $hints_string ) = ( $message =~ m/^#\. (\w+: .*)$/m ) ) {
        my %hints = ();
        my ($label, $hints) = split ': ', $hints_string;
        return 'core' if $label eq 'common';
        foreach my $hint (split $HINTS_SEPARATOR, $hints) {
            $hint =~ s/::[^:]+$//;
            $hint =~ s#^(block/[^/]+).*#$1#;
            $hints{$label .': '. $hint} = 1;
        }
        my @hints = sort keys %hints;
        $hint = (first { $existing_hints->{$_} } @hints) || $hints[0];
    } else {
        ( $hint ) = ( $message =~ m/^#\. (\w+)$/m );
    }
    $hint //= ''; # WHY откуда берутся undef'ы?

    $hint =~ s/\s+/_/g;
    return $hint;
}


=head2 _extract_messages
    
    Вход: имя файла,
    возвращаем сообщения

=cut

sub _extract_messages
{
    my $filename = shift;
    my @messages = split /\n\n/, read_file($filename, binmode => ':utf8');
    shift @messages;
    return @messages;
}


=head2 get_po_filename

    По названию языка отдаем имя .po-файла

=cut

sub get_po_filename
{
    my $lang = shift;
    return $Yandex::I18n::LOCALE_PATH .  Yandex::I18n::get_locale($lang) . '.po';
}


=head2 get_pot_filename

    Отдаем имя .pot-файла

=cut

sub get_pot_filename
{
    return $Yandex::I18n::LOCALE_PATH . $Yandex::I18n::PROJECT_ID . '.pot';
}


=head2 merge_po
    
    Вливаем в файл для указанного языка некоторое содержимое
    Входные параметры: $lang, $content

=cut

sub merge_po
{
    my $lang = shift;
    my $content = shift;
    
    my ($tmp_fh, $tmp_filename) = tempfile();
    write_file($tmp_filename, {atomic => 1, binmode => ':utf8'}, \$content);
    my $po_filename = get_po_filename($lang);
    my $merged_content = yash_qx('msgcat', '--use-first', '--sort-output', $tmp_filename, $po_filename);
    unlink $tmp_filename;
    if (!utf8::is_utf8($merged_content)) {
        utf8::decode($merged_content);
    }
    # Избегаем сообщений-бэкапов
    $merged_content =~ s/^#\~.*$//mg;
    $merged_content =~ s/\n\n+/\n\n/g;
    write_file($po_filename, {atomic => 1, binmode => ':utf8'}, \$merged_content);

    # чтобы подчистить лишние фразы, которые могли попасть в $content
    update_po($lang);
}


=head2 get_filtered_pot($hint)

    Получаем содержимое .pot-файла, зафильтрованное по хинту $hint
    Отказались от использования msggrep в реализации, чтобы отдавать непересекающиеся срезы

=cut

sub get_filtered_pot
{
    my $hint = shift || '';

    my @messages = _extract_messages(get_pot_filename());
    my $content = get_po_header() . join("\n\n", grep { $hint eq _get_best_hint($_) } @messages);
    $content =~ s/^#\./#:/mg; # !!! танкер учитывает только reference, а msgmerge их всегда склеивает, в отличие от комментариев типа #.
    return $content;
}


=head2 get_filtered_po($lang, $hint)

    Получаем содержимое .po-файла для языка $lang, зафильтрованное по хинту $hint

=cut

sub get_filtered_po
{
    my $hint = shift || '';
    my $lang = shift || return '';

    my @messages = _extract_messages(get_po_filename($lang));
    my $content = get_po_header() . join("\n\n", grep { $hint eq _get_best_hint($_) } @messages);
    $content =~ s/^#\./#:/mg;
    return $content;
}


=head2 send_to_l10n_team($branch, $hint, $content, $no_language, $keysets, $format)

    Отправляем словарь на перевод
    $content может отсутствовать, в этом случае отправляем отфильтрованную по $hint часть pot-файла

    если передано истинное значение позиционного параметра $no_language, то при запросе в Танкер язык не указывается,
    в противном случае передается язык по умолчанию

=cut

sub send_to_l10n_team
{
    my $branch = shift;
    my $hint = shift || 'default hint';
    my $content = shift || get_filtered_pot($hint);
    my $no_language = shift;
    my $keysets = shift || {map {$_ => 1} get_keysets($branch)};
    my $format = shift // 'po';

    if (!utf8::is_utf8($content)) {
        utf8::decode($content);
    }

    my ($tmp_fh, $tmp_filename) = tempfile();
    write_file($tmp_filename, {atomic => 1, binmode => ':utf8'}, $content);

    my $ua = LWP::UserAgent->new(timeout => $Yandex::Tanker::HTTP_TIMEOUT);
    my $url = $Yandex::Tanker::L10N_API_URL.($keysets->{$hint} ? '/keysets/replace/' : '/keysets/create/');
    $hint = Encode::encode_utf8($hint);
    my %request = (
        _authorization_header(),
        Content_Type => 'form-data',
        Content => [
            'project-id' => $Yandex::I18n::PROJECT_ID,
            'branch-id'  => $branch,
            'keyset-id'  => $hint,
            ($no_language ? () : (language => Yandex::I18n::default_lang())),
            file     => [$tmp_filename],
            format       => $format
        ]
    );

    my $response;
    for my $iter (1..$TRIES_TO_UPLOAD) {
        $response = $ua->request(
            HTTP::Request::Common::POST $url,
            %request
        );
        if ($response->is_success) {
            last;
        }
    }

    unlink($tmp_filename);

    if (!$response->is_success) {
        warn $response->code;
        warn $response->decoded_content;
        return;
    }

    # если это Outdated, то нам нужно дополнительно сохранить переводы
    if ($hint eq $OUTDATED_KEYSET) {
        if ($format ne 'po') {
            warn "Can't send a part of a $format format file";
            return;
        }
        send_keyset_translations($branch, $hint, [Yandex::I18n::get_other_langs(Yandex::I18n::default_lang())]);
    }

    return 1;
}


=head2 send_keyset_translations($branch, $hint, $languages)

    Отправляем переводы для определенного кейсета
    Переводы берем из отфильтрованной по $hint части po-файла

=cut

sub send_keyset_translations
{
    my $branch = shift;
    my $hint = shift;
    my $languages = shift;
    
    for my $lang (@$languages) {
        my $content = get_filtered_po($hint, $lang);

        if (!utf8::is_utf8($content)) {
            utf8::decode($content);
        }

        my ($tmp_fh, $tmp_filename) = tempfile();
        write_file($tmp_filename, {atomic => 1, binmode => ':utf8'}, $content);

        my $keysets = {map {$_ => 1} get_keysets($branch)};
        my $ua = LWP::UserAgent->new(timeout => $Yandex::Tanker::HTTP_TIMEOUT);
        my $url = $Yandex::Tanker::L10N_API_URL.($keysets->{$hint} ? '/keysets/replace/' : '/keysets/create/');
        $hint = Encode::encode_utf8($hint);
        my %request = (
            _authorization_header(),
            Content_Type => 'form-data',
            Content => [
                'project-id' => $Yandex::I18n::PROJECT_ID,
                'branch-id'  => $branch, 
                'keyset-id'  => $hint,
                'language' => ($lang eq 'ua' ? 'uk' : $lang),
                file     => [$tmp_filename],
                format       => 'po'
            ]
        );

        my $response;
        for my $iter (1..$TRIES_TO_UPLOAD) {
            $response = $ua->request(
                HTTP::Request::Common::POST $url,
                %request
            );
            if ($response->is_success) {
                last;
            }
        }

        unlink($tmp_filename);

        if (!$response->is_success) {
            warn $response->code;
            warn $response->decoded_content;
            return;
        }
    }
    return 1;    
}


=head2 send_everything_to_l10n_team

    Отправляем все фразы

=cut

sub send_everything_to_l10n_team
{
    my $branch = shift;

    my %stats = get_translated_stats();

    my $hints = $stats{hints};
    # для ускорения загрузим список кейсетов заранее и передадим как параметр, чтобы не загружать его 500 раз заново в send_to_l10n_team
    my $keysets = {map {$_ => 1} get_keysets($branch)};
    foreach my $hint (sort keys %$hints) {
        if ($hint) {
            send_to_l10n_team($branch,  $hint, undef, undef, $keysets) or warn "failed to send $hint";
        }
    }
}


=head2 get_all_phrases_from_tanker
    
    Загружаем в direct.pot все фразы из танкера

=cut

sub get_all_phrases_from_tanker
{
    my $branch = shift;

    my $lang = Yandex::I18n::default_lang();
    my $url = "$Yandex::Tanker::L10N_API_URL/projects/export/xml/?project-id=$Yandex::I18n::PROJECT_ID&branch-id=$branch&language=$lang";
    my $xml = http_get($url, timeout => $Yandex::Tanker::HTTP_TIMEOUT) or die "couldn't get tanker export from $url";
    my $translated_messages_by_lang = Yandex::Tanker::xml_to_po($xml, (skip_keyset => 'Glossary'));
    my $content = get_po_header() . join("\n", map {qq[#. $OUTDATED_KEYSET\nmsgid "$_->{msgid}"\nmsgstr ""\n]} @{ $translated_messages_by_lang->{$lang} });
    
    my ($tmp_fh, $tmp_filename) = tempfile();
    write_file($tmp_filename, {atomic => 1, binmode => ':utf8'}, \$content);
    my $pot_filename = get_pot_filename();
    my $merged_content = yash_qx('msgcat', '--sort-output', $tmp_filename);
    unlink $tmp_filename;
    if (!utf8::is_utf8($merged_content)) {
        utf8::decode($merged_content);
    }
    # Избегаем сообщений-бэкапов
    $merged_content =~ s/^#\~.*$//mg;
    $merged_content =~ s/\n\n+/\n\n/g;
    write_file($pot_filename, {atomic => 1, binmode => ':utf8'}, \$merged_content);
}


=head2 merge_everything_from_l10n_team

    Забираем все переводы, формируем .po-файлы и сливаем с нашими словарями

    тестирование:
    perl -Iprotected -Iyandex-lib/tanker/lib -Iyandex-lib/i18n/lib/ -MYandex::I18nTools -ME -e 'Yandex::I18nTools::merge_everything_from_l10n_team()'

=cut

sub merge_everything_from_l10n_team
{
    my $branch = shift;

    my @langs = Yandex::I18n::get_other_langs();
    # FIXME ua_UA -> uk_UA
    my $url = "$Yandex::Tanker::L10N_API_URL/projects/export/xml/?project-id=$Yandex::I18n::PROJECT_ID&branch-id=$branch&language=" . join(',', map {$_ eq 'ua' ? 'uk' : $_} @langs);
    my $xml = http_get($url, timeout => $Yandex::Tanker::HTTP_TIMEOUT) or die "couldn't get tanker export from $url";
    my $translated_messages_by_lang = Yandex::Tanker::xml_to_po($xml);

    # FIXME ua_UA -> uk_UA
    if (grep {$_ eq 'uk'} @langs) {
        $translated_messages_by_lang->{uk} = delete $translated_messages_by_lang->{ua};
    }

    foreach my $lang (@langs) {
        my $content = get_po_header() . join("\n", map {qq[msgid "$_->{msgid}"\nmsgstr "$_->{msgstr}"\n]} @{ $translated_messages_by_lang->{$lang} });
        merge_po($lang, $content);
    }
}


=head2 update_po($lang)

    Обновляем .po-файлы новыми фразами, собранными в direct.pot
    А также, убираем из них все лишнее (вызов в merge_po)

    Если не указан язык, то делаем для всех

=cut

sub update_po
{
    my $lang = shift;

    if ($lang) {
        _update_po_for_lang($lang);
    } else {
        foreach my $lang (Yandex::I18n::get_other_langs()) {
            _update_po_for_lang($lang);
        }
    }
}


=head2 _update_po_for_lang($lang)

    Обновление .po-файла для указанного языка

=cut

sub _update_po_for_lang
{
    my $lang = shift;

    my $po_filename  = get_po_filename($lang);
    my $pot_filename = get_pot_filename();
    if (!-e $po_filename) {
        write_file($po_filename, {atomic => 1, bindmode => ':utf8'}, \get_po_header());
    }
    my $content = yash_qx('msgmerge', '--silent', '--no-fuzzy-matching', '--sort-output', $po_filename, $pot_filename);
    if (!utf8::is_utf8($content)) {
        utf8::decode($content);
    }
    $content =~ s/^#\~.*$//mg;
    $content =~ s/\n\n+/\n\n/g;
    write_file($po_filename, {atomic => 1, binmode => ':utf8'}, \$content);
}


=head2 get_keysets

    Получаем список keyset'ов, зарегистрированных у группы локализации

=cut

sub get_keysets
{
    my $branch = shift;

    my $keysets_json = http_get("$Yandex::Tanker::L10N_API_URL/admin/project/$Yandex::I18n::PROJECT_ID/keysets/?branch=$branch", timeout => $Yandex::Tanker::HTTP_TIMEOUT) or die "couldn't get tanker keysets";
    my $keysets_struct = decode_json($keysets_json);
    my @keysets = map { $_->{name} } @{ $keysets_struct->{data}->{items} };
    return @keysets;
}


=head2 get_branches

    Получаем список названий ветвей в Танкере

=cut

sub get_branches
{
    my $response = http_get("$Yandex::Tanker::L10N_API_URL/admin/project/$Yandex::I18n::PROJECT_ID/branches/") or die "couldn't get branches list";
    return [map {$_->{name}} @{ decode_json($response)->{data}->{items} }];
}


=head2 create_branch($branch)

    Создание ветви в Танкере

=cut

sub create_branch
{
    my $branch = shift;

    my $ua = LWP::UserAgent->new(timeout => $Yandex::Tanker::HTTP_TIMEOUT);
    my %request = (
        _authorization_header(),
        Content_Type => 'application/json',
        Content => encode_json({ref => 'master', name => $branch}),
    );
    my $url = "$Yandex::Tanker::L10N_API_URL/admin/project/$Yandex::I18n::PROJECT_ID/branch/";
    my $response = $ua->request(
        HTTP::Request::Common::POST $url,
        %request
    );
    if ($response->is_success) {
        return 1;
    } else {
        warn $url;
        warn $response->code;
        warn $response->decoded_content;
        return;
    }
}


=head2 delete_keyset($keyset)

    Удаляем кейсет из танкера

=cut

sub delete_keyset
{
    my $branch = shift;
    my $keyset = shift;

    # слеш после keyset в урле обязателен!
    # используем QUERY_STRING, т.к. в урл нельзя вставить кейсет со слешем и нельзя вставить заэскейпленное название
    my $url = "$Yandex::Tanker::L10N_API_URL/admin/project/$Yandex::I18n::PROJECT_ID/keyset/?branch-id=$branch&keyset=".uri_escape_utf8($keyset);
    my $ua = LWP::UserAgent->new(timeout => $Yandex::Tanker::HTTP_TIMEOUT);
    my $response = $ua->request(
        HTTP::Request::Common::DELETE $url,
        _authorization_header(),
    );
    if ($response->is_success) {
        return 1;
    } else {
        warn $url;
        warn $response->code;
        warn $response->decoded_content;
        return;
    }
}


sub get_po_header {
    my $PO_HEADER = <<HEADER
msgid \"\"
msgstr \"\"
\"Project-Id-Version: $PROJECT_ID_VERSION\\n\"
\"MIME-Version: 1.0\\n\"
\"Content-Type: text/plain; charset=utf-8\\n\"
\"Content-Transfer-Encoding: 8bit\\n\"

HEADER
;
    return $PO_HEADER;

}

# -----------------------------------------------------------------------------

=head2 send_emails_to_l10n_team

    Собираем xml-файл и отправляем в проект с письмами

    тестирование:
    perl -Iprotected -Iyandex-lib/i18n/lib/ -MYandex::I18nTools -ME -e 'p Yandex::I18nTools::send_emails_to_l10n_team()'

=cut

sub send_emails_to_l10n_team {
    my $branch = shift;

    my @emails;
    my @templates = Yandex::MailTemplate::get_email_template_list_names();

    my $keysets = {};
    my $default_lang = Yandex::I18n::default_lang();
    for my $template_name (@templates) {
        next unless defined $template_name && $template_name ne ""; # WHY откуда берутся пустые имена шаблонов?
        my $content = Yandex::MailTemplate::get_raw_email_template($template_name, $default_lang);
        my $par_num = 1;
        for my $paragraph (@{Yandex::MailTemplate::get_email_template_paragraphs($content)}) {
            next if $paragraph->{type} eq 'whitespace';
            my $content = $paragraph->{content};
            my $key;
            if ( $SURROGATE_EMAILS_KEYS ){
                # ключ делаем "синтетический": название шаблона и номер абзаца
                # русский текст в качестве ключа плохо обрабатывается в Танкере (слишком длинные ключи)
                $key = $template_name . "-" . sprintf("%03d", $par_num++);
            } else {
                $key = escape_key($content);
            }
            $keysets->{$template_name}->{$key} = {$default_lang => $content};
        }
    }
    my $keysets_xml = Yandex::Tanker::hashref_to_xml($keysets);

    # API Танкера требует при отправке указывать существующий кейсет (произвольный)
    send_to_l10n_team($branch, $templates[0], $keysets_xml, undef, undef, 'xml');
}

# -----------------------------------------------------------------------------

sub get_emails_filename {
    return $Yandex::I18n::LOCALE_PATH.'/emails.json';
}

# -----------------------------------------------------------------------------

=head2 get_emails_translation

    Забираем из Танкера переводы писем и сохраняем на диск

=cut

sub get_emails_translation {
    my $branch = shift;

    my @langs = Yandex::I18n::get_other_langs();
    # Если ключи синтетические, то дополнительно запрашиваем русский язык, 
    # чтобы для записи в файл заменить искусственные идентификаторы обратно на русские тексты
    my $languages = join(',', 
        ($SURROGATE_EMAILS_KEYS ? 'ru' : ()), 
        sort map {$_ eq 'ua' ? 'uk' : $_} @langs
    ); # FIXME ua_UA -> uk_UA
    my $url = "$Yandex::Tanker::L10N_API_URL/projects/export/tjson/?project-id=$Yandex::I18n::PROJECT_ID&branch-id=$branch&language=$languages";
    my $json = http_get($url, timeout => $Yandex::Tanker::HTTP_TIMEOUT) or die "Couldn't fetch emails export from Tanker: $url";

    if ( $SURROGATE_EMAILS_KEYS ) {
        my $emails = decode_json($json);
        for my $keyset_content ( values %{$emails->{keysets}} ){
            for my $surrogate_key ( sort keys %{$keyset_content->{keys}} ){
                next unless exists $keyset_content->{keys}->{$surrogate_key};
                my $natural_key = escape_key($keyset_content->{keys}->{$surrogate_key}->{translations}->{ru}->{form});
                $keyset_content->{keys}->{$natural_key} = delete $keyset_content->{keys}->{$surrogate_key};
            }
        }
        $json = Encode::encode_utf8(to_json( $emails, { pretty => 1, canonical => 1 } ));
    }

    write_file(get_emails_filename(), {atomic => 1, binmode => ':raw'}, \$json);
}

# -----------------------------------------------------------------------------

=head2 update_emails

    Собираем шаблоны на других языках
    Если перевода нет, либо он неполон, записываем в файл пустую строку:
    при отправке предполагается fallback на другой язык

    тестирование:
    perl -Iprotected -Iyandex-lib/i18n/lib/ -MYandex::I18nTools -ME -e 'Yandex::I18nTools::update_emails()'
    less data/t/emails/en/...

=cut

sub update_emails {
    my $translations = from_json(scalar read_file(get_emails_filename(), binmode => ':utf8'));
    my @templates = Yandex::MailTemplate::get_email_template_list_names();
    my @langs = Yandex::I18n::get_other_langs();
    for my $template (@templates) {
        for my $lang (@langs) {
            my $template_content = assemble_email_template_translation($template, $lang eq 'ua' ? 'uk' : $lang, $translations); # FIXME ua_UA -> uk_UA
            Yandex::MailTemplate::save_raw_email_template($template, $lang, $template_content);
        }
    }
}

# -----------------------------------------------------------------------------

=head2 assemble_email_template_translation($name, $lang, $translations)

    Возвращаем полный перевод шаблона, либо пустую строку, если не все абзацы переведены.

    $translations -- хеш, получаемый в формате tjson из Танкера:
    http://doc.yandex-team.ru/Tanker/api-reference/concepts/formats.xml#format-tjson

=cut

sub assemble_email_template_translation {
    my $name = shift;
    my $lang = shift;
    my $translations = shift;

    my $content = Yandex::MailTemplate::get_raw_email_template($name, Yandex::I18n::default_lang());
    my $paragraphs = Yandex::MailTemplate::get_email_template_paragraphs($content);    
    my @result = ();
    for my $paragraph (@$paragraphs) {
        if ($paragraph->{type} eq 'text') {
            my $key = escape_key($paragraph->{content});
            return '' unless exists $translations->{keysets}->{$name}->{keys}->{$key}; # параграф отсутствует в экспорте Танкера, перевод неполон
            my $translation = $translations->{keysets}->{$name}->{keys}->{$key}->{translations}->{$lang};
            if (($translation->{status} // '') eq 'approved' && $translation->{form}) {
                push @result, $translation->{form};
            } else {
                return ''; # перевод неполон
            }
        } else {
            push @result, $paragraph->{content};
        }
    }
    return join "\n", @result;
}


=head2 escape_key

    В ключах запрещены спецсимволы https://jira.yandex-team.ru/browse/TANKER-84461,
    поэтому экранируем переводы строк

=cut

sub escape_key
{
    my $key = shift;
    $key =~ s/\n/\\n/g;
    return $key;
}

=head2 _authorization_header

Функция возвращает заголовок (в виде пары ключ-значение) для авторизации в танкере
Для старой авторизации: AUTHORIZATION: $AUTHORIZATION
Для новой авторизации: Authorization: $AUTHORIZATION

Решение принимается по содержимому $AUTHORIZATION, оно должно начинаться с 'OAuth ' для новой авторизации

=cut

sub _authorization_header
{
    if ($AUTHORIZATION =~ /^OAuth /) {
        return (Authorization => $AUTHORIZATION);
    }
    else {
        return (AUTHORIZATION => $AUTHORIZATION);
    }
}

1;
