=head1 NAME

MailTemplate - работа с шаблонами писем

=head1 DESCRIPTION

работа с шаблонами писем

=head1 FUNCTIONS

=cut

package Yandex::MailTemplate;

use warnings;
use strict;

use File::Slurp;

use Yandex::I18n;
use Yandex::SendMail qw/send_alert/;
use utf8;

require Exporter;

our $VERSION = '0.01';
our @ISA = qw(Exporter);
our @EXPORT = qw(
    get_email_template
    save_email_template
    get_raw_email_template
    save_raw_email_template
    get_email_template_list
    get_email_template_list_names
);

# путь к шаблонам писем, при использовании модуля требуется переопределить (например в Settings.pm)
# внутри должны находится поддиректории для всех языков (/ru/... /en/...)
our $EMAIL_TEMPLATES_FOLDER;

=head2 $EMAIL_TEMPLATE_ADDITIONAL_PROPERTIES

    Массив возможных дополнительных(кроме description и subject) атрибутов шаблонов писем
    ключи:
        name - название атрибута
        desc - описание

=cut

our $EMAIL_TEMPLATE_ADDITIONAL_PROPERTIES;

# -----------------------------------------------------------------------------

=head2 get_raw_email_template($name, $lang)

    возвращаем необработанный шаблон письма

=cut

sub get_raw_email_template {
    my ($name, $lang) = @_;

    if (! $lang || ! -e "$EMAIL_TEMPLATES_FOLDER/$lang/$name") {
        $lang = "ru";
    }

    my $filename = "$EMAIL_TEMPLATES_FOLDER/$lang/$name";
    my $text = (-e $filename) ? read_file($filename, binmode => ':utf8') : '';
}

# -----------------------------------------------------------------------------

=head2 save_raw_email_template($name, $lang, $content)

    Сохраняем необработанный шаблон письма

=cut

sub save_raw_email_template($$$) {
    my ($name, $lang, $content) = @_;

    my $filename = "$EMAIL_TEMPLATES_FOLDER/$lang/$name";
    return write_file($filename, {binmode => ':utf8', atomic => 1}, $content) ? 1 : 0;
}

# -----------------------------------------------------------------------------

=head2 get_email_template ($name, $lang)

    Извлечение шаблона письма $name из соответствующего файла для языка $lang

    $name - Название наблона письма
    $lang - язык
    $die_on_error - упасть (вместо вызова send_alert), если в шаблоне нет поля subject 

    шаблоны i_* - содержат только текст, предназначены только для включения в другие шаблоны

=cut

sub get_email_template {
    my ($name, $lang, $die_on_error) = @_;

    my $text = get_raw_email_template($name, $lang) or return undef;
    my ($headers, $content);
    my %result; 

    if ($name =~ /^i_.+/) {
        $content = $text;
    } else {
        ($headers, $content) = split("\n\n", $text, 2);
        %result = map {/^([\w-]+):\s*(.*)/ ? (lc($1) => $2) : die "Template $name has wrong format"} split /\n/, $headers;

        if (! defined $result{"subject"}) {
            if ($die_on_error) {
                die "Template $name has no subject field";
            } else {
                send_alert("mail template $name ($lang) not found", 'error: template not found');
                return undef;
            }
        }
    }

    $result{"content"} = $content;
    $result{"lang"} = $lang;
    $result{"name"} = $name;

    return \%result;
}

# -----------------------------------------------------------------------------

=head2 save_email_template($name, $lang, $data)

    сохраняем шаблон письма

=cut

sub save_email_template
{
    my ($name, $lang, $data) = @_;

    if (! $lang || ! -e "$EMAIL_TEMPLATES_FOLDER/$lang/$name") {
        $lang = "ru";
    }

    my $folder = "$EMAIL_TEMPLATES_FOLDER/$lang/";

    unless (-d $folder) {
        die "Email's folder not exists: $folder";
    }

    my $filename = $folder.$name;

    my $text;

    if ($name =~ /^i_.+/) {
        $text = $data->{content};
    } else {
        $text = join ("\n"
                      , "Description: $data->{description}"
                      , "Subject: $data->{subject}"
                      , (map {"$_: $data->{$_}"} grep {$data->{$_}} map {$_->{name}} @$Yandex::MailTemplate::EMAIL_TEMPLATE_ADDITIONAL_PROPERTIES)
                      , ""
                      , $data->{content});
    }

    $text =~ s/\r//gs;

    return write_file($filename, {binmode => ':utf8'}, $text) or die "Can't save template: $!";
}

# -----------------------------------------------------------------------------

=head2 get_email_template_list ($lang)

    Возвращает список всех шаблонов писем для языка $lang

=cut

sub get_email_template_list {
    my $lang = shift;
    my $data;
    my $folder = "$EMAIL_TEMPLATES_FOLDER/$lang";
    return [] if (! -e $folder);

    opendir(my $dh, $folder) || die "can't open dir $folder: $!";
    my @files = grep { -f $folder."/".$_ && ! m/^\./ } readdir($dh);
    for my $filename (@files) {
        my $result = get_email_template($filename, $lang);
        push @{$data}, $result;
    }

    closedir $dh;
    return $data;
}

# -----------------------------------------------------------------------------

=head2 get_email_template_list_names

    Возвращает список всех имен шаблонов писем, для русского языка

=cut

sub get_email_template_list_names {
    return map {$_->{name}}
           @{ get_email_template_list(Yandex::I18n::default_lang()) };
}

# -----------------------------------------------------------------------------

=head2 get_email_template_paragraphs($content)

    Разбиваем шаблон на параграфы, при этом одна tt-инструкция не попадает в разные абзацы,
    т.е. пустые строки внутри [% %] игнорируются

    Возвращаем [{type, content}], type: whitespace | text

=cut

sub get_email_template_paragraphs {
    my $content = shift;

    my @lines = split /\r?\n/, $content, -1; # Обязательно указывать -1, чтобы завершающие пустые строки не исчезли

    # Парсим последовательно по строкам, запоминаем $unclosed -- число незакрытых скобок "[%"
    # Завершаем параграф, если встречаем пустую (либо пробельную) строку и $unclosed == 0
    my $unclosed = 0;
    my $type = 'text'; # text | whitespace
    my @buffer = ();
    my @paragraphs = ();

    for my $line (@lines) {
        if ($line =~ /^\s*$/) {
            if ($type eq 'text' && $unclosed == 0) {
                _flush_buffer(\$type, \@paragraphs, \@buffer);
                $type = 'whitespace';
            }
        } else {
            $unclosed += _count_substr($line, '[%') - _count_substr($line, '%]');
            if ($type eq 'whitespace') {
                _flush_buffer(\$type, \@paragraphs, \@buffer);
                $type = 'text';
            }
        }
        push @buffer, $line;
    }
    _flush_buffer(\$type, \@paragraphs, \@buffer);

    return \@paragraphs;
}

sub _count_substr {
    my $str = shift;
    my $sub_str = quotemeta(shift);

    my $count =()= $str =~ /$sub_str/g;
    return $count;
}

sub _flush_buffer {
    my ($type_ref, $paragraphs_ref, $buffer_ref) = @_;

    return unless @$buffer_ref;
    push @$paragraphs_ref, {type => $$type_ref, content => join("\n", @$buffer_ref)};
    @$buffer_ref = ();
}

1;
