package Yandex::IDN;

# $Id$
# работа с национальными доменами

use Direct::Modern;
use Net::LibIDN ();
use Encode;

use base qw/Exporter/;
our @EXPORT_OK = qw (
    is_valid_email
    is_valid_domain
    idn_to_ascii
    idn_to_unicode
);

sub idn_to_ascii {
    my $uri = shift;
    my $aref = split_uri($uri);
    return '' unless($aref);
    
    $aref->[1] = Net::LibIDN::idn_to_ascii(Encode::encode_utf8($aref->[1]),"utf8");
    return '' unless($aref->[1]);

    return join('', @{$aref});
}

sub idn_to_unicode {
    my $uri = shift;
    my $aref = split_uri($uri);
    return '' unless($aref);
    
    $aref->[1] = Encode::decode_utf8(Net::LibIDN::idn_to_unicode(lc($aref->[1]), "utf8"));
    return '' unless($aref->[1]);

    return join('', @{$aref});
}

sub split_uri {
    my $uri = shift;
    return '' if (! defined $uri || $uri !~ m!^([a-zA-Z\s]+://)?((?:[[:alnum:]][[:alnum:]\-\_]*\.)+[[:alnum:]\-]+\.?)(.*)\z!s);
    my $proto  = $1 || '';
    my $domain = $2 || '';
    my $query  = $3 || '';
    return [$proto,$domain,$query];
}

# а теперь кусочек про валидацию доменов и емейлов

=head2 is_valid_email(email)

    проверяет валидность емейла и возвращает 1 либо 0

    PS. мы не работаем с адресами с кавычками в логине - вроде "John @ Doe"

=cut

sub is_valid_email {
    my ($email, %O) = @_;
    utf8::decode($email) unless (utf8::is_utf8($email));
    if ($O{'check_only_at_symbol'}) {
        return $email =~ m/@/ ? 1 : 0;
    }

    $email =~ s/^\s+|\s+$//g;

    return 0 if $email =~ /\s/;
    # надеюсь, никто никогда не пришлет валидный емейл с двумя собаками
    my ($login, $domain) = split('@', $email, 2);
    return 0 unless ($login && $domain);

    # логины (на данный момент) не могут содержать кириллицы в принципе
    return 0 unless ($login =~ /^\s*[a-z0-9_\-\+]+(?:\.[a-z0-9_\-\+]+)*$/i);

    return 0 unless is_valid_domain($domain);

    return 1;
}

=head2 is_valid_domain(domain)

    проверяет валидность домена и возвращает 1 либо 0

=cut

sub is_valid_domain {
    my $domain = shift;
    utf8::decode($domain) unless (utf8::is_utf8($domain));

    $domain =~ s/^\s+|\s+$//g;
    my $en_letters = "a-z";
    my $ru_letters = "а-яё";
    my $ua_letters = "\x{404}\x{406}\x{407}\x{410}-\x{429}\x{42C}\x{42E}-\x{449}"
        ."\x{44C}\x{44E}\x{44F}\x{454}\x{456}\x{457}\x{490}\x{491}";
    my $kz_letters = "\x{401}\x{406}\x{410}-\x{44F}\x{451}\x{456}\x{492}\x{493}"
        ."\x{49A}\x{49B}\x{4A2}\x{4A3}\x{4AE}\x{4AF}\x{4B0}\x{4B1}\x{4BA}\x{4BB}"
        ."\x{4D8}\x{4D9}\x{4E8}\x{4E9}";
    my $tr_letters = "a-z\x{11F}\x{11E}\x{FC}\x{DC}\x{15F}\x{15E}\x{F6}\x{D6}\x{E7}\x{C7}\x{131}\x{130}\x{307}"; # this is not exactly right, some en letters are not exists in tr

    my $letters_tld = join "|", map { "[${_}][${_}0-9-]{1,14}" } ($en_letters, $ru_letters, $ua_letters, $kz_letters, $tr_letters);
    my $second_level_letters = join "|", map { "[${_}0-9][${_}0-9-]{0,62}" } ($en_letters, $ru_letters, $ua_letters, $kz_letters, $tr_letters);
    my $high_level_letters = sprintf("[%s0-9][%s0-9-_]{0,62}", (join ("", ($en_letters, $ru_letters, $ua_letters, $kz_letters, $tr_letters))) x 2);

    # максимум длины для поддомена - 63 символа, TLD - от 2 до 15 символов
    return $domain
           && ( $domain =~ /^ (?:(?:$high_level_letters)\.)* (?:$second_level_letters)\. (xn--[a-z0-9-]{1,59}|$letters_tld) $/xi )
           && Net::LibIDN::idn_to_ascii(Encode::encode_utf8($domain), "utf8")
           ? 1 : 0;
}

1;
