#   
# $Id$
#
#   Функции для сглаживания данных
#   * Ступенчатая интерполяция
#   * Линейная интерполяция
#

package Yandex::Interpolate;

use strict;
use warnings;

use base 'Exporter';
use Yandex::HashUtils;
use List::Util qw/min max/;
use List::MoreUtils qw/all/;

our @EXPORT = qw(
    interpolate_const
    interpolate_linear
    interpolate_linear_fast
    resample
);

use utf8;



=head2 interpolate_const(x, @mass)
	@mass = ({x=>1,y=>1}, {x=>x2,y=>y2}, {x=>x3,y=>y3});

	|           ____
	|___-----___
	|_________________=> x
	
=cut

sub interpolate_const
{
    my $x = shift;
    my @aa = @_;
    
    die "No segments" if !@aa;
    
    my @bb = sort {$a->{x} <=> $b->{x}} @aa;
    my $prev = $bb[0];

    foreach my $b (@bb) {
	if ($x > $b->{x}) {
            $prev = $b;
            next;
        }
        
	return $prev->{y};
    }
	
    return $prev->{y};
}

=head2	interploate_linear(x, @mass)
 		@mass = ({x=>1,y=>1}, {x=>x2,y=>y2}, {x=>x3,y=>y3});
 		
	|    /\
	| /\/  \/\__
	|/__________\___=> x

    interploate_linear и interpolate_linear_fast: параметры одинаковые, результат одинаковый. 
    Но interploate_linear сортирует массив точек перед интерполяцией, 
    а interpolate_linear_fast -- не сортирует (предполагается, что передается уже отсортированный массив), и поэтому работает быстрее
 		
=cut

sub interpolate_linear
{
    my $x = shift;
    my @aa = @_;
    my @bb = sort {$a->{x} <=> $b->{x}} @aa;

    return interpolate_linear_fast($x, \@bb);
}


=head2 interpolate_linear_fast

    См. interploate_linear

=cut 
sub interpolate_linear_fast
{
    my $x = shift;
    my $bb = shift;
    
    die "No segments" if !@$bb;
    
    my $prev = $bb->[0];
    
    foreach my $b (@$bb) {
        if ($x > $b->{x}) {
            $prev = $b;
            next;
        }
        
	    return ($prev->{x} != $b->{x}) ?
	    ($prev->{y} - $b->{y}) * ($x - $b->{x}) / ($prev->{x} - $b->{x}) + $b->{y} : 
            $prev->{y};
    }
	
    return $prev->{y};
}

=head2 resample($data, $options)

    # массив с данными
    data: [
        {
            cost    => 123,
            cnt     => 12,
        },
        {
            cost    => 223,
            cnt     => 22,
        },
        {
            cost    => 323,
            cnt     => 32,
        },
        ... 
    ]
    
    дополнительные параметры
    options:
=cut

sub resample($;$) {
    my ($data, $opt) = @_;

    my $options = hash_merge {
        relative_gap    => 0.05,    # допустимая относительная ошибка при выкалывании точек.
        absolute_gap    => undef,    # допустимая абсолютная ошибка при выкалывании точек.
        max_len         => 7,       # максимальная длина вырезаемых кусков данных
        max_bad_len_count   => 3,   # максимальное количество неудачных попыток вырезать кусок данных
        x_key           => 'x',     # ключ первой переменной в массиве данных
        y_key           => 'y',     # ключ второй переменной в массиве данных
    }, $opt;

    my ($x, $y) = ($options->{x_key}, $options->{y_key});

    return $data if 2 >= scalar @$data; # Вырожденный случай

    my @aa = (@$data);
    @aa = sort {$a->{$x} <=> $b->{$x}} @aa;
    
    my $height  = max( map {$_->{$y}} @aa ) - min( map {$_->{$y}} @aa ); 
    my $max_error = max 1e-12, (defined $options->{absolute_gap} ? $options->{absolute_gap} : $height*$options->{relative_gap});

    my $calc_error = sub {
        my $dy_dx = ( $_[2]->{$x} - $_[0]->{$x} != 0 ) ? (($_[2]->{$y} - $_[0]->{$y})/($_[2]->{$x} - $_[0]->{$x})) : 0;
        return $_[0]->{$y} + $dy_dx*($_[1]->{$x} - $_[0]->{$x})- $_[1]->{$y};
    };

    my @bb=();          # result
    my $nice_len = 1;   # nex step length
    push @bb, $aa[0];  # first node is always within the result

    for ( my $i=0; $i < $#aa; $i+=$nice_len) {
        $nice_len = 1;
        my $bad_len_count=0;    # если интерполяция для нескольких значений len оказалась неудачной - останавливаемся на первом удачном значении
        for(my $len = 2; $len <= min($#aa-$i, $options->{max_len}) and  $bad_len_count < $options->{max_bad_len_count};  $len ++ ) {
            if ( all {  abs ($calc_error->($aa[$i], $aa[$_], $aa[$i+$len])) < $max_error } $i+1 .. $i+$len-1 ) {
                # Хорошая интерполяция по метрике R1
                $nice_len =$len;
                next;
            } else {
                # плохая интерполяция
                $bad_len_count ++;
            }
        }
        push @bb, $aa[$i+$nice_len];    # запоминаем максимально удёлённый элемент с хорошим приближением пропущенных точек
    }
    return [@bb];
}

1;

