package Yandex::JugglerQueue;

=head1 NAME

    Yandex::Juggler

=head1 SYNOPSIS

    use Yandex::Juggler;

    queue_juggler_event(status => 'OK', service => 'bsClientData.shard_1.std_1');
    queue_juggler_event(host        => 'virtual.hostname',
                        service     => 'CollectAgencyOfflineReportStat',
                        status      => 'INFO',
                        description => 'Last processed date is 2014-06-14',
    );

=head1 DESCRIPTION

    Perl-обертка для консольной утилиты juggler_queue_event.
    Предназначена для отложенной (через локальную очередь) отправки событий ("raw event") в juggler из перловых скриптов.

    В случае неверно указанных входных параметров (например ссылки вместо скаляров) или ошибки запуска консольной утилиты - умирает.

=cut

use Direct::Modern;
use Yandex::HashUtils;
use Yandex::Shell;
use Net::INET6Glue::INET_is_INET6;
use LWP::UserAgent;
use Encode;
use JSON;
use Yandex::Retry;

use Socket qw(getaddrinfo SOCK_DGRAM AI_CANONNAME);
use Sys::Hostname qw(hostname);
use Yandex::Hostname;

use base qw/Exporter/;
our @EXPORT = qw/
    queue_juggler_event

    juggler_event
    juggler_check

    juggler_ok
    juggler_warn
    juggler_crit
/;

=head1 PARAMS

=head2 $SERVICE_NAME_GENERATOR

    Ссылка на функцию, которая должна генерировать значение service (имя сырого события)
    при его отсутствии в аргументах функций juggler_*.
    Функции передается значение параметра service_suffix, переданного в juggler_*.

    Опциональный параметр, по умолчанию - отсутствует. В этом случае наличие service - обязательно.

=cut

our $SERVICE_NAME_GENERATOR;

use constant MAXIMUM_SERVICE_LENGTH     => 128;
use constant MAXIMUM_DESCRIPTION_LENGTH => 1024;
use constant JUGGLER_API_ADDRESS => 'http://localhost:31579/events';

my %STATUS_TO_CODE = (
    OK      => 0,
    WARN    => 1,
    CRIT    => 2,
    INFO    => 3,
);
my %CODE_TO_STATUS = reverse %STATUS_TO_CODE;

sub _truncate_str {
    my ($str, $len) = @_;
    $str = substr($str, 0, $len-2).'..' if length($str) > $len;
    return $str;
}

=head1 SUBROUTINES

=head2 queue_juggler_event(%params)

    Ставит событие в очередь отправки в juggler. Принимает следующие именованные параметры:
    Обязательные:
        service     => 'some_service_name', # имя сервиса (название события). Максимум 128 символов.
    Необязательные:
        status      => 'OK',                # статус события. допустимые значения: OK, WARN, CRIT, INFO. По-умолчанию 'INFO'.
        host        => 'virtual.hostname',  # имя хоста, на который будет записано событие.
                                                Если не указано, то juggler_queue_event использует FQDN сервера.
        description => 'text desc',         # описание события (комментарий). По-умолчанию '<empty>'. Максимум 1024 символа.
        offset      => 30,                  # deprecated опция. Больше не используется.

=cut

sub queue_juggler_event {
    my %params = @_;
    
    unless ($params{service} && !ref $params{service}) {
        die q/'service' must be specified/;
    }
    if ($params{status} && (ref $params{status}
                            || !(exists $STATUS_TO_CODE{ uc($params{status}) } || exists $CODE_TO_STATUS{ $params{status} }))
    ) {
        die q/'status' has invalid value/;
    }
    if ($params{description} && ref $params{description}) {
        die q/'description' has invalid value/;
    }
    if ($params{host} && ref $params{host}) {
        die q/'host' has invalid value/;
    }
    if (not exists $params{host}) {
        my $fqdn = Yandex::Hostname::hostfqdn();
        $params{host} = $fqdn;
    }

    $params{service} = _truncate_str($params{service}, MAXIMUM_SERVICE_LENGTH);
    $params{description} = _truncate_str($params{description}, MAXIMUM_DESCRIPTION_LENGTH);

    my $ua = LWP::UserAgent->new;
    $ua->timeout(5);

    my $req = HTTP::Request->new(POST => JUGGLER_API_ADDRESS);
    $req->header('content-type' => 'application/json');
    my %event = ( 
                       "host" => $params{host},
                       "service" => $params{service},
                       "status" => $params{status},
                       "description" => $params{description}
                 );
    my %post_data = ( 
                       "source" => "juggler_events",
                       "events" => [\%event]
                    );
    my $json_data = encode_json(\%post_data);
    $req->content($json_data);
    my $resp = $ua->request($req);

    retry tries => 2, pauses => [7], sub {
        my $resp = $ua->request($req);

        if (!$resp->is_success) {
	    die "Error sending juggler event, HTTP-code: " . $resp->code . ", message: " . $resp->message;
        }
    }
}

my @queue_juggler_event_params = qw/
    service
    status
    description
/;
my @juggler_event_params = (
    @queue_juggler_event_params,
    'service_suffix',
);

=head2 juggler_event

    Обертка над queue_juggler_event, умеющая самостоятельно определять service (если не задан) с помощью $SERVICE_NAME_GENERATOR.
    Параметры именованные:
        service => ,        # имя события (по умолчанию будет сгенерировано из имени вызывающего скрипта)
        service_suffix => , # дополнительная строка к автоматически генерируемому имени события
                            # (допустим только для случаев, когда service не задан, иначе - породит исключение)
        status => ,         # статус события (OK/WARN/CRIT/INFO) (по умолчанию CRIT)
        description => ,    # описание события (по умолчанию OK)

=cut

sub juggler_event {
    my %data = @_;
    die "only one of 'service' or 'service_suffix' params should be defined" if $data{service} && defined $data{service_suffix};
    die "service or \$SERVICE_NAME_GENERATOR is not specified" if !$data{service} && (!$SERVICE_NAME_GENERATOR || ref $SERVICE_NAME_GENERATOR ne 'CODE');
    $data{service} ||= $SERVICE_NAME_GENERATOR->($data{service_suffix});
    $data{status} //= 'CRIT';
    $data{description} ||= 'OK';
    queue_juggler_event(%{ hash_cut(\%data, @queue_juggler_event_params, 'host') });
}

=head2 juggler_check

    Проверка значения value на пороги warn и crit и сообщить в juggler.
    Поддерживает все параметры juggler_event, кроме status.
    Дополнительные параметры: value, warn, crit
    Если определена только одна из границ, или warn <= crit - считаем проблемой превышение порога
    Если warn > crit - считаем, проблемой слишком маленькое значение value

    juggler_check(value => 12, crit => 1);

=cut

sub juggler_check {
    my %data = @_;

    my ($value, $warn, $crit) = delete @data{qw/value warn crit/};

    my ($status, $desc) = ("OK", "$value");
    if (!defined $value) {
        ($status, $desc) = ('CRIT', "undefined value");
    } elsif (!defined $crit && !defined $warn) {
        ($status, $desc) = ('CRIT', "undefined borders for $value");
    } else {
        if (!defined $warn || !defined $crit || $warn <= $crit) {
            if (defined $crit && $value >= $crit) {
                ($status, $desc) = ('CRIT', "$value >= $crit");
            } elsif (defined $warn && $value >= $warn) {
                ($status, $desc) = ('WARN', "$value >= $warn");
            } 
        } else {
            if (defined $crit && $value <= $crit) {
                ($status, $desc) = ('CRIT', "$value <= $crit");
            } elsif (defined $warn && $value <= $warn) {
                ($status, $desc) = ('WARN', "$value <= $warn");
            } 
        }
    }

    $data{status} = $status;
    $data{description} = ($data{description} // $data{service} // '').": $desc";

    juggler_event(%{ hash_cut(\%data, @juggler_event_params, 'host') });
}

=head2 juggler_ok

    Сообщить в juggler статус OK.
    Поддерживает все параметры juggler_event, кроме status.

=cut

sub juggler_ok {
    my %data = @_;
    $data{status} = 'OK';
    juggler_event(%{ hash_cut(\%data, @juggler_event_params) });
}

=head2 juggler_warn

    Сообщить в juggler статус WARN.
    Поддерживает все параметры juggler_event, кроме status.

=cut

sub juggler_warn {
    my %data = @_;
    $data{status} = 'WARN';
    juggler_event(%{ hash_cut(\%data, @juggler_event_params) });
}

=head2 juggler_crit

    Сообщить в juggler статус CRIT.
    Поддерживает все параметры juggler_event, кроме status.

=cut

sub juggler_crit {
    my %data = @_;
    $data{status} = 'CRIT';
    juggler_event(%{ hash_cut(\%data, @juggler_event_params) });
}

1;
