package Yandex::Ketama;

=head2 NAME

    Yandex::Ketama - аглоритм консистентного хэширования

=head2 DESCRIPTION

    простейшая реализация алгоритма консистентного хэша ketama
    http://www.lastfm.ru/user/RJ/journal/2007/04/10/rz_libketama_-_a_consistent_hashing_algo_for_memcache_clients

=cut

use strict;
use warnings;

use Carp qw/croak/;

sub new {
    my ($class, %params) = @_;
    my $self = bless {servers => [], weights_sum => 0, points => undef}, $class;

    if (!defined $params{hash_func}) {
        require Digest::MD5;
        $self->{hash_func} = sub {
            return hex(substr(Digest::MD5::md5_hex(shift), 0, 8));
        };
    } else {
        $self->{hash_func} = $params{hash_func};
    }
    if (ref $self->{hash_func} ne 'CODE') {
        croak "Incorrect param hash_func";
    }

    if( !defined $params{servers} || ref $params{servers} ne 'ARRAY' || @{$params{servers}} < 1 || @{$params{servers}} > 1000 ){
        require Data::Dumper;
        croak "Incorrect param servers: " . Data::Dumper::Dumper( $params{servers} );
    }

    for my $s (@{$params{servers}}) {
        my $n = ref $s eq 'HASH' ? $s->{name} : $s;        
        my $w = ref $s eq 'HASH' ? $s->{weight} // 0 : 1;
        $self->{weights_sum} += $w;
        push @{$self->{servers}}, {name => $n, weight => $w};
    }
    if ($self->{weights_sum} == 0) {
        $_->{weight} = 1 for @{$self->{servers}};
        $self->{weights_sum} = @{$self->{servers}};
    }

    if (defined $params{points}) {
        croak "Incorrect param points: '$self->{points}'"
            if $params{points} !~ /^[0-9]+$/ || $params{points} > 1e6 || $params{points} < 1;
        $self->{points} = $params{points};
    } else {
        # количество точек на единицу веса. может быть дробным
        $self->{points} = 1000 / $self->{weights_sum};
    }

    # инициализация
    $self->init();

    return $self;
}

# инициализируем массивы
sub init {
    my $self = shift;
    $self->{_zero_weight_servers} = [];
    # создаём континуум
    my @cont;
    for my $s (@{$self->{servers}}) {
        if ($s->{weight}) {
            my $points = int($s->{weight} * $self->{points} + 0.5);
            for(my $p = 0; $p < $points; $p++) {
                push @cont, [$self->{hash_func}->("$s->{name}-$p"), $s->{name}];
            }
        } else {
            push @{$self->{_zero_weight_servers}}, $s->{name};
        }
    }
    @cont = sort {$a->[0] <=> $b->[0]} @cont;

    # удаляем дубликаты
    $self->{_points_arr} = [];
    $self->{_servers_arr} = [];
    for my $c (@cont) {
        if (@{$self->{_servers_arr}} && $self->{_servers_arr}->[-1] eq $c->[1]) {
            $self->{_points_arr}->[-1] = $c->[0];
        } else {
            push @{$self->{_points_arr}}, $c->[0];
            push @{$self->{_servers_arr}}, $c->[1];
        }
    }
}

# найти все сервера
sub find_all {
    my ($self, $text) = @_;

    my @res = $self->find($text);
    while(defined (my $s = $self->next())) {
        push @res, $s;
    }

    return @res;
}

# найти наиболее подходящий сервер
sub find {
    my ($self, $text) = @_;

    my $x = $self->{hash_func}->($text);

    $self->{found_servers} = {};
    $self->{found_idx} = _find_idx($self->{_points_arr}, $x);
    $self->{zero_weight_idx} = 0;

    #use Data::Dumper;
    #print Dumper $x, $self->{found_idx};

    return $self->next();
}

# поиск значения в массиве
sub _find_idx {
    my ($arr, $x) = @_;

    # крайние случаи
    if ($arr->[0] >= $x || $arr->[-1] < $x) {
        return 0;
    }

    # дихотомия рулит, наверное
    my $low = 0;
    my $high = @{$arr}-1;

    while(1) {
        my $try = int(($low + $high)/2);
        if ($arr->[$try] < $x) {
            $low = $try;
        } elsif ($arr->[$try] > $x) {
            $high = $try;
        } else {
            return $try;
        }
        if ($low >= $high-1) {
            return $high;
        }
    }
}

# выдать следующий сервер
sub next {
    my ($self) = @_;

    # если все сервера уже достали
    if (keys(%{$self->{found_servers}}) == @{$self->{servers}} - @{$self->{_zero_weight_servers}}) {
        if ($self->{zero_weight_idx} < @{$self->{_zero_weight_servers}}) {
            return $self->{_zero_weight_servers}->[$self->{zero_weight_idx}++];
        } else {
            return undef;
        }
    }

    # пробегаемся по кольцу до следующего сервера
    my $arr_len = @{$self->{_servers_arr}};
    for(my $i = 0; $i < $arr_len; $i++) {
        my $server = $self->{_servers_arr}->[$self->{found_idx}];
        if (!exists $self->{found_servers}->{$server}) {
            $self->{found_servers}->{$server} = undef;
            return $server;
        } else {
            $self->{found_idx} = ($self->{found_idx} + 1) % $arr_len;
        }
    }
    return undef;
}

1;
