package Yandex::LiveFile;

=head1 NAME

    Yandex::LiveFile - данные из файла с обновлением

=head1 DESCRIPTION

    Прозрачное получение свежего содержимого файла.

    При обращении к данным, файл перечитывается, если изменился,
    но не чаще 1 раза в секунду.

=head1 SYNOPSIS

    my $f = Yandex::LiveFile->new(filename => "test.txt");
    print $f->data;

=cut

use strict;
use warnings;

use Params::Validate qw/validate/;

=head2 new

    Конструктор, принимает именованные параметры:
    filename - имя файла, из которого загружаются файлы

=cut
sub new {
    my $class = shift;
    validate(@_, {
        filename => 1,
        });
    bless {@_}, $class;
}

=head2 data

    Получение содержимого файла (если файл изменился - получаем свежее содержимое)

=cut
sub data {
    my ($self) = (shift);
    $self->refresh();
    return $self->{_data};
}

=head2 refresh

    Проверка, изменился ли файл.
    Если файл изменился - его содержимое загружается в память.
    Возвращает 1 - файл был загружен, 0 - загрузка не потребовалась.

=cut
sub refresh {
    my $self = shift;
    if (defined $self->{_data} && $self->{_load_time} == time) {
        return 0;
    }
    my $fileinfo = join(":", (stat($self->{filename}))[1,7,9]);
    if ($self->{_fileinfo} && $self->{_fileinfo} eq $fileinfo) {
        return 0;
    }

    open(my $fh, "<:utf8", $self->{filename}) || die "Can't open $self->{filename}";
    $self->{_data} = do {local $/; <$fh>};
    close($fh) || die "Can't close $self->{filename}: $!";

    $self->{_fileinfo} = $fileinfo;
    $self->{_load_time} = time();
    return 1;
}

1;

