package Log::Any::Adapter::YandexLog;

use 5.010;
use strict;
use warnings;
use utf8;

use base qw(Log::Any::Adapter::Base);

use Carp;

use Log::Any::Adapter::Util qw(make_method);
use Yandex::Log;
use Yandex::HashUtils qw/hash_cut/;

=head1 NAME

Log::Any::Adapter::YandexLog -- Log::Any adapter for Yandex::Log

=head1 SYNOPSIS

    # initialize adapter
    use Log::Any::Adapter;
    Log::Any::Adapter->set('YandexLog', msg_prefix => "[$$]", severity => 'warning');
    
    # (or attach to existent y-logger)
    my $ylog = Yandex::Log->new(...);
    Log::Any::Adapter->set('YandexLog', log => $ylog);

    # somewhere in code
    my $log = Log::Any->get_logger();
    ...
    $log->trace("resetting flag"); # skipped for 'warning' severity
    $flag->reset();
    $log->warn("weird situation detected")  if $flag->is_set();

=head1 DESCRIPTION

Log::Any::Adapter::YandexLog accepts all Yandex::Log constructor settings.

Additional options:

=over

=item *

category        -- used instead of log_file_name (if omitted)

=item *

severity        -- logging level (default: info)

=item *

extra_prefix    -- additional prefix; "<cat>" will be replaced by message category

=back

=head1 SEE ALSO

L<Log::Any|Log::Any>, L<Log::Any::Adapter|Log::Any::Adapter>, L<Yandex::Log|Yandex::Log>

=cut


our $DEFAULT_SEVERITY = $ENV{YANDEX_LOG_SEVERITY} || 'info';

our @YL_KEYS = qw/
    log_file_name
    dir_name
    date_suf
    msg_prefix
    auto_rotate
    lock
    tee
    use_syslog
    syslog_prefix
    syslog_log_severity
    no_date
/;

our %SEVERITY_SKIP;


for my $method ( Log::Any->logging_methods() ) {
    state $sev = [];
    $SEVERITY_SKIP{$method} = { map {($_ => 1)} @$sev };
    push @$sev, $method;

    make_method( $method => _get_logging_method($method) );
}

# Patches for better y::log compatibility
{
    no warnings 'once';
    # new-style proxied logging
    if (%Log::Any::Proxy::) {
        *Log::Any::Proxy::out = *Log::Any::Proxy::info;
        *Log::Any::Proxy::die = sub { Log::Any::Proxy::fatal(@_); die $_[1] };
    }
    # old-style
    make_method( out => _get_logging_method("info") );
    make_method( die => _get_logging_method("critical", "die") );
}

sub _get_logging_method
{
    my ($severity, $ya_method) = @_;
    $ya_method //= 'out';

    croak "Unknown severity level: <$severity>"  if !$SEVERITY_SKIP{$severity};

    return sub {
        my $self = shift;
        return if $self->{skip_severity}->{$severity};

        my ($msg) = shift;
        if (my $prefix = $self->{extra_prefix}) {
            $prefix =~ s/<cat>/$self->{category}/g;
            $msg = $prefix . $msg;
        }
        $self->{logger}->$ya_method($msg);
        return;
    };
}


for my $method ( Log::Any->detection_methods() ) {
    make_method($method, sub {
            my $self = shift;
            return !$self->{skip_severity}->{$method};
        });
}


sub init
{
    my ($self, %opt) = @_;

    # got prepared y-logger
    if ($opt{log}) {
        $self->{logger} = $opt{log};
    }
    else {
        my $yl_opt = hash_cut \%opt, @YL_KEYS;
        $yl_opt->{log_file_name} //= $opt{category};
        $self->{logger} = Yandex::Log->new(%$yl_opt);
    }

    $self->{extra_prefix} = $opt{extra_prefix};
    $self->_set_severity($opt{severity});

    return;
}


sub _set_severity
{
    my ($self, $level) = @_;
    $level //= $DEFAULT_SEVERITY;
    croak "Unknown severity level: <$level>"  if !$SEVERITY_SKIP{$level};

    $self->{severity} = $level;
    $self->{skip_severity} = $SEVERITY_SKIP{$level};
    return;
}

1;
