##################################################
#
#  Direct.Yandex.ru
#
#  Log
#  tools for logs
#
#  authors:
#    Elena Bolshakova <lena-san@yandex-team.ru>
#    Georgy Ivanov <gerich@yandex-team.ru>
#    Maxim Kuzmin <kuzmin@yandex-team.ru>
#    Sergey Mudrik <msa@yandex-team.ru>
#    Sergey Zhuravlev <zhur@yandex-team.ru>
#    Vasiliy Bryadov <mirage@yandex-team.ru>
#
#  $Id$
#
#  (c) 2004 - 2010 Yandex
#
##################################################

=head1 NAME

Log

=head1 DESCRIPTION

  tools for logs

=head1 SYNOPSIS

  # create /var/www/ppc.yandex.ru/protected/logs/balance_update.log.200707
  my $log = Yandex::Log->new(log_file_name => 'balance_update.log', date_suf => '%Y%m');
  $log->out('message 1');
  $log->out('message 2');

=head1 SYSLOG

На всех серверах rsyslog настроен для работы с Yandex::Log следующим образом:
Если создать лог Yandex::Log->new(use_syslog => 1, syslog_prefix => 'my_prefix', log_file_name => 'my_log.log') [ no_log => 1 опционально, чтобы писать только в syslog ]
то логи будут писаться в 
/var/log/yandex/my_prefix/my_log.log.yyyymmdd

=cut

package Yandex::Log;

use warnings;
use strict;
use Carp;
use Fcntl qw/:flock/;
use IO::Handle;
use JSON;
use Log::Syslog::Fast qw/:all/;
use Yandex::Hostname;
use POSIX qw/strftime/;
use Time::HiRes qw//;
use Try::Tiny;
use Yandex::Retry qw/retry/;
use Yandex::Log::MsgPrefixGuard;

use utf8;

=head1 VARIABLES

=head2 $LOG_ROOT

    Default directory for log files

=cut
our $LOG_ROOT ||= '/var/log';

=head2 $LOG_UMASK (default undef)

    default umask

=cut
our $LOG_UMASK;
our $NO_FILE_LOG;

=head2 $USE_SYSLOG

    Whether or not to use syslog

=cut
our $USE_SYSLOG;
our $SYSLOG_HOST = 'localhost';
our $SYSLOG_PORT = 514;
our $_LOGGER;
our $LOG_SEVERITY;

=head2 $SYSLOG_PREFIX

    Определяется на уровне проекта,
    нужен для кластеризации логов по проектам на серверах-агрегаторах

=cut
our $SYSLOG_PREFIX;

=head1 FUNCTIONS

=cut

#======================================================================

=head2 new

 my $log = Yandex::Log->new(option => 'value', ...);
 options:
   log_file_name => 'balance_update.log'   # имя файла лога, если нет no_log - обязательный параметр
   dir_name      => 'sub dir of $LOG_ROOT' # or undef
   date_suf      => '%Y%m'                 # strftime syntax for suff for log_file for log rotate
   msg_prefix    => 'my_pref:'             # out "my_pref:\tlog message" to log
   auto_rotate   =>  1 | 0                 # auto rotate log
   tee           =>  1 | 0                 # duplicate all out to STDERR
   lock          =>  1 | 0                 # flock fd before write
   no_log        =>  1 | 0                 # не писать ничего в файл, но выполнять остальные действия
   use_syslog    =>  1 | 0                 # переопределить $USE_SYSLOG для данного лога
   syslog_prefix => 'my_syslog_pref'       # переопределить $SYSLOG_PREFIX для данного лога
   syslog_log_severity => LOG_INFO         # (константа из Log::Syslog::Fast) переопределить severity для данного лога
   no_date      => 1 | 0                   # не выводить дату/время в лог, только само сообщение
   formatter     => sub(time, msg_prefix, data) # колбек для ручного форматирования строк лога
=cut

sub new
{
    my $this = shift;
    my $class = ref($this) || $this;

    my $self = {@_};
    if (($self->{use_syslog} || !$self->{no_log}) && !$self->{log_file_name}) {
        croak "log_file_name is required";
    }

    bless $self, $class;

    return $self;
}

=head2 msg_prefix

    $log->msg_prefix("[$$]")

=cut

sub msg_prefix
{
    my $self = shift;
    if (@_) {
        return $self->{msg_prefix} = shift;
    } else {
        return $self->{msg_prefix};
    }
}

=head2 msg_prefix_guard

    $guard = $log->msg_prefix_guard("[$$]")

=cut

sub msg_prefix_guard
{
    my ($self, $new_prefix) = @_;
    return Yandex::Log::MsgPrefixGuard->new($self, $new_prefix);
}

#======================================================================

=head2 reopen

  $log->reopen();

=cut

sub reopen
{
    my $self = shift;

    return if $NO_FILE_LOG || $self->{no_log};

    if ($self->{fd} && $self->{fd}->opened()) {
        $self->{fd}->close();
    }

    my $old_umask;
    my $error;
    try {
        # если нужно - ставим umask
        my $umask = defined $self->{umask} ? $self->{umask} : $LOG_UMASK;
        if (defined $umask) {
            $old_umask = umask($umask);
            die "Umask error: $!" if !defined $old_umask;
        }
        if ($self->{dir_name} && ! -d $LOG_ROOT . "/" . $self->{dir_name}) {
            mkdir $LOG_ROOT . "/" . $self->{dir_name};
        }

        # use $0 as log name
        if (! defined $self->{log_file_name}) {
            die "set log_file_name param for Yandex::Log\n";
        }

        my $filename = $self->filename();

        open $self->{fd}, ">>:utf8", $filename or die "Can't open $filename: $!";
        $self->{fd}->autoflush(1);
    } catch {
        $error = $_;
    };
    # возвращаем старый umask
    if (defined $old_umask) {
        die "Umask error: $!" if !defined umask($old_umask);
    }
    die $error if $error;

    return;
}

sub _date_suf {
    my $self = shift; 
    return $self->{cur_date_suf} = $self->{date_suf} ? "." . strftime($self->{date_suf}, localtime) : '';
}

#======================================================================

=head2 out

  $log->out("log message");

=cut

sub out
{
    my $self = shift;

    my ($profile);
    if (exists $INC{'Yandex/Trace.pm'}) {
        $profile = Yandex::Trace::new_profile('yandex_log:out', no_flush => 1);
    }

    my ($time, $micros) = Time::HiRes::gettimeofday();

    my @messages;
    if ($self->{formatter}) {
        @messages = map { $self->{formatter}->($time, $micros, $self->{msg_prefix}, _get_text_message($_))."\n" } @_;
    } else {
        my $date_str = $self->{no_date} ? '' : strftime("%Y-%m-%d\t%H:%M:%S", localtime($time))."\t";
        my $msg_prefix = $self->{msg_prefix} ? "$self->{msg_prefix}\t" : '';
        @messages = map { $date_str . $msg_prefix . _get_text_message($_) . "\n" } @_;
    }

    if ($self->{tee}) {
        print STDERR $_ for @messages;
    }

    $self->{_records_count} += @messages;

    if (!$NO_FILE_LOG && !$self->{no_log}) {
        $self->reopen_if_need();

        if ($self->{lock}) {
            flock ($self->{fd}, LOCK_EX) or warn "flock(): $!";
        }

        try {
            syswrite($self->{fd}, join "", @messages);
        } catch {
            warn $_;
        };

        if ($self->{lock}) {
            flock ($self->{fd}, LOCK_UN);
        }
    }

    if (defined $self->{use_syslog} ? $self->{use_syslog} : $USE_SYSLOG) {
        local $LOG_SEVERITY = $LOG_SEVERITY;
        $LOG_SEVERITY = $self->{syslog_log_severity} // LOG_INFO;

        my $syslog_prefix = $self->{syslog_prefix} // $SYSLOG_PREFIX;

        $self->logger->set_severity($LOG_SEVERITY);
        $self->logger->set_name("$syslog_prefix.$self->{log_file_name}");
        for my $message (@messages) {
            $message =~ s/\t|\n(?!$)/ /g;
            local $SIG{PIPE} = 'IGNORE';
            retry tries => 5, pauses => [0, 1], sub {
                if ($Yandex::Retry::iteration > 0) {
                    $self->_build_logger;
                }
                $self->logger->send($message, $time);
            };
        }
    }

    return;
}


=head2 num_records

    my $n = $log->num_records();

=cut

sub num_records {
    my $self = shift;

    return $self->{_records_count} || 0;
}


sub _get_text_message
{
    my $message = shift;
    if (ref $message) {
        return to_json($message, {
            allow_unknown => 1,
            allow_blessed => 1,
            convert_blessed => 1
        });
    } else {
        return defined $message ? $message : '';
    }
}

sub logger
{
    my $self = shift;

    unless ($_LOGGER) {
        $self->_build_logger;
    }
    return $_LOGGER;
}

sub _build_logger
{
    my $self = shift;

    die '$SYSLOG_PREFIX is not set' unless $SYSLOG_PREFIX // $self->{syslog_prefix};
    local $SYSLOG_PREFIX = $SYSLOG_PREFIX;
    $SYSLOG_PREFIX = $self->{syslog_prefix} if defined $self->{syslog_prefix};
    eval {
        my $fqdn = Yandex::Hostname::hostfqdn();
        $_LOGGER = Log::Syslog::Fast->new(LOG_TCP, $SYSLOG_HOST, $SYSLOG_PORT, LOG_LOCAL0, LOG_INFO, $fqdn, "$SYSLOG_PREFIX.$self->{log_file_name}");
        $_LOGGER->set_format(LOG_RFC5424);
    };
    if ($@) {
        Carp::cluck("can not connect to syslog at $SYSLOG_HOST:$SYSLOG_PORT: $@ (\$! = $!)");
    }
}

#======================================================================

=head2 filename()

    Имя текущего файла с логом

=cut
sub filename 
{
    my $self = shift;
    
    my $sub_dir = $self->{dir_name} ? $self->{dir_name} . "/" : '';
    return $LOG_ROOT . "/" . $sub_dir . $self->{log_file_name} . $self->_date_suf();
}

#======================================================================

=head2 reopen_if_need

  (пере)открываем файл если нужна ротация или если файл еще не был открыт

=cut

sub reopen_if_need
{
    my $self = shift;

    my $cur_date = defined $self->{cur_date_suf} ? $self->{cur_date_suf} : '';
    if ($self->{auto_rotate} && $cur_date ne $self->_date_suf() || ! ($self->{fd} && $self->{fd}->opened())) {
        $self->reopen();
    }
}

#======================================================================

=head2 die

  $log->die("log message");

=cut

sub die
{
    my $self = shift;

    local $LOG_SEVERITY = LOG_ERR;
    $self->out(@_);
    croak _get_text_message(@_);
}

#======================================================================

=head2 warn

  $log->warn("log message");

=cut

sub warn
{
    my $self = shift;

    local $LOG_SEVERITY = LOG_WARNING;
    $self->out(@_);
    carp _get_text_message(@_);
}

#======================================================================

=head2 destructor

  undef $log;

=cut

sub DESTROY
{
    my $self = shift;

    if ($self->{fd} && $self->{fd}->opened()) {
        $self->{fd}->close();
    }
}

1;

=head1 AUTHORS

in alphabetical order:

  Elena Bolshakova C<lena-san@yandex-team.ru>
  Georgy Ivanov C<gerich@yandex-team.ru>
  Maxim Kuzmin C<kuzmin@yandex-team.ru>
  Sergey Mudrik C<msa@yandex-team.ru>
  Sergey Zhuravlev C<zhur@yandex-team.ru>
  Vasiliy Bryadov C<mirage@yandex-team.ru>

=head1 COPYRIGHT

Copyright (c) 2004 - 2010 Yandex. All rights reserved.

=cut
