package Yandex::Log::Messages;

=pod

=encoding utf8

=head1 NAME

Yandex::Log::Messages

=head1 DESCRIPTION

Наследник Yandex::Log для записи в общий syslog-овый messages.log
От Yandex::Log отличается новым методом bulk_out для оптовой записи данных

=head1 SYNOPSYS

my $log = Yandex::Log::Messages->new();


$log->bulk_out([$data, "message", ...]);
$log->bulk_out(bad_data => [...]);


# поддерживаются все методы Yandex::Log
$log->out("start");
$log->warn("something wrong");
$log->die("need help");

=cut

use Direct::Modern;

use JSON;
use Yandex::Hostname;
use POSIX qw/strftime/;

use base 'Yandex::Log';

our $_HOSTNAME;

BEGIN {
    $_HOSTNAME = Yandex::Hostname::hostfqdn();
};

=head1 CONSTANTS

=head2 DESIRED_BULK_LENGTH

    Приблизительный размер строки в байтах до которого группируются данные при bulk_out.

=cut

use constant DESIRED_BULK_LENGTH => 1 * 1024 * 1024;

=head2 MAX_BULK_LENGTH

    Максимальный размер строки в байтах, которая может быть выведна в лог при bulk_out.

    Подобран так, чтобы быть меньше размера сообщений в rsyslog ($MaxMessageSize)
      direct-utils/yandex-du-rsyslog-client-conf/10-ppcsyslog-modules.conf
    и меньше лимита LogBroker (12 Мб)

=cut

use constant MAX_BULK_LENGTH => 12 * 1024 * 1024 - 4 * 1024;

=head1 PUBLIC METHODS

=head2 new

    Конструктор. Без параметров.
    Требует, чтобы был задан $Yandex::Log::SYSLOG_PREFIX и подгружен модуль Yandex::Trace.

=cut

sub new {
    die '$Yandex::Log::SYSLOG_PREFIX is not set' unless $Yandex::Log::SYSLOG_PREFIX;
    die 'Yandex::Trace should be preloaded' unless exists $INC{'Yandex/Trace.pm'};

    my $self = _init(shift);

    $self->{_use_tracing_in_meta} = 1;

    return $self;
}

=head2 new_without_tracing

    Конструктор для работы без Yandex::Trace, с заранее заданными именами трейс-сервиса и метода.
    Требует, чтобы был задан $Yandex::Log::SYSLOG_PREFIX.

    Параметры именованные, обязательные:
        trace_service   - имя сервиса в терминах трейсинга, для использовая в метаданных
        trace_method    - имя метода в терминах трейсинга.

=cut

sub new_without_tracing {
    my $this = shift;
    my (%params) = @_;

    die '$Yandex::Log::SYSLOG_PREFIX is not set' unless $Yandex::Log::SYSLOG_PREFIX;
    for my $param_name (qw/trace_service trace_method/) {
        die "$param_name is mandatory" unless $params{$param_name};
        die "$param_name contains invalid characters: spaces, '#', ',', ':' or '/'" if $params{$param_name} =~ /[\s#,:\/]/;
    }

    my $self = _init($this);
    
    $self->{_use_tracing_in_meta} = 0;
    $self->{_trace_service} = $params{trace_service};
    $self->{_trace_method} = $params{trace_method};

    return $self;
}

=head2 bulk_out([$suffix ,] $objects)

    Массовая запись данных в лог.
    При превышении json-представлением любого из объектов длины MAX_BULK_LENGTH - умирает!
    Игнорирует $self->msg_prefix и не использует его при выводе в лог.

    Параметры:
        $suffix - опциональный - будет дописан к имени трейс-метода в метаданных
        $objects - arrayref - данные для вывода в лог

    Группирует элементы из $objects в массивы размером по ~DESIRED_BULK_LENGTH и сбрасывает их в лог.
    
    $log->bulk_out([messages, objects, etc, ...]);
    $log->bulk_out(method_suffix => [messages, objects, etc, ...]);

=cut

sub bulk_out {
    my $self = shift;

    my ($suffix, $rows);
    if (scalar @_ == 2) {
        ($suffix, $rows) = @_;
    } else {
        $rows = $_[0];
    }

    local $self->{_trace_method_suffix} = $suffix;
    local $self->{_is_bulk_out} = 1;

    for my $row (@$rows) {
        my $message = $self->{_json}->encode($row);

        my $mesasge_len;
        {
            use bytes;
            $mesasge_len = length($message);
        }

        if ($mesasge_len + length("[]\n") > MAX_BULK_LENGTH) {
            # снаружи может быть перехват ошибок с записью в лог через $log->out()
            local $self->{_is_bulk_out} = undef;

            croak "Too long data given to append: length is $mesasge_len";
        }

        if ($self->{_buf_len} > 0 && $self->{_buf_len} + $mesasge_len + length("[,]\n") >= DESIRED_BULK_LENGTH) {
            $self->_bulk_flush();
        }

        $self->{_buf} .= $message;
        $self->{_buf} .= ',';
        $self->{_buf_len} += $mesasge_len + length(',');
    }
    $self->_bulk_flush(); 
}

=head2 PRIVATE METHODS

=head3 _init

    Общая часть кода конструкторов new и new_without_tracing
    Инициализирует Yandex::Log и его форматтер

=cut

sub _init {
    my $self = shift->SUPER::new(
        log_file_name => 'messages',
        date_suf => "%Y%m%d",
        auto_rotate => 1,
        tee => $ENV{LOG_TEE} // $ENV{DEBUG},
        use_syslog => 1,
        no_log => 1,
    );

    $self->{_buf} = '';
    $self->{_buf_len} = 0;
    $self->{_json} = JSON->new->allow_unknown->allow_nonref->allow_blessed->convert_blessed;

    $self->{formatter} = sub {
        my ($time, $micros, $msg_prefix, $msg) = @_;

        state $prev_time = 0;
        state $prev_micros = 0;
        state $inc = 0;
        if ($time != $prev_time || $micros != $prev_micros) {
            $prev_time = $time;
            $prev_micros = $micros;
            $inc = 0;
        } else {
            $inc++;
        }

        my ($service, $method, $trace_id, $parent_id, $span_id);

        if ($self->{_use_tracing_in_meta}) {
            if (my $trace = Yandex::Trace::current_trace()) {
                $service = $trace->service();
                $method = $trace->method();
                $trace_id = $trace->trace_id();
                $parent_id = $trace->parent_id();
                $span_id = $trace->span_id();
            } else {
                ($service, $method) = ('unknown', 'unknown');
                ($trace_id, $parent_id, $span_id) = (0, 0, 0);
            }
        } else {
            $service = $self->{_trace_service};
            $method = $self->{_trace_method};
            ($trace_id, $parent_id, $span_id) = (0, 0, 0);
        }

        my $meta = join "",
                        $_HOSTNAME,
                        ',',
                        $service,
                        '/',
                        $method, (defined $self->{_trace_method_suffix} ? ('.', $self->{_trace_method_suffix}) : ()),
                        ",",
                        $trace_id, ":", $parent_id, ":", $span_id,
                        ($self->{_is_bulk_out} ? '#bulk' : ()),
                        ;

        my $formatted_time = strftime("%Y-%m-%d:%H:%M:%S", localtime $time).".".sprintf("%09d", $micros * 1000 + $inc);
        
        return join " ", $formatted_time, $meta, ($msg_prefix && !$self->{_is_bulk_out} ? $msg_prefix : ()), $msg;
    };

    return $self;
}

=head3 _bulk_flush

    Сбрасывает буфер в лог.

=cut

sub _bulk_flush {
    my $self = shift;

    return unless $self->{_buf_len} > 0;

    # удаляем последнюю запятую, дописываем скобки, чтобы получился корректный json-array
    $self->out('[' . substr($self->{_buf}, 0, -1) . "]");

    $self->{_buf} = '';
    $self->{_buf_len} = 0;
}

1;
