package Yandex::LWPRedirect;

=pod
    $Id$
    Модуль для отслеживания редиректов
=cut

use strict;

use base 'LWP::UserAgent';

use LWP::Protocol::http;
use HTTP::Response;
use Encode qw /encode is_utf8/;
use Data::Dumper;
use Time::HiRes qw/gettimeofday tv_interval/;

use Yandex::IDN;

# for debug
our $DEBUG = 0;

sub new {
    my ( $class, @args ) = @_;
    my $self = LWP::UserAgent->new( max_redirect => 5, protocols_allowed => [qw/http https/], @args );
    bless $self;
}

sub redirect_ok {
    my ($self, $new_request, $response) = @_;
    my $uri = Yandex::IDN::idn_to_ascii($new_request->uri);

    if ( $uri =~ s!([a-z]+://[^/]+)\/\.+(/|\?|$|#)!$1$2!i ) {
        $new_request->uri($uri);
    }
    $self->{last_redirect} = $uri;
    $self->{last_url} = $uri;

    push @{$self->{redirect_chain}}, {
        href => $uri
        , label => 'http'
        , request => $new_request->headers->as_string
        , response => $response->headers->as_string
        , code => $response->code
        , decoded_content => ($response->decoded_content || ''),
    };

    print STDERR "HTTP REDIRECT founded: $uri\n" if $DEBUG;
    return 0 if !$self->SUPER::redirect_ok($new_request, $response);
    return 1;
}

sub get {
    my $self = shift;
    my $href = shift;

    print STDERR "(get), href: $href\n" if $DEBUG;

    $self->{last_url} = $href;

    # разбиваем UTF8 символы на октеты. перекодирование в cp1251 срабатывает не для всех серверов, и вообще не по rfc 
    utf8::encode($href) if Encode::is_utf8($href);;
    #$href = Encode::encode("cp1251", $href) if Encode::is_utf8($href);
    my $resp = $self->SUPER::get($href);

    if ( $resp->is_success) {
        my $key;
        my %opt=();
        my $content = $resp->decoded_content || '';
        $content =~ s/<\!(-){2,4}>//g;
        $content =~ s/<\!--.*?-->//gs;
        #$content =~ s/<\!(?:--(?:[^-]*(?:-[^-]+)*)--\s*)*>//g; #   Realy true SGML comments but IE and FF goes in other way

        # на больших объёмах регулярка может очень сильно тормозить
        $content = substr $content, 0, 10_000;
        while ( $content=~/<meta((?:\s*
                                    [^\s=<>\'\"]+
                                    \s*=\s*
                                    (?: \'[^\']*\'
                                    | \"[^\"]*\"
                                    | [^\s>=]+
                                    )
                                )*)\s*\/?>/xgi ) {
            my $atrs = $1;
            while ( $atrs =~ /  ([^\s=<>\'\"]+)
                                \s*=\s*
                                (?:\'([^\']*)\'
                                | \"([^\"]*)\"
                                | ([^\s>=]+)
                                )/xgi) {
                $opt{lc($1)} = $^N;
            }

            if ( ($opt{'http-equiv'} and $opt{'http-equiv'} =~ /refresh/i or
                  $opt{'name'} and $opt{'name'} =~ /refresh/i ) and
                  $opt{content} ) {

                if ( $opt{content} =~ /\s*0*(\d+)\s*\;\s*url\s*=\s*(\w+\:\/\/\S+)\s*/i and $1 <= 60 ) {

                    $resp->{redirected_by_meta}={time=>$1, href=>$2};
                    
                    # сохраняем здесь т.к. далее может сработать redirect_ok
                    $self->{last_redirect} = $2;
                    push @{$self->{redirect_chain}}, {
                        href => $2
                        , label => 'meta'
                        , request => $resp->request->headers->as_string
                        , response => $resp->headers->as_string
                        , code => $resp->code
                        , decoded_content => ($resp->decoded_content || '')
                    };
                    
                    print STDERR "META REDIRECT FOUNDED, href:$2\n" if $DEBUG;
                    return $resp;
                }
            }
        }
    } elsif ($resp->is_error) {
        push @{$self->{redirect_chain}}, {
            href => $href
            , label => ($resp->code == 404 ? 'Not found' : 'Error')
            , request => $resp->request->headers->as_string
            , response => $resp->headers->as_string
            , decoded_content => ($resp->decoded_content || '')
            , code => $resp->code
        };
    }

    return $resp;
}

sub get_more {
    my $self = shift;
    my $href = shift;

    print STDERR "href: $href\n" if $DEBUG;
    # redirect_chain - для показа в интерфейсе
    push @{$self->{redirect_chain}}, {href => $href, label => 'start'};

    my $start_time = [gettimeofday];
    my $timeout = $self->timeout||7;
    $self->{redirect_dict} = {};
    my $cnt = 0;
    sub max{ return $_[0]>$_[1]?$_[0]:$_[1]; }

    local @LWP::Protocol::http::EXTRA_SOCK_OPTS = (@LWP::Protocol::http::EXTRA_SOCK_OPTS, MaxHeaderLines => 1000);

    while(1){
        $self->timeout(max(1,int($timeout - tv_interval($start_time,[gettimeofday]))));
        my $resp = $self->get($href);

        if ( $resp->is_error 
                or !defined $resp->{redirected_by_meta} 
                or ($resp->{redirected_by_meta}{href} eq $self->{last_url}) 
                        and defined  $resp->{redirected_by_meta}{href}
                        and defined $self->{last_url}
        ) {
            # если случился редирект на такой же урл
            #   либо случилась ошибка при запросе
            #   либо все ок, и мета-редиректа не обнаружено (но могли внутри быть http-редиректы,
            #       которые проходят внутри LWP и обрабатываются в функции redirect_ok)
            return $resp;
        } elsif ( ++$cnt > 3) {
            return new HTTP::Response (503, 'METAREDIRECT limit exceeded');
        } elsif ( tv_interval($start_time,[gettimeofday]) > 7 ) {
            return new HTTP::Response (503, 'METAREDIRECT read timeout');
        }

        $href = $resp->{redirected_by_meta}{href};
    }
}

sub last_url {
    my $self=shift;
    return $self->{last_redirect} || $self->{last_url};
}

1;
