package LWP::UserAgent::Zora;


# $Id$

=head1 DESCRRIPTION

HTTP-запросы через Online Zora: https://wiki.yandex-team.ru/Robot/Manual/Zora/UserGuide#dostuppoprotokoluhttp

По сути это LWP::UserAgent, куда автоматом добавляются нужные заголовки,
и который будет падать при недоступности зоры.

=head1 SYNOPSIS

    $LWP::UserAgent::Zora::ZORA_SOURCENAME = 'direct_userproxy';
    my $ua = $LWP::UserAgent::Zora->new();

    my $response;
    if ( eval { $response = $ua->get($url); 1 } ) {
        # do something with response
    }
    else {
        warn "Zora is not available; try again later";
    }

=cut

use 5.010;
use strict;
use warnings;
use utf8;

use base 'LWP::UserAgent';

use Carp;

use POSIX qw/ceil/;
use Exception::Class ( ZoraException => { fields => [ qw/ response / ] } );
use Log::Any '$log';

use Yandex::TVM2;

our $ZORA_URL ||= 'http://zora.yandex.net:8166/';
our $ZORA_TVM2_ID //= 2000193;

our $ZORA_SOURCENAME; 
our $ZORA_REQUESTTYPE = 'userproxy';
our $ZORA_TIMEOUT;

# список внутренних кодов, которые считаем ошибкой фетчера, а не сайта
# https://wiki.yandex-team.ru/Robot/KodyOshibok
our %ZORA_FAIL_CODE = map {( $_ => 1 )} (
    1012, # внутренняя ошибка робота
    1022, # робот был остановлен во время скачивания документа 
    1029, # Неизвестное имя источника для спайдера/зоры
    1030, # неопределённая ошибка zora 
    1032, # ошибка zora при обработке запроса 
    1033, # не удалось соединиться с proxy 
    1034, # соединение с proxy оборвалось 
    1036, # Ошибка промежуточной прокси (squid)
);


=head2 new

    my $ua = LWP::UserAgent::Zora->new(%opt);

Options: zora_sourcename + LWP::UserAgent's

=cut

sub new
{
    my ($class, %opt) = @_;
    my $sourcename = delete $opt{zora_sourcename} // $ZORA_SOURCENAME;
    croak "Zora sourcename is not defined"  if !$sourcename;

    my $zora_timeout = delete $opt{zora_timeout};
    
    my $self = $class->SUPER::new(%opt, env_proxy => 0);

    $self->zora_timeout($zora_timeout)  if $zora_timeout;

    $self->protocols_allowed(['http', 'https']);
    $self->proxy(['http', 'https'] => $ZORA_URL);
    $self->default_headers->push_header('X-Yandex-Sourcename' => $sourcename);
    $self->default_headers->push_header('X-Yandex-Requesttype' => $ZORA_REQUESTTYPE);

    $self->_elem('proxy_delegate_https' => 1);

    return $self;
}


sub zora_timeout
{
    shift->_elem('zora_timeout', @_);
}

sub _zora_tvm_ticket {
    return eval{Yandex::TVM2::get_ticket($ZORA_TVM2_ID)} or die "Cannot get ticket for $ZORA_TVM2_ID";
}

sub simple_request
{
    my $self = shift;

    if ( my $zora_timeout = $self->zora_timeout() // $ZORA_TIMEOUT // $self->timeout() ) {
        # zora не умеет дробные таймауты (но может округлять на своей стороне)
        # принудительно округляем вверх до целого числа секунд
        $zora_timeout = ceil($zora_timeout);
        $self->default_headers->header('X-Yandex-Response-Timeout' => $zora_timeout);
    }

    # обновляем тикет перед запросом
    if ($Yandex::TVM2::APP_ID) {
        $self->default_headers->header('X-Ya-Service-Ticket' => _zora_tvm_ticket());
    }

    $log->tracef("agent: %s", $self)  if $log->is_trace;
    $log->tracef("request: %s", $_[0])  if $log->is_trace;

    my $response = $self->SUPER::simple_request(@_);

    my $is_proxy_failed =
        # internal lwp failure
        (
            $response->code == 500
            && ($response->header("Client-Warning") // 0) eq "Internal response"
            # FIX: пока зора зависает на невалидных адресах - считаем таймаут фейлом адреса, а не зоры!
            && $response->content() !~ /^read timeout/
        ) ||
        # response code indicates zora failure
        ( $response->code != 200 && $response->message =~ /^zora:/ixms && !$response->header('X-Yandex-Http-Code') ) ||
        # yandex-code indicates spider failure
        ( $response->code != 200 && $ZORA_FAIL_CODE{$response->header('X-Yandex-Http-Code') // 0} ) ||
        0;

    $log->tracef("response: %s", $response)  if $log->is_trace;

    # die if proxy failed
    ZoraException->throw( error => 'Zora failed', response => $response )  if $is_proxy_failed;

    return $response;
}

1;
