package Yandex::MDS;

=pod

=encoding utf-8

=head1 DESCRIPTION

Модуль для работы со стораджем (MDS)
https://wiki.yandex-team.ru/mds/

=head1 SYNOPSYS

my $mds = Yandex::MDS->new();

my $key = $mds->upload('cats.pdf', $cats_pdf);
my $cats_pdf_copy = $mds->get($key);
$mds->delete($key);

=cut

use strict;
use warnings;
use utf8;
use feature 'state';

use Carp;
use Yandex::HTTP qw/http_parallel_request/;
use Yandex::Log;
use Yandex::Retry qw/retry/;
use Yandex::Shell;
use Yandex::Trace;
use XML::LibXML;
use URI::Escape qw//;
use JSON;
use Path::Tiny;
use Params::Validate qw/validate/;

our $MDS_PUT_HOST //= 'storage-int.mds.yandex.net:1111';
our $MDS_GET_HOST //= 'storage-int.mds.yandex.net';

=head2 $MDS_UPLOAD_FILE_CURL_THRESHOLD

Используется в upload_file. Если размер файла больше этого числа,
для загрузки запускается внешняя программа, которая не загружает файл
целиком в память.

=cut

our $MDS_UPLOAD_FILE_EXTERNAL_PROCESS_THRESHOLD //= 50 * 1024 * 1024;

my %LOG_OPTIONS = (
    date_suf => '%Y%m',
    log_file_name => 'mds',
    auto_rotate => 1,
);

=head2 $MDS_GLOBAL_EXPIRE

TTL файлов, например '17d', которые заливаем в MDS. Нужен для заливки с тестовых сред, с которых вручную удаляется не всё.
Не используется, если не задан.

=cut

our $MDS_GLOBAL_EXPIRE;

=head2 new

my $mds = Yandex::MDS->new();

Опциональные именованные параметры:
get_host => имя хоста для GET запросов
put_host => имя хоста для PUT запросов
namespace => пространство имен в MDS
authorization => http basic authorization key

=cut

sub new
{
    my $class = shift;
    my (%opt) = validate(@_, {
        get_host => { optional => 1, regex => qr/^.+$/ },
        put_host => { optional => 1, regex => qr/^.+$/ },
        namespace => 1,
        authorization => 1,
    });
    my $self = {
        get_host => $opt{get_host} // $MDS_GET_HOST,
        put_host => $opt{put_host} // $MDS_PUT_HOST,
        namespace => $opt{namespace},
        authorization => $opt{authorization},
    };
    return bless $self, $class;
}

sub _extract_key_from_upload_xml {
    my ($filename, $xml) = @_;

    my $doc = XML::LibXML->new()->parse_string($xml)->documentElement();
    my $xpc = XML::LibXML::XPathContext->new($doc);
    my $post = ($xpc->findnodes("/post"))[0];
    my $key = $post->getAttribute("key");
    unless ($key) {
        _mds_log("update attempt for '$filename'");
        $key = (($xpc->findnodes("/post/key"))[0])->textContent;
    }
    unless ($key) {
        die "failed to get key from xml:\n$xml";
    }

    return $key;
}

=head2 upload

my $key = $mds->upload($filename, $file_content);

Сохранить под именем $filename данные $file_content
$file_content может быть ссылкой на скаляр, чтобы не копировать память лишний раз;
$key - внутреннее имя, с помощью которого можно в дальнейшем работать с этим файлом
(его придется где-то сохранить)

=cut

sub upload
{
    my $self = shift;
    my ($filename, $data) = @_;
    if ($filename =~ /^\.*$/) {
        croak "filename can not consist only of '.'";
    }
    my $profile = Yandex::Trace::new_profile('mds:upload');
    my $filename_esc = URI::Escape::uri_escape($filename);
    my $xml = retry tries => 3, pauses => [0.1, 1], sub {
        return $self->_http_request(upload => $filename_esc,
            body => $data,
            headers => {
                Authorization => "Basic $self->{authorization}",
            },
        );
    };
    undef $profile;
    _mds_log("$filename: $xml");

    return _extract_key_from_upload_xml($filename, $xml);
}

=head2 upload_file

my $path = '/path/to/file';
my $key = $mds->upload_file($path);
my $key = $mds->upload_file($path, 'name');

Загрузить файл в МДС
Второй параметр - имя файла, по умолчанию берется из $path

Возвращает mds-ключ от файла

=cut

sub upload_file
{
    my ($self, $path_str, $name) = @_;

    my $path = path($path_str);

    $name //= $path->basename;

    if ( $path->stat->size <= $MDS_UPLOAD_FILE_EXTERNAL_PROCESS_THRESHOLD ) {
        return $self->upload($name, $path->slurp_raw);
    }

    my $profile = Yandex::Trace::new_profile('mds:upload_file');
    my $xml = retry tries => 3, pauses => [0.1, 1], sub {
        my $command = [
            'curl', '-s',
            $self->make_url( upload => URI::Escape::uri_escape($name) ),
            '--request' => 'PUT',
            '--header' => "Authorization: Basic $self->{authorization}",
            '--upload-file' => $path_str,
        ];

        _mds_log( { command => $command } );

        return yash_qx(@$command);
    };
    undef $profile;

    _mds_log("$name: $xml");

    return _extract_key_from_upload_xml($name, $xml);
}

=head2 get_url

Ссылка, по которой данные доступны для скачивания

=cut

sub get_url
{
    my $self = shift;
    my $key = shift;
    return "http://$self->{get_host}/get-$self->{namespace}/$key";
}

=head2 get

my $file_content = $mds->get($key)

Получить содержимое файла по ключу.
Ключ - значение, которое возвращает upload

Умирает при ошибке

=cut

sub get
{
    my ($self, $key) = @_;
    my $profile = Yandex::Trace::new_profile('mds:get');
    return $self->_http_request(get => $key);
}

=head2 delete

$mds->delete($key);

Удалить файл по ключу $key

=cut

sub delete
{
    my ($self, $key) = @_;
    _mds_log("delete $key");
    my $profile = Yandex::Trace::new_profile('mds:delete');
    return $self->_http_request(delete => $key,
        headers => {
            Authorization => "Basic $self->{authorization}",
        },
    );
}

=head2 statistics

Ручка статистики
Работает только из внутренних сетей

=cut

sub statistics
{
    my $self = shift;
    return decode_json($self->_http_request('statistics'));
}

=head2 make_url

my $url = $mds->make_url('get', $key, %opt);
$url = "http://.../get-$ns/$key"

Формирование ссылки для МДС
%opt:
    host => $host -- использовать указанный хост вместо заданных в конструкторе

=cut

sub make_url
{
    my ($self, $action, $key, %opt) = @_;

    my $url;
    my $ns = $self->{namespace};
    if ($action =~ /^get|statistics$/) {
        my $host = $opt{host} // $self->{get_host};
        $url = "http://$host/$action-$ns";
        if ($action ne 'statistics') {
            $url .= "/$key";
        }
    }
    elsif ($action =~ /^upload|delete$/) {
        my $host = $opt{host} // $self->{put_host};
        $url = "http://$host/$action-$ns/$key";
        if ($MDS_GLOBAL_EXPIRE) {
            $url .= "?expire=$MDS_GLOBAL_EXPIRE";
        }
    }
    else {
        croak "unknown mds action '$action'";
    }

    return $url;
}

=head2 get_host

Возвращает текущий хост для чтения

=cut

sub get_host
{
    return shift->{get_host};
}

=head2 put_host

Возвращает текущий хост для записи

=cut

sub put_host
{
    return shift->{put_host};
}

=head2 _mds_log

=cut

sub _mds_log
{
    state $log = Yandex::Log->new(%LOG_OPTIONS);
    $log->out(@_);
}

{
my %action2method = (
    'upload' => 'POST',
    'get' => 'GET',
    'delete' => 'GET',
    'statistics' => 'GET',
);

=head2 _http_request

=cut

sub _http_request
{
    my ($self, $action, $key, %opt) = @_;

    unless (exists $action2method{$action}) {
        croak "unknown mds action '$action'";
    }

    my $res = http_parallel_request($action2method{$action}, { 1 => { url => $self->make_url($action => $key), %opt } }, ipv6_prefer => 1 )->{1};
    if ($res->{is_success}
        # 403 возникает при попытке перезаписи существующего файла
        # при этом возвращается ключ
        || ($action eq 'upload' && $res->{headers}->{Status} == 403)
        # если удаляемый файл не существует - считаем что это ОК
        || ($action eq 'delete' && $res->{headers}->{Status} == 404)
    ) {
        return $res->{content};
    }
    die "MDS $action ($action2method{$action}): $res->{headers}->{Status} $res->{headers}->{Reason}";
}
}

1;
