package Yandex::Memcached;

# $Id$

=head1 NAME
    
    Yandex::Memcached

=head1 SYNOPSIS

    # init
    my $md = Yandex::Memcached->new(
                                    servers => ['server1', 'server2'], 
                                    );
    $md->add('key', 'val');

=head1 DESCRIPTION

    Модуль - обёртка вокруг Cache::Memcached::Fast для работы с memcached в стиле Директа.

    Принимает массив из серверов и пытается работать с первым.
    Если сервер падает - переключается на следующий.

    Имеет постоянные коннекты к memcached.

=cut

use strict;
use warnings;

use Cache::Memcached::Fast;
use Params::Validate qw/validate ARRAYREF/;

use Yandex::Trace;
use Yandex::HashUtils qw(hash_cut);

# Здесь хранятся текущие соединения для набора серверов, если один вдруг окажется недоступным, перезапишем
my %CONNECTIONS;

=head2 $MEMCACHED_CONNECT_OPTIONS

    Настройки для Cache::Memcached::Fast

=cut
our $MEMCACHED_CONNECT_OPTIONS ||= {
    connect_timeout => 0.2,
    io_timeout => 0.5,

    max_failures => 2,
    failure_timeout => 5,
};

use constant PASSTHROUGH_KEYS => qw(
    namespace
    connect_timeout
    io_timeout
    close_on_error
    compress_threshold
    compress_ratio
    compress_methods
    max_failures
    failure_timeout
    ketama_points
    nowait
    hash_namespace
    serialize_methods
    utf8
    max_size
);

=head1 CONSTRUCTOR

=head2 new

    Параметры:

    servers - ref of array of string, servers list (servername:port)
    namespace => 'my:',
    connect_timeout => 0.2,
    io_timeout => 0.5,
    close_on_error => 1,
    compress_threshold => 100_000,
    compress_ratio => 0.9,
    compress_methods => [ \&IO::Compress::Gzip::gzip,
                          \&IO::Uncompress::Gunzip::gunzip ],
    max_failures => 3,
    failure_timeout => 2,
    ketama_points => 150,
    nowait => 1,
    hash_namespace => 1,
    serialize_methods => [ \&Storable::freeze, \&Storable::thaw ],
    utf8 => ($^V ge v5.8.1 ? 1 : 0),
    max_size => 512 * 1024,

=cut
sub new
{
    my $this = shift;
    my $class = ref($this) || $this;

    validate(@_, {
            servers => {
                type => ARRAYREF,
                callbacks => {nonempty => sub { scalar @{$_[0]} }}
            },
            map { ($_ => 0) } PASSTHROUGH_KEYS,
        }
    );
    my $self = {@_};

    my $conn = join(',', map { ref($self->{$_}) || !defined $self->{$_} ? '' : $self->{$_} }  PASSTHROUGH_KEYS);

    # идентификатор списка серверов
    $self->{conn_id} = join(',', sort @{$self->{servers}}, $conn);
    
    bless $self, $class;

    return $self;
}

=head1 METHODS

=head2 _mc_request

    Единичный запрос к Memcached серверу

=cut

sub _mc_request
{
    my $self = shift;
    my ($cmd, @params) = @_;

    my $profile = Yandex::Trace::new_profile("memcache:$cmd");

    # сначала пытаемся запросить текущий сервер
    if (ref($CONNECTIONS{$self->{conn_id}}) && defined(my $result = $CONNECTIONS{$self->{conn_id}}->$cmd(@params))) {
        return $result;
    }

    my %params = (
        %$MEMCACHED_CONNECT_OPTIONS,
        %{ hash_cut($self, PASSTHROUGH_KEYS) },
    );

    # если он недоступен, пытаемся подключиться к другому
    for my $srv (@{$self->{servers}}) {
        $CONNECTIONS{$self->{conn_id}} = new Cache::Memcached::Fast {%params, servers => [$srv]};
        if (defined(my $result = $CONNECTIONS{$self->{conn_id}}->$cmd(@params))) {
            return $result;
        }
    }

    # если терпим фиско, возвращаем undef
    return $CONNECTIONS{$self->{conn_id}} = undef;
}

=head2 set cas add replace append prepend get gets incr decr delete

    Эти методы по смыслу аналогичны методам из Cache::Memcached::Fast,
    дополнительно поддерживается переключение с упавших серверов и профайлинг

    Методы *_multi тоже доступны.

=cut
# делаем обёртки для методов работы с данными Cache::Memcached::Fast
{
    no strict 'refs';
    foreach my $method (map {($_, $_."_multi")} qw/set cas add replace append prepend get gets incr decr delete/) {
        *{__PACKAGE__ . "::$method"} = sub {
            local *__ANON__ = $method;
            return shift->_mc_request($method, @_);
        }
    }
}

1;
