package Yandex::Memcached::Lock;

# $Id$

=head1 NAME
    
    Yandex::Memcached::Lock

=head1 SYNOPSIS

    # init && lock
    my $mcl = Yandex::Memcached::Lock->new(
                                            servers => ['server1', 'server2'], 
                                            entry => 'some_id',
                                            max_locks_num => 10,
                                            expire => 40, #seconds
                                        );

    # init and take lock manually
    my $mcl = Yandex::Memcached::Lock->new(
                                            servers => ['server1', 'server2'], 
                                            entry => 'some_id',
                                            max_locks_num => 10,
                                            expire => 40, #seconds
                                            no_auto_lock => 1,
                                        );

    my $locked = $mcl->lock();

    # remove lock
    $mcl->unlock();

    # get lock
    my $locked = $mcl->get_lock();

=head1 DESCRIPTION

    Модуль предназначен для сохранения локов (например, для api пользователей).

    Умеет переключаться между серверами, если какой-то сервер падает. Минус данного
    подхода в том, что если текущий сервер упадет, то единовременно теряются все
    записанные на него данные.

    В переменной пакета хранится текущий сервер для переданного набора серверов,
    в случае его недоступности будет выбран следующий доступный из списка, если
    такого не окажется, то метод lock будет возвращать undef

    При разрушении объекта, лок удаляется из кэша.

=cut

use strict;
use warnings;

use Data::Dumper;
use List::Util qw /shuffle min/;
use Time::HiRes qw//;

use Yandex::HashUtils;
use Yandex::Memcached;
use Yandex::Trace;
use Yandex::Log;

our $LOG_FILE_NAME;

=head2 Настройки блокирующий получений локов

    $TIMEOUT_FIRST - пауза после первой неудачи
    $TIMEOUT_COEF - экспонента для уверичения пауз
    $TIMEOUT_JITTER - раздомизация пауз

=cut
our $TIMEOUT_FIRST ||= 0.2;
our $TIMEOUT_COEF ||= 1.2;
our $TIMEOUT_JITTER ||= 0.1;

my $EXPIRE_TIME = 900; # seconds
my $MAX_LOCKS_NUM = 10;

=head1 CONSTRUCTOR

=head2 new

    Параметры:
    
    все параметры, которые понимает Yandex::Memcached
    (на текущий момент - servers)

    entry - string, some entry id, which is supposed to be locked

    expire - integer, expiration time (optional, default = 900)
    max_locks_num - integer, number of maximum locks per specified entry (optional, default = 10)
    no_auto_lock - boolean, set or not lock during initialization (optional, default = false)
    no_auto_unlock - boolean, not auto unlock when destroy object (default = false)

    timeout - сколько секунд пытаться получить блокировку, по умолчанию 0 - одна попытка

=cut

sub new
{
    my ($this, %O) = @_;
    
    my $self = \%O;
    $self->{_md} = Yandex::Memcached->new(%{hash_cut \%O, qw/servers/});

    $self->{expire} ||= $EXPIRE_TIME;
    $self->{max_locks_num} ||= $MAX_LOCKS_NUM;
    $self->{no_auto_unlock} ||= 0;

    $self->{entry} or die "Yandex::Memcached::Lock: entry field must be specified";

    $self->{key} = undef;
    $self->{value} = undef;
    $self->{locked} = undef;

    if ($LOG_FILE_NAME) {
        $self->{log} = new Yandex::Log(
            log_file_name => $LOG_FILE_NAME,
            date_suf => "%Y%m%d",
            msg_prefix => "[$$]",
        );
    }

    bless $self, ref($this) || $this;
    unless ($self->{no_auto_lock}) {
        $self->lock();
    }

    return $self;
}

# внутренний метод, возвращаем инициализированный Yandex::Memcached
sub _md 
{
    shift->{_md};
}

# внутренний метод, логгируем, если лог инициализирован
sub _log
{
    my $self = shift;
    $self->{log}->out(@_) if $self->{log};
}

=head1 METHODS

=head2 lock

    Вычисляет доступный ключ, и пытается выставить по нему лок
    возвращает:
        0 - если ключ не найден, или выставить лок не получилось - 
        1 - если операцию прошла успешно
        undef - произошла ошибка или потеряли все сервера

=cut

sub lock
{
    my $self = shift;

    if (!defined $self->{timeout} || $self->{timeout} <= 0) {
        return $self->_lock_iter();
    }

    my $current_time = Time::HiRes::time;
    my $border_time = $current_time + $self->{timeout};
    my $sleep_time = 0;
    my $res;
    while($current_time <= $border_time) {
        $self->_log("lock: iteration sleep: $sleep_time");
        if ($sleep_time > 0) {
            my $profile = Yandex::Trace::new_profile("memcache:lock:sleep");
            Time::HiRes::sleep(min $sleep_time, $border_time - $current_time);
        }
        if ($res = $self->_lock_iter()) {
            last;
        }
        
        $current_time = Time::HiRes::time;
        if (!$sleep_time) {
            $sleep_time = $TIMEOUT_FIRST;
        } else {
            $sleep_time *= $TIMEOUT_COEF + (rand($TIMEOUT_JITTER) - $TIMEOUT_JITTER/2);
        }
    }

    $self->_log("lock: return ".(defined $res ? $res : 'undef'));
    
    return $res;
}

sub _lock_iter {
    my $self = shift;

    my @keys = map {$self->{entry}.'-'.$_} (1..$self->{max_locks_num});

    $self->_log('LOCK ITER START');
    
    my $locks = $self->_md->get_multi(@keys);

    my $value = time();
    
    $self->_log("current locks entry: $self->{entry}", $locks);

    if (ref($locks) eq 'HASH') {
        $self->{locked} = 0;
        for my $key (shuffle (grep {!$locks->{$_}} @keys)) {
            my $res = $self->_md->add($key, $value, $self->{expire});
            if (!defined $res) {
                $self->{locked} = undef;
                last;
            } elsif ($res) {
                $self->{key} = $key;
                $self->{value} = $value;
                $self->{locked} = 1;
                last;
            }
        }
    }

    $self->_log("lock entry: $self->{entry}, key: ".(defined $self->{key} ? $self->{key} : 'undef')
                .", result: ".(defined $self->{locked} ? $self->{locked} : 'undef')
        );
    $self->_log('LOCK ITER END');

    return $self->{locked};
}

=head2 renew_lock

    Пытается заново выставить время на локе по единственному ключу
    возвращает:
        0 - если выставить время не получилось (false for negative server reply)
        1 - если операцию прошла успешно
        undef - если ключей более одного или произошла ошибка

=cut

sub renew_lock
{
    my $self = shift;

    $self->_log('RENEW LOCK START');

    if ($self->{max_locks_num} != 1) {
        $self->_log('renew_lock called with multiple locks');
        return undef;
    }

    my $res;
    my $key = $self->{entry}.'-1';

    if ($self->{key} && $self->{value} && $self->{key} eq $key) { # обновить время
        my $cas_val = $self->_md->gets($key);
        $self->_log("current locks entry: $key ", $cas_val->[1]);
        if ($cas_val->[1] == $self->{value}) {
            $res = $self->_md->cas($key, $cas_val->[0], $cas_val->[1], $self->{expire});
        } else {
            $self->_log('stored value differs at renew_lock');
            $res = 0;
        }
    } else {
        $self->_log('renew_lock called without setting a lock first');
        return undef;
    }

    if (!defined $res) {
        $self->{locked} = undef;
    } elsif ($res) {
        $self->{locked} = 1;
    } else {
        $self->{locked} = 0;
    }

    $self->_log("lock entry: $self->{entry}, key: ".(defined $self->{key} ? $self->{key} : 'undef')
                .", result: ".(defined $self->{locked} ? $self->{locked} : 'undef')
        );
    $self->_log('RENEW LOCK END');

    return $self->{locked};
}

=head2 associate_lock(key, value)

    Связывает объект с уже существующим локом (проставляет {key, value} в объекте, если value совпадает с тем, что в локе)
    возвращает:
        undef - если не удалось связать (стоит попытаться взять новый)
        1 - если операция прошла успешно
        0 - если произошёл неправильный вызов метода

=cut

sub associate_lock
{
    my ($self, %O) = @_;

    $self->_log('ASSOCIATE LOCK START');

    unless ((defined $O{key}) && (defined $O{value})) {
        $self->_log('wrong associate_lock call (lock entry: $self->{entry})');
        return 0;
    }

    if ((defined $self->{key}) || (defined $self->{value}) || $self->{locked}) {
        $self->_log('unsuitable object (lock entry: $self->{entry})');
        return 0;
    }
    my $res;

    my $val = $self->_md->get($O{key});
    $self->_log("current lock key: $O{key} ", $val);
    if (defined $val) {
        if ($val eq $O{value}) {
            $self->{locked} = 1;
            $self->{key} = $O{key};
            $self->{value} = $O{value};
        } else {
            $self->_log('stored value differs at associate_lock');
            $self->{locked} = undef;
        }
    } else {
        $self->_log('no lock with such key');
        $self->{locked} = undef;
    }

    $self->_log('ASSOCIATE LOCK END');

    return $self->{locked};
}

=head2 no_auto_unlock

    При передаче параметра меняет значение свойства 'no_auto_unlock' на переданное.
    Возвращает значение свойства 'no_auto_unlock'.

=cut

sub no_auto_unlock
{
    my $self = shift;

    if (scalar @_) {
        my $new_val = shift;
        $self->_log('no_auto_unlock change for entry '.$self->{entry}.' to '.$new_val);

        $self->{no_auto_unlock} = $new_val;
    }

    return $self->{no_auto_unlock};
}

=head2 unlock

    Удаляет лок по текущему ключу,
    возвращает:
        0 - если ключ не задан, или удалить лок не получилось - 
        1 - если операцию прошла успешно
        undef - произошла ошибка или потеряли все сервера

=cut

sub unlock
{
    my $self = shift;
    my $unlocked = 0;
    
    if ($self->{key} && $self->{value}) {
        my $cas_val = $self->_md->gets($self->{key});
        if ($cas_val->[1] == $self->{value}) {
            $unlocked = $self->_md->delete($self->{key});
        }
    }

    if ($unlocked) {
        $self->{locked} = undef;
        delete $self->{value};
    }

    $self->_log("unlock key: ".(defined $self->{key} ? $self->{key} : 'undef').", result: $unlocked");

    return $unlocked;

}

=head2 unlock_by_entry

    Удаляет лок со всего entry (всех ключей от 1 до max_locks_num)

    возвращает:
        0 - если ключ не найден
        1 - если операцию прошла успешно

=cut

sub unlock_by_entry {
    my $self = shift;

    $self->_log('unlock_by_entry START');

    my $unlocked = 0;
    my @keys = map {$self->{entry} . '-' . $_} (1 .. $self->{max_locks_num});
    my $locks = $self->_md->get_multi(@keys);

    $self->_log("current locks entry: $self->{entry}", $locks);

    if (ref($locks) eq 'HASH') {
        for my $key (keys %$locks) {
            my $res = $self->_md->delete($key);
            if (! defined $res) {
                $unlocked = 0;
                last;
            } elsif ($res) {
                $unlocked = 1;
            }
        }
    }

    $self->{locked} = 0 if $unlocked;

    $self->_log('unlock_by_entry END');

    return $unlocked;
}

=head2 get_lock

    Возвращает текущий статус лока

=cut

sub get_lock
{
    my $self = shift;

    return $self->{locked};

}

=head2 DESTROY

    Освобождаем лок при разрушении объекта

=cut

sub DESTROY
{
    my $self = shift;

    if (! $self->{no_auto_unlock}) {
        $self->_log('DESTROY');
        $self->unlock();
    }
}

1;
