use strict;
use warnings;
use utf8;

package Yandex::Metadata;

=head1 NAME

Yandex::Metadata - чтение метаданных из файлов с кодом

=cut

use Config::General;

require Exporter;

our $VERSION = '0.01';
our @ISA = qw/Exporter/;
our @EXPORT = qw/
    extract_metadata
    deserialize_metadata
/;


=head2 extract_metadata

    выбирает pod-секцию или docstring METADATA из одного заданного файла

    Файл читается, и из него регекспом выбираются фрагменты, достаточно похожие на pod-секцию (или docstring) METADATA. 

    podselect не используется, потому что с ним ощутимо медленнее работает (20 секунд и 1 секунда)
    кроме того, podselect не понимает python docstring

    Pod::POM -- быстр, но с ним сложно толерантно обрабатывать не вполне корректный pod, и при этом не пропускать совсем уж грубые ошибки. 

    Тесты на случай, если будет меняться реализация: 
      * файл вообще без pod'ов
      * пропущенная пустая строка после =head1 METADATA (Pod::POM считает весь следующий текст заголовком)
      * пропущенная пустая строка перед =cut (Pod::POM не может разобрать такой pod)
      * после корректной секции =head1 METADATA -- секция =pod, в т.ч. без пустых строк после заголовка
      * после корректной секции =head1 METADATA -- секция =head2, в т.ч. без пустых строк после заголовка
      * пустые строки в секции METADTA (с Pod::POM непонятно, как отделить ровно содержимое секции верхнего уровня, без "вложенных" секций)

=cut

sub extract_metadata
{
    my ($file, $text) = @_;
    
    #my $md = qx{podselect -section 'METADATA/!.+' $file};
    #return $md;

    # вынимаем из файла "текст, похожий на pod-секцию METADATA"
    my @pod_md_texts = $text =~ /^(=head1\s+METADATA\s*\n.*?^=(?:cut|head1.*?)\s*)$/gsm;
    if (@pod_md_texts) {
        foreach my $pod_md_text_piece (@pod_md_texts) {
            $pod_md_text_piece =~ s/^=head1\s.*?$//sm;
            $pod_md_text_piece =~ s/^=(?:cut|head1.*?)\s*$//sm;
        }
    }

    my $pod_md_text = join "\n\n", @pod_md_texts;
    # проверка, что в $crontab_pod не попало лишнего
    die "wrong pod from file $file\n$pod_md_text\nwrong pod from file $file (see above)" if $pod_md_text =~ /^=/m;

    my @docstring_md_texts = $text =~ /^("""\s+METADATA\s*\n.*?^"""\s*)$/gsm;
    if (@docstring_md_texts) {
        foreach my $docstring_md_text_piece (@docstring_md_texts) {
            $docstring_md_text_piece =~ s/^"""\s.*?$//sm;
            $docstring_md_text_piece =~ s/^"""\s*$//sm;
        }
    }
    my $docstring_md_text = join "\n\n", @docstring_md_texts;

    return "" unless $pod_md_text || $docstring_md_text;
    return join("\n\n", $pod_md_text || '', $docstring_md_text || '');
}

# на входе -- текст из секции METADATA, на выходе -- ссылка на хеш
sub deserialize_metadata
{
    my ($str) = @_;

    return {} unless $str;

    my $conf_obj = new Config::General( "-String" => $str, -CComments => 0, -SplitDelimiter => '\s*:\s*', -SplitPolicy => 'custom' );

    return {$conf_obj->getall};
}

1;
