#!/usr/bin/perl -w

=head1 TEST

    проверяем, правильно ли извлекаются метаданные из скриптов

    $Id$

=cut

use strict;
use warnings;

use Test::More;
use Test::Exception;
use utf8;

use Yandex::Metadata;

# корректные скипты. Должны корректно обрабатываться
my @correct_tests = (
    {
        name => 'no_cron_no_pod',
        cron_count => 0,
        script => q%#!/usr/bin/perl
print "hello world\n";
        %,
    },
    {
        name => 'pod_without_cron',
        cron_count => 0,
        script => q%#!/usr/bin/perl

=head1 SMTH

text

=cut
print "hello world\n";
        %,
    },
    {
        name => 'metadata_without_cron',
        cron_count => 0,
        script => q%#!/usr/bin/perl
=head1 METADATA

    <smth>
        a: 1
    </smth>

=cut
print "hello world\n";
        %,
    },
    {
        name => 'cron_simple',
        cron_count => 1,
        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: * * * * *
        package: scripts
    </crontab>

=cut

print "hello world\n";
        %,
    },
    {
        name => 'cron_with_c_style_comment',
        cron_count => 1,
        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: */3 * * * *
        package: scripts
    </crontab>

=cut

print "hello world\n";
        %,
    },
    {
        name => 'cron_without_empty_lines',
        cron_count => 1,
        script => q%#!/usr/bin/perl
=head1 METADATA
    <crontab>
        time: * * * * *
        package: scripts
    </crontab>
=cut
print "hello world\n";
        %,
    },
    {
        name => 'cron_without_empty_lines_2',
        cron_count => 2,
        script => q%#!/usr/bin/perl
=head1 METADATA
    <crontab>
        time: * * * * *
        package: scripts
    </crontab>
    <crontab>
        time: 3 * * * *
        package: scripts
        params: --three
    </crontab>
=cut
print "hello world\n";
        %,
    },
    {
        name => 'cron_without_empty_lines_3',
        cron_count => 2,
        script => q%#!/usr/bin/perl
=pod
=head1 METADATA
    <crontab>
        time: * * * * *
        package: scripts
    </crontab>
    <crontab>
        time: 3 * * * *
        package: scripts
        params: --three
    </crontab>
=cut
print "hello world\n";
        %,
    },
    {
        name => 'multi_cron_with_empty_lines',
        cron_count => 2,
        script => q%#!/usr/bin/perl
=head1 METADATA
    <crontab>
        time: * * * * *
        package: scripts
    </crontab>

    <crontab>
        time: * * * * *
        package: scripts
    </crontab>
=cut
print "hello world\n";
        %,
    },
    {
        name => 'comments',
        cron_count => 0,
        script => q%#!/usr/bin/perl
=head1 METADATA
    # <crontab>
    #     time: * * * * *
    #     package: scripts
    # </crontab>
=cut
print "hello world\n";
        %,
    },
############### python
    {
        name => 'no_cron_no_docstring',
        cron_count => 0,
        script => q%#!/usr/bin/env python
# -*- coding: utf-8 -*-
print("hello world\n");
        %,
    },
    {
        name => 'docsting_without_cron',
        cron_count => 0,
        script => q%#!/usr/bin/env python
# -*- coding: utf-8 -*-

""" SMTH

text

"""
print("hello world\n");
        %,
    },
    {
        name => 'metadata_without_cron',
        cron_count => 0,
        script => q%#!/usr/bin/env python
# -*- coding: utf-8 -*-
""" METADATA

    <smth>
        a: 1
    </smth>

"""
print("hello world\n");
        %,
    },
    {
        name => 'cron_simple',
        cron_count => 1,
        script => q%#!/usr/bin/env python
# -*- coding: utf-8 -*-

""" METADATA

    <crontab>
        time: * * * * *
        package: scripts
    </crontab>

"""

print("hello world\n");
        %,
    },
    {
        name => 'cron_with_c_style_comment',
        cron_count => 1,
        script => q%#!/usr/bin/env python
# -*- coding: utf-8 -*-

""" METADATA

    <crontab>
        time: */3 * * * *
        package: scripts
    </crontab>

"""

print("hello world\n");
        %,
    },
    {
        name => 'cron_without_empty_lines',
        cron_count => 1,
        script => q%#!/usr/bin/env python
# -*- coding: utf-8 -*-
""" METADATA
    <crontab>
        time: * * * * *
        package: scripts
    </crontab>
"""
print("hello world\n");
        %,
    },
    {
        name => 'cron_without_empty_lines_2',
        cron_count => 2,
        script => q%#!/usr/bin/env python
# -*- coding: utf-8 -*-
""" METADATA
    <crontab>
        time: * * * * *
        package: scripts
    </crontab>
    <crontab>
        time: 3 * * * *
        package: scripts
        params: --three
    </crontab>
"""
print("hello world\n");
        %,
    },
    {
        name => 'multi_cron_with_empty_lines',
        cron_count => 2,
        script => q%#!/usr/bin/env python
# -*- coding: utf-8 -*-
""" METADATA
    <crontab>
        time: * * * * *
        package: scripts
    </crontab>

    <crontab>
        time: * * * * *
        package: scripts
    </crontab>
"""
print("hello world\n");
        %,
    },
    {
        name => 'comments',
        cron_count => 0,
        script => q%#!/usr/bin/env python
# -*- coding: utf-8 -*-
""" METADATA
    # <crontab>
    #     time: * * * * *
    #     package: scripts
    # </crontab>
"""
print("hello world\n");
        %,
    },
    {
        name => 'cron_outside_metadata',
        cron_count => 1,
        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: */3 * * * *
        package: scripts
    </crontab>

=head1 SMTH

    <crontab>
        time: */2 * * * *
        package: scripts
    </crontab>

=cut

print "hello world\n";
        %,
    },
);

my @incorrect_tests = (
    {
        name => 'metadata_integrated_with_other_pod',
        script => q%#!/usr/bin/perl
=head1 METADATA
    <crontab>
        time: * * * * *
        package: scripts
    </crontab>

    <crontab>
        time: * * * * *
        package: scripts
    </crontab>

=head2

    text    

=cut
print "hello world\n";
        %,
    },
    {
        name => 'incorrect_config_syntax',
        script => q%
=head1 METADATA

    <crontab>
        time: * * * * *
        package: scripts

=cut
print "hello world\n";
        %,
    },

);

Test::More::plan(tests => scalar(@correct_tests) + scalar(@incorrect_tests));

for my $t (@correct_tests){
    my $crontab = deserialize_metadata(extract_metadata($t->{name} => $t->{script}))->{crontab} || [];
    $crontab = [$crontab] unless ref $crontab eq 'ARRAY';
    is(scalar @$crontab, $t->{cron_count}, $t->{name});
}

for my $t (@incorrect_tests) {
    dies_ok { deserialize_metadata(extract_metadata($t->{name} => $t->{script})) } $t->{name};
}
