package Yandex::MisspellChecker;

=head1 NAME

    $Id$
    Модуль реализующий XML-API к сервису проверки опечаток Большого Поиска.
    Автор:
        Черненко Дмитрий <trojn@yandex-team.ru>
        Andrei Lukovenko <aluck@yandex-team.ru>

=head1 SYNOPSIS

    Пример вызова:
        my $ms_checker = new Yandex::MisspellChecker ( 'is_test' => 1 );
        my $results = $ms_checker->check( [ 'фраза с очепяткой' ] );

=head1 DESCRIPTION

    Конструктор класса:

        new Yandex::MisspellChecker

        В конструктор можно передать хеш с настройками для будущих запросов к сервису.

        Пример хеша с необязательными настройками (в нём указаны значения по умолчанию):
            'lang'    => 'ru,en,uk',    # Языковая отметка. Список доступных значений:
                                        # "be", "en", "kk", "ru", "tr", "uk", "" и "*".
                                        # "" и "*" - использовать все языки.
                                        # Можно указывать несколько языков перечислением через запятую.

            'options' => MSOPT_MARKUP + MSOPT_IGNORE_NICKNAMES,
                                        # опции, описаны константами MSOPT_*

            'dbgwzr'  => 0,             # Включает выдачу отладочной информации. Возможные значения: 0, 1, 2, 3

            'fix'     => 1,             # не только проверять текст, но и показывать исправления

            'is_test' => 0,             # Указывает тип используемого сервиса: тестовый или боевой.

        NOTE: параметр ie не поддерживаем, всегда делаем запросы в utf-8

    Подробнее о параметрах:
        http://wiki.yandex-team.ru/LingvisticheskieTexnologii/Opechatki/ServisOpechatok/SpellcheckerApi

    Пример запроса:
        http://erratum-test3:19036/misspell/check?srv=tools&fix=1&text=%D0%BF%D0%B0%D0%BB%D0%B8%D1%82%D0%B5%D0%BA%D0%B0%20%D0%BF%D0%B0%D1%80%D1%82%D0%B5%D0%B8

=cut

use utf8;
use strict;
use warnings;
use feature 'state';

use Carp;
use Encode qw/encode_utf8 decode_utf8/;
use HTTP::Request::Common;
use Module::Load 'load';

use Yandex::HTTP qw( http_parallel_request );
use Yandex::Trace;

use constant {
    MSOPT_MARKUP           => 0x0001, # выделять исправленный фрагмент.
    MSOPT_NO_PORNO_RULE    => 0x0040, # отключить порно-словарь
    MSOPT_NO_TRANSLIT      => 0x0080, # отключить правило "Translit"
    MSOPT_IGNORE_NICKNAMES => 0x0100, # игнорировать hash-tags и nicknames (ex. #hello, @world)
    MSOPT_NO_KEYBOARD      => 0x0400, # отключить правило "Keyboard"
};

our $MISSPELL_REQUEST_SCHEME ||= 'http';

our $MISSPELL_HOST ||= 'misc-spell.yandex.net:19036';
our $MISSPELL_REQUEST_URI ||= '/misspell/check';

our $MISSPELL_TEST_HOST ||= 'erratum-test3.yandex.ru:19036';
our $MISSPELL_TEST_URI  ||= '/misspell/check';

our $RESPONSE_CLASS //= 'Yandex::MisspellChecker::Response';

our $MISSPELL_SRV  //=  'definemepleaseforprod';

=head3 $MISSPELL_MAX_ATTEMPTS

    Количество попыток запроса к серверу

=cut

our $MISSPELL_MAX_ATTEMPTS ||= 3;

=head3 $MISSPELL_PARALLEL_REQS

    Количество одновременных запросов к серверу

=cut

our $MISSPELL_PARALLEL_REQS ||= 4;

my $MISSPELL_ALLOWED_OPTIONS = {
    'lang'    => 1,
    'dbgwzr'  => 1,
    'options' => 1,
};

our $MISSPELL_DEFAULT_OPTIONS ||= {
    'lang'    => 'ru,en,uk',
    'fix'     => 1,
    'srv'     => $MISSPELL_SRV,
    'ie'      => 'utf-8',
    'dbgwzr'  => 0,
    'options' => MSOPT_MARKUP + MSOPT_IGNORE_NICKNAMES,
};

=head2 $MISSPEL_TIMEOUT

    Тайм-аут на выполнение HTTP-запроса. По умолчанию равен 1 секунде.

    $Yandex::MisspellChecker::MISSPEL_TIMEOUT = 5; # тайм-аут выставлен на 5 секунд

=cut

our $MISSPEL_TIMEOUT ||= 1;

=head2 new

    my $checker = new Yandex::MisspellChecker ();

    my $checker = new Yandex::MisspellChecker ( 'lang' => 'ru' );

=cut

sub new {
    my ( $class, %params ) = @_;

    my $is_test = $params{is_test};

    # переведём все ключи в нижний регистр (во избежание опечаток в параметрах)
    # удалим все параметры, которые не знает Опечаточник
    foreach my $key ( keys( %params ) ) {
        my $lc_key = lc( $key );

        if ( $MISSPELL_ALLOWED_OPTIONS->{ $lc_key } ) {
            $params{ $lc_key } = delete( $params{ $key } );
        } else {
            delete $params{ $key };
            if ((!exists($MISSPELL_DEFAULT_OPTIONS->{ $lc_key })) && ($lc_key ne 'is_test')) {
                carp( "Invalid parameter: $key" );
            }
        }
    }

    %params = ( %{ $MISSPELL_DEFAULT_OPTIONS }, %params );

    my $url = $is_test
                ? $MISSPELL_REQUEST_SCHEME . '://' . $MISSPELL_TEST_HOST. $MISSPELL_TEST_URI
                : $MISSPELL_REQUEST_SCHEME . '://' . $MISSPELL_HOST. $MISSPELL_REQUEST_URI;

    return( bless( {
        'request_options' => \%params,
        'url' => $url,
    }, $class ) );
}


=head2 check

    my $checker_responses = $checker->check( $texts, %options );

    Выполняет запрос к серверу опечаточника

    $texts - arrayref на проверяемые тексты
    %options - необязательные параметры:
        safety - не падать при ошибке во время выполнения запроса.
        timeout - время таймаута в секундах. По умочланию равно 1.
        need_highlight - необязательный параметр, возможные значения 1 или 0.
                         Если значение TRUE, то в случае наличия опечатки будет произведено обрамление
                         опечаток в исходном тексте html-тегами <font color="red">опечатка</font>
                         За подробностями смотрите пакет Yandex::MisspellChecker::Response.
       on_error => sub { my ( $error_msg, $status_code ) = @_; ... }
                Вызывается при HTTP error, параметры - текст и код (AnyEvent::HTTP)

    Возвращает arrayref на Yandex::MisspellChecker::Response (в том же порядке, что и входные значения),
    либо undef при ошибке

=cut

sub check {
    my ( $this, $texts, %options ) = @_;

    my $profile = Yandex::Trace::new_profile('misspellchecker:check');

    croak( __PACKAGE__ . ': missing mandatory parameter $texts' ) unless ($texts);

    my $results;
    eval {
        $results = $this->_request( $texts, %options );
    };

    if ( $@ ) {
        my $error_text =  __PACKAGE__ . ': ' . $@;
        $results = undef;

        if ( $options{ 'safety' } ) {
            carp( $error_text );
        } else {
            croak( $error_text );
        }
    }

    return $results;
}


=head2 _request

   Выполняет http-запрос и разбирает ответ от сервера опечаточника.

   $texts - Arrayref на проверяемые тексты. Обязательный параметр.
   %options - дополнительные параметры:
       timeout - время таймаута в секундах. По умочланию равно 1
       need_highlight - необязательный параметр, возможные значения 1 или 0.
                        Если значение TRUE, то в случае наличия опечатки будет произведено обрамление
                        опечаток в исходном тексте html-тегами <font color="red">опечатка</font>
                        За подробностями смотрите пакет Yandex::MisspellChecker::Response.
       on_error => sub { my ( $error_msg, $status_code ) = @_; ... }

=cut

sub _request {
    my ( $this, $texts, %options ) = @_;

    my $timeout = $options{'timeout' } || $MISSPEL_TIMEOUT;
    my $on_error = delete($options{'on_error'}) || sub {};

    my %params = %{ $this->{ 'request_options' } };
    my @params = map { $_, encode_utf8( $params{$_} ) } keys %params;

    my %requests;
    my $text_no = 0;
    foreach my $text (@$texts) {
        $requests{$text_no++} = {
                url     => $this->{url},
                body    => [ @params, 'text', encode_utf8($text) ],
            };
    }

    my $responses = http_parallel_request(
                'POST',
                \%requests,
                headers         => {
                        'Host'          => $this->{url},
                        'Content-Type'  => 'application/x-www-form-urlencoded',
                        'Accept'        => 'text/html',
                    },
                maxreq          => $MISSPELL_PARALLEL_REQS,
                timeout         => $timeout,
                num_attempts    => $MISSPELL_MAX_ATTEMPTS
            );

    my @results;
    foreach my $res_id (sort { $a <=> $b } keys %$responses) {
        my $response = $responses->{$res_id};
        if ( $response->{is_success} ) {
            my $checker_response = $this -> response_class() -> new();
            $checker_response->parse(
                    'data' => decode_utf8($response->{content}),
                    'need_highlight' => $options{ 'need_highlight' } || 0
                );

            push @results, $checker_response;
        } else {
            $on_error->( $response->{headers}{Reason}, $response->{headers}{Status} );
            push @results, undef;
        }
    }

    return \@results;
}

=head2 response_class()

Возвращает строку, представляющую имя класса объекта ответа сервиса.

=cut

sub response_class {

    state $dummy = do { load $RESPONSE_CLASS; 1; };

    return $RESPONSE_CLASS
}

1;
__END__
