package Yandex::MisspellChecker::Response;

# $Id$

=head1 NAME

=head1 SYNOPSIS

=head1 DESCRIPTION

    Сервис возвращает результат в XML элементе <MisspellResult>

    <?xml version="1.0" encoding="utf-8" ?> 
    <MisspellResult code="201" lang="ru,en" rule="Volapyuk" flags="512" r="100">
     <srcText>maskva</srcText> 
     <text>м­(о)­сква</text> 
    </MisspellResult>

    <?xml version="1.0" encoding="utf-8"?>
    <MisspellResult code="201" lang="ru,en" rule="Misspell" flags="0" r="8000">
     <srcText>п<fix>а</fix>л<fix>е</fix>тика пар<fix>тте</fix>и</srcText>
     <text>п<fix>о</fix>л<fix>и</fix>тика парт<fix>и</fix>и</text>
    </MisspellResult>

    Элемент <MisspellResult> содержит следующие атрибуты:

        code - код возврата. Возможные значения:
            OK = 200 - запрос не содержит ошибки;
            CORRECTED = 201 - запрос полностью исправлен;
            NOT_CORRECTED = 202 - запрос содержит несловарные слова, которые не были исправлены
            (определение "словарности" делается по морф. словарю. Этот метод имеет высокую точность, но низкую полноту.
            В результате код 202 может быть возвращен для орфографически правильного запроса. Поэтому не рекомендуется
            по-разному обрабатывать коды 200 и 202);
            BAD_QUERY = 401 - некорректный запрос. Сейчас некорректными считаются запросы превышающие 1000 символов.
        lang - языковая отметка запроса.
        rule - сработавшее правило:
            Misspell - исправлена опечатка;
            KeyboardLayout - исправлена раскладка клавиатуры;
            Volapyuk - транслитное исправление.
        flags - дополнительная информация (битовая маска). Возвращаются следующие флажки:
            MSRF_UNRELIABLE = 0x0004 - исправление не должно автоисправляться,
            MSRF_PORNO = 0x0008 - признак "porno",
            MSRF_CONTEXT = 0x0020 - наличие "контекстных" исправлений (словарного слова на словарное),
            MSRF_SPLIT = 0x0040 - были "разрезаны" слова,
            MSRF_JOIN = 0x0080 - были "склеены" слова,
            MSRF_REJECTED = 0x0100 - есть "отвергнутая" подсказка (например, некликабельная или порно-подсказка). Таким образом, запрос не был исправлен "полностью".
            MSRF_VOLAPYUK = 0x0200 - транслитное исправление ("moskva" -> "москва")
            MSRF_KEYBOARD = 0x0400 - клавиатурное исправление ("vjcrdf" -> "москва")
        r степень надежности исправления:
            10000 - подсказка высоконадежная (подтвержденная кликами)
            8000 - подсказка надежная (но не подтвержденная кликами)
            100 - подсказка малонадежная

    Если запрос был исправлен, то <MisspellResult> содержит следующие элементы:

        srcText - исходный запрос;
        text - подсказка.

    При установленной опции MSOPT_MARKUP = 1 исправленный фрагмент выделяется в обоих элементах <srcText> и <text>.
    Выделение может отсутствовать, если Сервис не смог его сделать или посчитал его излишним (например, Volapyuk-подсказки никогда не выделяются).

=cut

use strict;
use XML::LibXML;
use Scalar::Util qw/blessed/;

use constant {
    MSRESP_CODE_OK              => 200,
    MSRESP_CODE_CORRECTED       => 201,
    MSRESP_CODE_NOT_CORRECTED   => 202,
    MSRESP_CODE_BAD_QUERY       => 401,
};

=head2 new
 
    Конструктор.
    
    my $ms_response = new Yandex::MisspellChecker::Response ( 'data' => $xml_string );
    my %result = $ms_response->parse();
 
        или
    
    my $ms_response = new Yandex::MisspellChecker::Response ();
    my %result = $ms_response->parse( 'data' => $xml_string );

=cut

sub new {
    my ( $class, %params ) = @_;
    return( bless( \%params, $class ) );
}

=head2 parse

    Парсит XML-данные.

    $ms_response->parse( 'data' => $xml_string, 'need_highlight' => 0 );

        или

    my $ms_response = new Yandex::MisspellChecker::Response ( 'data' => $xml_string );
    my %result = $ms_response->parse( 'need_highlight' => 0 );

    Принимает в качестве параметра хеш со следующими ключами:
        data - необязательный параметр, строка XML - ответ от сервиса поискового опечаточника.
               Если не указана будет взято значение переданное в конструктор.
        need_highlight - необязательный параметр, возможные значения 1 или 0.
               Если значение TRUE, то в случае наличия опечатки будет произведено обрамление
               опечаток в исходном тексте html-тегами <font color="red">опечатка</font>.

    Возвращает хеш со следующими ключами:
        misspells - список слов, где были найдены опечатки;
        srcText - исходный текст запроса;
        fixedText - исправленный текст;
        hilightedText - подсвеченный текст (только в случае need_highlight != FALSE)
        lang - языковая отметка запроса (например, 'ru,en');
        flags - битовая маска;
        reliability - степень надежности исправления;
        rule - сработавшее правило;
        code - код возврата.

=cut

sub parse {

    my ( $this, %options ) = @_;

    $this->{ 'data' } = $options{ 'data' } // $this->{ 'data' } // '';

    my $data = $this->{ 'data' };
    # my $doc = XML::LibXML->load_xml( 'string' => $data, 'no_network' => 1 ); #Такой вызов будет работать только начиная с ubuntu precise

    my $parser = XML::LibXML->new();
    my $doc = $parser->parse_string($data);

    my $root_node = ( $doc->getElementsByTagName( 'MisspellResult' ) )[0];
    my @attributes = qw/code lang rule flags r/;

    $this->{ 'result' } = {};

    foreach my $attr ( @attributes ) {
        $this->{ 'result' }{ $attr eq 'r' ? 'reliability' : $attr } = $root_node->getAttribute( $attr );
    }

    # если сервис поискового опечаточника считает. что запрос к нему не содержал ошибок,
    # либо он их не смог исправить, либо был сделан к нему неправильный запрос,
    # то вернём только значения атрибутов корневого элемента
    return( %{ $this->{ 'result' } } ) if ( $this->{ 'result' }{ 'code' } != &MSRESP_CODE_CORRECTED );

    # запрос к поисковому опечаточнику содержал ошибки и они были исрпавлены

    # исходный текст запроса
    my $srcText_node = ( $root_node->getChildrenByTagName( 'srcText' ) )[0];
    my $srcText = $this->{ 'result' }{ 'srcText' } = $srcText_node ? $srcText_node->textContent() : '';

    # исправленный текст
    my $text_node = ( $root_node->getChildrenByTagName( 'text' ) )[0];
    my $text = $this->{ 'result' }{ 'fixedText' } =  $text_node ? $text_node->textContent() : '';

    if ( $srcText ) {

        my $tagName = $srcText_node->localname();

        $srcText = $srcText_node->toString();
        $srcText =~ s#</?$tagName.*?>##g;

        $this->{ 'result' }{ 'misspells' } = [
            map {
                ( my $word = $_ ) =~ s#</?fix>##g;
                $word =~ s/^\W+//;
                $word =~ s/\W+$//;
                $word;
            }
            grep { $_ =~ m/fix>/ }
            split( /\s+/, $srcText )
        ];

        if ( $options{ 'need_highlight' } ) {
            ( $this->{ 'result' }{ 'hilightedText' } = $srcText ) =~ s#<fix>(.*?)</fix>#<font color="red">$1</font>#g;
        }
    }

    return( %{ $this->{ 'result' } } );
}

=head2 result

    Возвращает результат работы метода parse.
=cut

sub result {
    return( %{ shift( @_ )->{ 'result' } || {} } );
}

=head2 is_success

    Возвращает значение ИСТИНА, если код возврата от сервиса опечаточника равен коду запроса без ошибок.
    Иначе вернёт значение ЛОЖЬ.

=cut

sub is_success {
    return( shift( @_ )->code() == &MSRESP_CODE_OK );
}

=head2 is_misspell

    Возвращает значение ИСТИНА, если код возврата от сервиса опечаточника равен коду запроса, содержащему исправленную опечатку.
    Иначе вернёт значение ЛОЖЬ.

=cut

sub is_misspell {
    return( shift( @_ )->code() == &MSRESP_CODE_CORRECTED );

}

=head2 is_not_corrected

    Возвращает значение ИСТИНА, если код возврата от сервиса опечаточника равен коду запроса, содержащему неисправленную опечатку.
    Иначе вернёт значение ЛОЖЬ.

=cut

sub is_not_corrected {
    return( shift( @_ )->code() == &MSRESP_CODE_NOT_CORRECTED );
}

=head2 is_error

    Возвращает значение ИСТИНА, если код возврата от сервиса опечаточника равен коду плохого запроса или коду запроса с неисправленной ошибкой.
    Иначе вернёт значение ЛОЖЬ.

=cut

sub is_error {

    my $this = shift( @_ );
    my $code = $this->code();

    return( $code == &MSRESP_CODE_BAD_QUERY );
}

=head2 xml_data

    Возвращает XML строку ответа от сервиса опечаточника (параметр 'data' из конструктора).

    my $xml_data = $this->xml_data();

=cut

sub xml_data {

    my $this = shift( @_ );

    return( $this->{ 'data' } );
}

=head2 srcText

    Возвращает исходный текст запроса.
    Только для случаев успешно исправленного запроса.

    my $srcText;

    if ( $ms_response->code() == 201 ) {
        $srcText = $ms_response->srcText();
    }

=cut

=head2 fixedText

    Возвращает исправленный текст запроса.
    Только для случаев успешно исправленного запроса.

    my $fixedText;

    if ( $ms_response->code() == 201 ) {
        $fixedText = $ms_response->fixedText();
    }

=cut

=head2 hilightedText

    Возвращает исправленный текст запроса с html-подстветкой опечаток.
    Только для случаев успешно исправленного запроса.

    my $hilightedText;

    if ( $ms_response->code() == 201 ) {
        $fixedText = $ms_response->hilightedText();
    }

=cut

=head2 lang

    Возвращает языковую отметку запроса.

    my $lang = $ms_response->lang();

=cut

=head2 flags

    Возвращает битовую маску.

    my $flags = $ms_response->flags();

=cut

=head2 reliability

    Возвращает степень надежности исправления.

    my $reliability = $ms_response->reliability();

=cut

=head2 rule

    Возвращает название сработавшего правила.
    В случае кода возврата равному 202 (ошибка в тексте не исправлена) принимает пустое значение "".

    my $rule = $ms_response->rule();

=cut

=head2 code

    Возвращает код возврата.

    my $code = $ms_response->code();

=cut

=head2 misspells

    Возвращает ссылку на массив со словами, где были найдены опечатки.
    Если опечатки не были найдены, то вернёт значение undef.

    my $code = $ms_response->code();

=cut

sub AUTOLOAD {

    my ( $this, @args ) = @_;
    my %fields = map { $_ => 1 } qw/srcText fixedText hilightedText lang flags reliability rule code misspells/;
    # my %fields = map { $_ => 1 } keys( %{ $this->{ 'result' } || {} } );

    my $package = ref( $this ) || __PACKAGE__;
    ( my $method = $Yandex::MisspellChecker::Response::AUTOLOAD ) =~ s/.*:://;

    die( "Method '${method}' is not exist in package ${package}." ) unless ( $method && $fields{ $method } && defined( blessed( $this ) ) );

    eval "*${package}::${method} = sub {

        {   # Дадим имя функции, чтобы понятнее было разбирать stacktrace.
            no strict 'refs';

            *${package}::__ANON__ = '${method}';
        }

        my \$self = shift( \@_ );

        return( ( \$self->{ 'result' } || {} )->{ '${method}' } );
    };";

    die( $@ ) if ( $@ );

    return( $this->$method( @args ) );
}

sub DESTROY {
}

1;
__END__
