package Yandex::ML::SClope::Online;
use Yandex::ML::SClope::FP;

use List::MoreUtils qw(uniq);

use constant CHART           => 0;
use constant TRANSACTIONS    => 1;

use Data::Dumper;
use List::Util qw(max);
use Yandex::Trace;
use strict;
use warnings;

# для Online-кластеризации на вход поступает список транзакций вида
# [ 
#   {
#       id => 123,
#       attributes => [123,555,666],
#   },
#   {
#       id => 567,
#       attributes => [765,123,888]
#   }, 
#   ...
# ]

sub new {
    # инищализируем поточную кластеризацию
    my ($class, %params) = @_;
    my $self = { %params };
    bless $self, $class;

    $self->{tree} = Yandex::ML::SClope::FP->new();
    return $self;
};

sub add_transaction
{
    my ($self, $transaction)  = @_;
    
    my $profile = Yandex::Trace::new_profile('ml:sclope:online:add_transaction');

    $self->{transactions}++;

    $transaction->{attributes} = [ sort {
                                            my $max = max ((values %{$self->{freqs}}), 1);

                                            my $l =  (exists $self->{freqs}{$a}) ? $self->{freqs}{$a} : rand();
                                            my $r =  (exists $self->{freqs}{$b}) ? $self->{freqs}{$b} : rand();

                                            $r <=> $l
                                        } @{ $transaction->{attributes} } ];

    $self->{tree}->insert_transaction( $transaction );

    foreach my $attr (@{$transaction->{attributes}}) {
        $self->{freqs}{$attr}++;
    }

}

my $CACHE_VALUE = {}; # кэш значение width**(-r)

sub _get_microcluster_features
{
    my ($self, $cluster_id) = @_;
    
#    die Dumper($self->{tree}{microclusters});

    my $cluster = $self->{tree}{microclusters}{$cluster_id};

    my $width = scalar( keys %{$cluster->[CHART]} );
    my $order = scalar( keys %{$cluster->[TRANSACTIONS]} );
    my $size = 0;

    for my $attr ( keys %{$cluster->[CHART]} ) {
        $size += $cluster->[CHART]{$attr};
    };
    
    #print STDERR Dumper([$width, $order, $size, $cluster]);

    return ($width, $order, $size);
}

sub profit {
    # вычисление прибыли для текущих микрокластеров
    my ( $self, @clusters ) = @_;
    
    my $REPULSION = $self->{repulsion};

    # вычисляем размеры всех кластеров по атрибутам и транзакциям
    @clusters = keys %{$self->{tree}{microclusters}} unless scalar @clusters;

    my $profit = 0;
    my $total_order = 0;

    for my $cluster_id ( @clusters ) {
        my ($width, $order, $size) = $self->_get_microcluster_features($cluster_id);

        if ( !exists($CACHE_VALUE->{$width}) ) {
            $CACHE_VALUE->{$width} = $width ** (-$REPULSION);
        }

        $total_order += $order;
        $profit += $size * $CACHE_VALUE->{$width} * $order;
        #   print STDERR "cluster:\t$cluster_id,width:\t$width,size:\t$size,order:\t$order\tprofit:$profit\n";
    }

#   $profit = $profit / $total_order;
    return ($profit, $total_order);
}

sub merge_clustering {
    my ( $self, $cluster_limit ) = @_;
    my $profile = Yandex::Trace::new_profile('ml:sclope:online:merge_clustering');

    # если кластеров меньше, чем задано, выходим
    if ( scalar( keys %{$self->{tree}{microclusters}} ) < $cluster_limit ) {
        return $self->{tree}{microclusters};
    }
    
    my @non_leafs = grep { scalar( keys %{ $self->{tree}{nodes}[$_][5] } > 1 ) } (0..scalar(@{$self->{tree}{nodes}}));
    
    #Подбираем длину префикса.
    my $mClusters_count = 0;
    my $common_prefix = 2;
    my $cnt = $self->{transactions};

    #print STDERR "Optimal clusters count: ".int(sqrt($self->{transactions}))." ".$self->{transactions}."\n";
    
    do {
        $common_prefix++;
        my @candidates = grep { $self->{tree}{nodes}[$_][8] >= $common_prefix } @non_leafs;

        $mClusters_count = keys %{ $self->{tree}{microclusters} };
        
        my %merged = ();

        foreach my $c (@candidates) {
            my @clusters = grep {! exists $merged{$_} } keys(%{$self->{tree}{nodes}[$c][7] });
            $mClusters_count -= scalar(@clusters);
            $mClusters_count++;
        }
        
        #print STDERR "prefix_len: $common_prefix, mClusters_count: $mClusters_count\n";

      }  while ($mClusters_count < int(sqrt($self->{transactions})) && $cnt-- );

      #print STDERR "calculated common prefix: $common_prefix\n";

    # в дереве найдем все ноды, у которых больше, чем два дитя отсортируем их по убыванию пути от root
    # берем все ноды кроме нулевой
    # оставляем из них только те, у которых больше 1 дитя
    # сортируем по убыванию по росту
    # сортируем по убыванию по количеству детей
    my $candidate_nodes =
        [
            sort { scalar(keys %{$self->{tree}{nodes}[$b][5]} ) <=> scalar(keys %{$self->{tree}{nodes}[$a][5]} ) }
            sort { $self->{tree}{nodes}[$b][8] <=> $self->{tree}{nodes}[$a][8] } #
            grep { $self->{tree}{nodes}[$_][8] >= $common_prefix  } # в дереве найдем все ноды, у которых больше, чем два дитя
            @non_leafs
        ];

#warn "...scalar(candidate_nodes):\t" . scalar(@$candidate_nodes) . "...";

#warn "total clusters = " . scalar( keys %{ $self->{tree}{microclusters} } ) . "\n";
    # готово.
    
    for my $node2split ( @$candidate_nodes ) {
        # сливаем все кластеры, которые есть в этой ноде
        my $microcluster_ids = [ keys %{$self->{tree}{nodes}[$node2split][7] } ];
        
        $self->merge_microclusters(@$microcluster_ids);
        
        # если домержевались до предела
        last if ( scalar( keys %{ $self->{tree}{microclusters} } ) <= $cluster_limit );
        # иначе переходим к следующей ноде
    }
        
    return $self->{tree}{microclusters};
}

sub merge_microclusters
{
    my ($self,  @microcluster_ids) = @_;

    # создаем новый микрокластер
    
    my $new_cluster = [];
    $new_cluster->[TRANSACTIONS]    = {};
    $new_cluster->[CHART]       = {};

    for my $microcluster_id ( grep { defined $self->{tree}{microclusters}{$_} }  @microcluster_ids ) {
        for my $trans_id ( keys %{$self->{tree}{microclusters}{$microcluster_id}[TRANSACTIONS]} ) {
            $new_cluster->[TRANSACTIONS]{$trans_id} = 1;
        } # for trans
        
        for my $chart_attribute ( keys %{$self->{tree}{microclusters}{$microcluster_id}[CHART]} ) {
            $new_cluster->[CHART]{$chart_attribute} += $self->{tree}{microclusters}{$microcluster_id}[CHART]{$chart_attribute};
        } # for chart
    } # for mc_id
    
    for my $microcluster_id ( @microcluster_ids ) {
        delete $self->{tree}{microclusters}{$microcluster_id}; # удалим все микрокластеры
    } # for mc_id

    # добавим новый микрокластер к ним...
    my $new_cluster_id = ++$self->{tree}{nClusterCount};
    
    #print STDERR "$new_cluster_id == (".join(" ", @microcluster_ids).")\n";

    # транзакции
    for my $trans_id ( keys %{ $new_cluster->[TRANSACTIONS] } ) {
        $self->{tree}{microclusters}{$new_cluster_id}[TRANSACTIONS]{$trans_id} = 1;
    }

    # атрибуты
    for my $chart_attr ( keys %{ $new_cluster->[CHART] } ) {
        $self->{tree}{microclusters}{$new_cluster_id}[CHART]{$chart_attr} = $new_cluster->[CHART]{$chart_attr};
    }
    
    return $new_cluster;
}

1;
