package Yandex::CheckMobileRedirect;

=head1 DESCRIPTION

Модуль для простукивания ссылок на мобильные приложения

=head1 SYNOPSYS

use Yandex::CheckMobileRedirect qw/check_href_match/;

my $res = check_href_match([$url0, $url1]);

$res->[0] == [ { store1 description }, { store2 descritpion } ] # result for $url0

=cut

use strict;
use warnings;
use utf8;

use JSON;

use Yandex::HTTP qw/http_parallel_request/;
use Yandex::URL qw/get_num_level_domain get_host strip_protocol/;
use URI::Escape qw/uri_unescape/;

use base qw/Exporter/;

our @EXPORT_OK = qw/
    check_href_match
    parse_store_url
/;

=head2 KNOWN_TRACKING

Хеш с описанием правил скачивания и парсинга ссылок трекингов

домен_первого_уровня => {
    get => \&_task_sub,
    parse => \&_parse_sub,
}

_task_sub принимает url и возвращает хеш (или список хешей), пригодный для передачи в http_parallel_request

_parse_sub принимает результат от http_parallel_request (хеш с ->{content} и ->{headers}) и возвращает
url, на который эта страница делает редирект.

Если редирект делается с помощью http заголовков, то parse можно не указывать

=cut

my %KNOWN_TRACKING = (
    'measurementapi.com' => {
        get => \&_task_simple,
        parse => \&_parse_measurementapi,
    },
    'onelink.me' => {
        get => \&_task_onelink,
        parse => \&_parse_onelink,
    },
    'adjust.com' => {
        get => \&_task_adjust,
        parse => \&_parse_adjust,
    },
    'yandex.ru' => {
        get => \&_task_simple,
    },
    'kochava.com' => {
        get => \&_task_simple,
    },
);

# разные юзер агенты
my %USER_AGENT = (
    ios => 'Mozilla/5.0 (iPhone; CPU iPhone OS 6_1_3 like Mac OS X) AppleWebKit/536.26 (KHTML, like Gecko) Version/6.0 Mobile/10B329 Safari/8536.25',
    android => 'Mozilla/5.0 (Linux; U; Android 2.2; en-gb; GT-P1000 Build/FROYO) AppleWebKit/533.1 (KHTML, like Gecko) Version/4.0 Mobile Safari/533.1',
);

our $DEBUG //= 0;

=head2 REQUEST_PARAM

Параметры по-умолчанию для http запросов

=cut

our $REQUEST_PARAM //= {
    # зора пока что не умеет в https
    # https://st.yandex-team.ru/ZORA-153
    # proxy => 'zora.yandex.net:8166',
    # headers => {
    #     'X-Yandex-Requesttype' => 'userproxy',
    #     'X-Yandex-Sourcename' => 'direct_userproxy',
    # },
};

=head2 check_href_match($links, %options)

Проверить ссылки
$links - ссылка на массив строк - ссылки
%options:
    max_req => 20 -- количество одновременных запросов

Возвращает массив массивов хешей:
[
    [
        {
            app_id => '',
            store_name => '',
            redirect_href => '',
        },
        ...
    ]
    ...
]

i-й элемент массива-результата соответствует i-му элементу из запроса
если редирект невалидный - на соответствующей позиции будет undef

=cut

sub check_href_match
{
    my ($links, %opt) = @_;
    return [] unless $links && @$links;
    
    my %link2ids = ();
    my %req;
    my $i = 0;
    for my $link (@$links) {
        for my $req (_get_task_by_href($link)) {
            $req{$i} = $req;
            push @{$link2ids{$req->{url}}}, $i;
            $i++;
        }
    }

    my $fetched = http_parallel_request(
        GET => \%req,
        %$REQUEST_PARAM,
        max_req => $opt{max_req} // 20,
    );

    my @result;
    my %seen;
    for my $id (sort { $a <=> $b } keys %$fetched) {

        next if $seen{$id};

        my $link = $req{$id}->{url};
        my @ids  = @{$link2ids{$link}};

        $seen{$_}++ for @ids;

        my @page_result;

        for my $page_id (@ids) {
            my $page = $fetched->{$page_id};
            my $parsed = [ map { parse_store_url($_) } grep { defined } _parse_by_result($page) ];
            push @page_result, @$parsed;
        }
        push @result, \@page_result;
    }
    return \@result;
}

=head2 _parse_by_result

По полученным данным вернуть ссылку(ки), на которые идет редирект

Параметр: 
{
    content => "page body",
    headers => {
        # response headers,
    },
}

=cut

sub _parse_by_result
{
    my ($page) = @_;

    my $url = $page->{headers}->{URL};

    my ($proto) = ($url =~ m!^(.+?)://!);

    if ($proto =~ /https?/) {
        # некоторые трекеры при мобильном юзер-агенте сразу переадресуют на market:// или itms-apps://
        if (!$page->{is_success}) {
            if ($DEBUG) {
                warn "failed to fetch $page->{headers}->{URL}";
            }
            return undef;
        }
    }

    my $host = get_num_level_domain(get_host($url), 2);
    if (exists $KNOWN_TRACKING{$host}) {
        return $KNOWN_TRACKING{$host}->{parse}->($page);
    }
    elsif ($url =~ m!^https?://(?:itunes|apps)\.apple.com/! || $url =~ m!^https?://play\.google\.com/!) {
        return $url;
    }
    elsif ($url =~ m!^itms-apps://(?:itunes|apps).apple.com!) {
        return $url =~ s!itms-apps://!https://!r;
    }
    elsif ($url =~ m!^market://details!) {
        return $url =~ s!^market://!https://play.google.com/store/apps/!r;
    }

    if ($DEBUG) {
        warn "can not parse '$url'";
    }
    
    return undef;
}

=head2 _get_task_by_href

По указанной ссылке вернуть один или более хешей, пригодных для передачи в http_parallel_request:

_get_task_by_href('http://...') => ( { url => 'http://', headers => {}, ... }, ... );

=cut

sub _get_task_by_href
{
    my $href = shift;

    my $host = get_num_level_domain(get_host($href), 2);

    if (exists $KNOWN_TRACKING{$host}) {
        return $KNOWN_TRACKING{$host}->{get}->($href);
    }
    if ($DEBUG) {
        warn "unknown host '$host' ($href)";
    }
    return ();
}

### parsers / tasks

=head2 _task_simple

Простое задание для http_parallel_request - только url

=cut

sub _task_simple
{
    return { url => shift };
}


sub _parse_measurementapi
{
    my ($page) = @_;
    # var urls = ["yandexmail://?mat_click_id=52ec3b191028fa50272d5aa0f7c3572d-20150624-17058","https://itunes.apple.com/ru/app/andeks.pocta/id441785419?mt=8\u0026at=11l9Wx\u0026ct=yamail_MAT","https://itunes.apple.com/ru/app/andeks.pocta/id441785419?mt=8\u0026at=11l9Wx\u0026ct=yamail_MAT"];
    my ($urls_str) = ($page->{content} =~ m!var \s+ urls \s+ = \s+ (\[.*\]);!x);
    unless ($urls_str) {
        return;
    }
    my $urls = eval { decode_json($urls_str); };
    if ($@) {
        warn "failed to parse urls from string '$urls_str' : $@";
        return;
    }
    for my $url (@$urls) {
        my $host = get_host($url);
        if ($host eq 'itunes.apple.com' || $host eq 'apps.apple.com') {
            return $url;
        }
    }
    return;
}

sub _task_onelink
{
    my $url = shift;
    my @res = ();
    push @res, {
        url => $url,
        headers => {
            'User-Agent' => $USER_AGENT{ios},
        }
    };
    push @res, {
        url => $url,
        headers => {
            'User-Agent' => $USER_AGENT{android},
        }
    };
    return @res;
}

sub _parse_onelink
{
    my $page = shift;

    if ($page->{content} =~ m!var storeURL = "itms-apps://((?:itunes|apps)\.apple\.com/.+?)";!) {
        return 'https://'.$1;
    }

    if ($page->{content} =~ m!var storeLink = "market://(details.+?)";!) {
        return 'https://play.google.com/store/apps/'.$1;
    }

    warn "can not parse onelink page";
    return;
}

sub _task_adjust
{
    my $url = shift;
    my @res = ();
    push @res, {
        url => $url,
        headers => {
            'User-Agent' => $USER_AGENT{ios},
        }
    };
    push @res, {
        url => $url,
        headers => {
            'User-Agent' => $USER_AGENT{android},
        }
    };
    return @res;
}

sub _parse_adjust
{
    my $page = shift;
    if ($page->{content} =~ m!window.location.replace\("(https://(?:itunes|apps).apple.com/.+?)"\);!) {
        return $1;
    }
    return;
}

=head2 parse_store_url

    Парсит URL cтора (Google Play или Apple App Store), извлекая из него информацию про тип стора, приложение, страну и категорию.
    Если удалось распарсить url, то возвращает ссылку хеш, иначе undef.
    Особенности Google Play:
        Если в параметрах ссылки страна не определена, то будет Россия (ru).
        В параметре hl украинский язык uk заменяется на код страны ua.
    Особенности Apple App Store:
        Если в параметрах ссылки страна не определена, то будет США (us).

    Параметры:
        $url - строка, URL стора
            Примеры: 'https://play.google.com/store/apps/details?id=com.avtofriend.com.avtofriend&foo=baz&hl=tr&param=blah'
                     'https://itunes.apple.com/ru/app/garageband/id408709785'

    Результат:
        $result - ссылка на хеш или undef, если не удалось извлечь даннные
        $result = {
                    store_content_id => 'id408709785',
                    store_country => 'ru',
                    os_type => 'iOS',
                    content_type => 'app'
                  }
        $result = {
                    store_content_id => 'com.avtofriend.com.avtofriend',
                    store_country => 'ru',
                    os_type => 'Android',
                    content_type => 'app'
                  }

=cut

sub parse_store_url {

    my $url = shift;

    return undef unless $url;

    # избавляемся от двойных кавычек
    $url =~ s/^"(.*)"$/$1/;

    # Отделяем параметры от ссылки
    my ($link, $params) = split('\?', strip_protocol($url), 2);
    # Параметры преобразуем в хеш
    my %hash_params = map { my ($param, $value) = split('=', $_, 2); { uri_unescape($param) => uri_unescape($value) } } split('&', $params // '');
    my $domain = get_host($link);

    my $result;

    if ($domain) {
        if ($domain eq 'play.google.com') {
            my $content_type;
            my @values = split('/', $link);
            # ищем что-то похожее на категорию
            foreach my $value (@values) {
                # Пока поддерживаем только приложения, остальные типы контента недоступны
                # $content_type = $value if $value =~ /^(apps|movies|music|books|newsstand)$/;
                $content_type = $value if $value =~ /^(apps)$/;
            }
            # валидируем id приложения
            my $store_content_id = $hash_params{id} if ($hash_params{id} =~ /^([a-z][a-z_0-9]*\.)*[a-z][a-z_0-9]*$/i);
            my $store_country = $hash_params{gl} if ($hash_params{gl} && $hash_params{gl} =~ /^[a-z]{2}$/i);
            # Извлекаем параметры из ссылки на Google Play: gl - страна, hl - язык
            if ($store_content_id && $content_type) {
                $result = {store_content_id => $store_content_id, store_country => $store_country, os_type => 'Android', content_type => $content_type};
            }
        } elsif ($domain eq 'itunes.apple.com' || $domain eq 'apps.apple.com') {
            # Извлекаем параметры из ссылки на Apple App Store
            # Пока поддерживаем только приложения ($content_type = 'app'), остальные типы контента (movie|album|book|podcast) недоступны
            my ($country, $content_type, $app_id) = ($link =~ /(?:itunes|apps)\.apple\.com\/+(?:([a-z]{2})\/)?(app)\/(?:.+\/)?(id[0-9]+)/i);
            if ($app_id && $content_type) {
                $result = {store_content_id => $app_id, store_country => lc ($country // ''), os_type => 'iOS', content_type => $content_type};
            }
        }
    }

    # исправляем результат парсинга
    if ($result) {
        $result->{is_default_country} = 0;
        if ($result->{os_type} eq 'Android') {
            # для Google Play
            my $content_types = {
                'apps' => 'app',
                'movies' => 'movie',
                'music' => 'music',
                'books' => 'book',
                'newsstand' => 'newsstand',
            };
            $result->{content_type} =  $content_types->{$result->{content_type}};

            unless ($result->{store_country}) {
                $result->{store_country} = 'ru'; # Россия (ru) по умолчанию
                $result->{is_default_country} = 1;
            } else {
                $result->{store_country} = substr ($result->{store_country}, 0, 2);  # в параметре может быть ru_RU - язык_СТРАНА
            }
        } elsif ($result->{os_type} eq 'iOS') {
            # для Apple App Store
            $result->{content_type} =  'music' if $result->{content_type} eq 'album';
            $result->{store_country} = 'us' if $result->{store_country} eq 'en';
            unless ($result->{store_country}) {
                $result->{store_country} = 'ru'; # Россия (ru) по умолчанию
                $result->{is_default_country} = 1;
            }
        }
        $result->{store_country} = uc $result->{store_country};
    }
    else {
        if ($DEBUG) {
            warn "can not parse store url '$url'";
        }
    }

    return $result;
}

1;
