package Yandex::MongoDB;

=pod

    Функции модуля:
        1. освобождает от необходимости знаний про коннекты
        2. уметь сохранять файл в монго
        3. уметь читать файл из монго

    Checklist:

        1. прослойка на файл-хранилищах
            должна уметь по урлу искать отдавать файлы

        2. пакет для перла
            с монгой, с файл-хранилищем

        + 3. авторизация доступа по локальным паролям
            поскольку права на доступ по паролю - то храним пароль локально на серверах

        4. настройка nginx'a для проксирования (выбор сервера из текущего ДЦ)
            на уровне конфига nginx делается во время postinst запись про ближайший по ДЦ сервер mongodb и туда транслирует запросы
            

    Описание общей логики:
        1. скрипты и интерфейс пишет и читает файлы через Yandex::MongoDB (напрямую в MongoDB без прослоек) (row-storage)
        2. ссылки на api-отчеты проксируются через api-nginx на сервера с хранилищами (api-storage)
        3. на каждом сервере-хранилище есть nginx для отдачи контента (api-storage)


    Логин-пароль для авторизации с базой лежит в файле $MONGO_CONFIG_FILE
    Авторизация вынесена отдельно т.к. настройки прав как в MySQL отсутствует. Доступ контролируется только по логин-паролю.
    Предполагается, что на продакшене будут сгененрированы секретный
    Формат:
        login:password (например, adiuser:utro)
=cut

use Direct::Modern;

use MongoDB;
use MongoDB::GridFS;

use Yandex::Log;
use Yandex::HashUtils;
use Yandex::Trace;

use JSON;
use FileHandle;
use File::Slurp;
use Scalar::Util qw/reftype/;
use Scope::Guard qw/guard/;


=head2 $CONFIG_FILE

    Файл со списком серверов с mongodb

    JSON-сериализованный конфиг со списком серверов с базой
    {servers_list:["server1", "server2"]}

=cut

our $CONFIG_FILE;

=head2 $AUTH_CONFIG_FILE

    Файл с авторизационными настройками к БД

    {
        db_auth:{
            "ppcfiles":{
                login:"anylogin2",
                pass:"anypass2"
            }
        }
    }

=cut

our $AUTH_CONFIG_FILE;

=head2 @SERVERS_LIST

    Список серверов с mongodb (приоритет выше, чем у $CONFIG_FILE)

=cut

our @SERVERS_LIST;

=head2 MONGODB_LOG_FILE


=cut

our %MONGODB_LOG_FILE;
%MONGODB_LOG_FILE = (
    log_file_name => "mongodb.log",
    date_suf => "%Y%m%d",
) if !%MONGODB_LOG_FILE;

=head2 $MIN_REPL_SAVE

    Кол-во реплик на которые должны быть реплицированы записанные файлы - иначе выдаст ошибку
        либо можно установить одну из констант:
                        veryImportant: {"dc": 3}, # копирование в 3 ДЦ (не считая backup)
                        sortOfImportant: {"dc": 2}, # копирование в 2 ДЦ (не считая backup)
                        backedUp {"dc": 2, "backup": 1} # копирование в 2 ДЦ (включая backup)

=cut

our $MIN_REPL_SAVE ||= "sortOfImportant";

=head2 EXAMPLE

    my $mongodb = Yandex::MongoDB->new(db => 'mongodb_name');

    my $result = $mongo->gridfs_put_data("collection_name", $content, {filename => "xxx"});
    
    my $result1 = $mongo->gridfs_get_data("collection_name", {filename => "xxx"});
    if ($result1->{success}) {
        my $result2 = $mongo->gridfs_remove_data("collection_name", {filename => "xxx"});
    } else {
        warn join ", ", @{$result1->{errors} || []};
    }

=head2 TODO

    1. список серверов сделать переменной величиной в конструкторе объекта класса
    2. сделать get_servers_list методом класса, возвращающим список из объекта
    
=cut
    
our %HMONGO;

sub new
{
    my $this = shift;
    my $class = ref($this) || $this;

    my $self = {@_};

    die "Mongo's db name is not defined" unless $self->{db};
    
    $MONGODB_LOG_FILE{msg_prefix} = $$;
    $self->{log} = Yandex::Log->new(%MONGODB_LOG_FILE);

    my @list_mongo_servers = get_servers_list();
    
    # проверяем наличие файла с авторизационными данными
    # TODO: сделать возможность хранения авторизационных данных в Settings ?
    $self->{_auth_info} = $AUTH_CONFIG_FILE && -f $AUTH_CONFIG_FILE ?
                                from_json(scalar(read_file($AUTH_CONFIG_FILE))) 
                                : undef;

    # строка получается со списком всех серверов, отсортированных в правильном порядке
    # когда функция get_servers_list будет возвращать список серверов в оптимальном порядке (с учетом ДЦ)
    my $mongo_servers_list = shift @list_mongo_servers;
    if (scalar @list_mongo_servers > 0) {
        $mongo_servers_list .= ",".join(",", @list_mongo_servers);
    }

    if (defined $HMONGO{$mongo_servers_list} && $HMONGO{$mongo_servers_list}{pid} == $$) {

        $self->{_connection} = $HMONGO{$mongo_servers_list}->{handle};

    } else {

        eval {
            $self->{_connection} = MongoDB::Connection->new(
                                        host => "mongodb://$mongo_servers_list"
                                        , timeout => $self->{timeout} || 1000
                                        , query_timeout => $self->{query_timeout} || 6000 # in ms
                                        , wtimeout => $self->{wtimeout} || 1000
                                        , find_master => 1
                                        , auto_connect => 1
                                   );
        };

        if ($@) {
            $self->{log}->out($@);
            $self->{log}->die("Failed connect to servers: $mongo_servers_list.");
        }

        $HMONGO{$mongo_servers_list} = { handle => $self->{_connection}, pid => $$};
    }

    $self->{min_repl_save} ||= $MIN_REPL_SAVE;

    # Инициализируем хэндлеры для записи с базу и таблицу
    if ($self->{db}) {
        
        # авторизация на уровне отдельной базы
        if (defined $self->{_auth_info}
                && defined $self->{_auth_info}->{db_auth}
                && defined $self->{_auth_info}->{db_auth}{$self->{db}}) {

            my $mongo_login = $self->{_auth_info}->{db_auth}{$self->{db}}{login};
            my $mongo_pass = $self->{_auth_info}->{db_auth}{$self->{db}}{password};
            
            $self->{_connection}->authenticate($self->{db}, $mongo_login, $mongo_pass);
        }
        
        $self->{_database} = $self->{_connection}->get_database($self->{db});
    }
    
    $self->{_collection} = $self->{_database}->get_collection($self->{collection}) if $self->{collection} && $self->{_database};

    return bless $self, $class;
}

=head2 get_servers_list

    Возвращает список серверов с mongodb

=cut

sub get_servers_list
{
    my @servers_list = @SERVERS_LIST;
    
    if (! @SERVERS_LIST && $CONFIG_FILE) {
        my $config = from_json(scalar(read_file($CONFIG_FILE)));
        @servers_list = @{ $config->{servers_list} || [] };
    }

    return @servers_list;
}

=head2 _check_error

    Проверяет результат последнего запроса по следующим параметрам:
        1. результат скопирован на $MIN_REPL_SAVE реплик
        2. не было ошибок на стороне сервера
        3. время записи не более 180 сек

    Подробности о других возможностях:
        http://www.mongodb.org/display/DOCS/getLastError+Command

    Возвращает структуру:
    {
        'connectionId' => 83,
        'err' => undef,
        'n' => 0,
        'ok' => '1',    # означает что-то странное и всегда 1 (если достучались до бд)
        'wtime' => 0
    }

    При ошибке, например:
    {
        'connectionId' => 80,
        'err' => 'norepl',
        'n' => 0,
        'ok' => '1',
        'wnote' => 'no replication has been enabled, so w=2+ won\'t work'
    }

=cut

sub _check_error
{
    my ($self) = @_;

    my $last_timeout = $self->{_connection}->query_timeout;
    my $g = guard {
        $self->{_connection}->query_timeout($last_timeout);
    };

    my $query_timeout = $self->{wtimeout} || 180_000;
    $self->{_connection}->query_timeout($query_timeout);
    my $err = $self->{_database}->last_error({
        w => $self->{min_repl_save},
        wtimeout => $query_timeout 
    });

    $self->{log}->out($err);

    return $err;
}

sub _process_result
{
    my ($last_error, $mongodb_gridfs_file) = @_;

    my @errors = ();
    push @errors, $last_error->{err} if $last_error && $last_error->{err};
    
    if (ref $mongodb_gridfs_file && ref $mongodb_gridfs_file !~ /^MongoDB/
                || ! ref $mongodb_gridfs_file && defined $mongodb_gridfs_file && $mongodb_gridfs_file != 1) {
        push @errors, 'bad return value';
    }

    return {
        'errors' => \@errors
        , 'success' => (scalar @errors || !$last_error->{ok} ? 0 : 1)
        , 'status' => $last_error
        , 'metadata' => $mongodb_gridfs_file
    };
}
=pod

    Функции для работы с GridFS

=cut

=head2 gridfs_put_data(self, collection, content, {filename => "any_name"})

    Сохраняем данные в mongodb(gridfs)

    content может быть скаляром (сами данные) или открытым файлом с данными
    
    Возвращает структуру:
        {
            metadata => obj MongoDB::GridFS::File
            , status => {err => '', ok => 1, ... }
        }

=cut

sub gridfs_put_data($$$$)
{
    my ($self, $collection, $data, $metadata) = @_;

    die "Not specified filename" unless $metadata->{filename};
    
    my $profile = Yandex::Trace::new_profile('mongodb:gridfs_put_data', tags => "$self->{db}~put");

    my $basic_fh;
    if (!ref $data) {
        # чтобы открыть строчку как файл, надо ее сначала превратить в байтовую
        # на perl 5.14 работало и без этого, на 5.18 перестало
        utf8::encode($data) if utf8::is_utf8($data);
        open($basic_fh, '<', \$data);
    }
    # если получили уже готовый файлхендл
    elsif (reftype $data eq 'GLOB') {
        $basic_fh = $data;
    }
    else {
        croak "Unsupported data source: " . reftype $data;
    }

    # turn the file handle into a FileHandle
    my $fh = FileHandle->new;
    $fh->fdopen($basic_fh, 'r');

    my $grid = $self->{_database}->get_gridfs($collection);

    # удаляем старый файл (если был)
    my $rm_res = $grid->remove({"filename" => $metadata->{filename}}, {safe => 1});

    # добавляем новый
    my $file = $grid->insert($fh, {"filename" => $metadata->{filename}}, {safe => 1});

    # проверяем отсутствие ошибок при добавлении файла
    my $status = $self->_check_error();
    $self->{log}->out({collection => $collection, put => $metadata}, {status => $status});

    return _process_result($status, $file);
}

=head2 gridfs_get_data(self, collection, {filename => "any_name"})

    Получает данные из mongodb(gridfs)
    
    Возвращает структуру:
        {
            metadata => obj MongoDB::GridFS::File
            , content => "data ..."
            , status => {err => '', ok => 1, ... }
        }

=cut

sub gridfs_get_data($$$)
{
    my ($self, $collection, $metadata) = @_;

    die "Not specified filename" unless $metadata->{filename};

    my $profile = Yandex::Trace::new_profile('mongodb:gridfs_get_data', tags => "$self->{db}~get");

    local $MongoDB::Cursor::slave_okay = 1;
        
    my $grid = $self->{_database}->get_gridfs($collection);
    my $file = $grid->find_one({"filename" => $metadata->{filename}});
    
    $self->{log}->out({collection => $collection, get => $metadata});

    my ($content);
    my $status = $self->_check_error();
    
    if ($file && ref $file eq 'MongoDB::GridFS::File') {
        # без этой обвязки - не работает получение содержимого файла (при этом из консоли работает)
        my ($basic_fh);
        open($basic_fh, '>', \$content);

        # turn the file handle into a FileHandle
        my $fh = FileHandle->new;
        $fh->fdopen($basic_fh, 'w');

        # печатаем содержимое файла в переменную (через filehandle)
        $file->print($basic_fh);
    }

    my $result = _process_result($status, $file);
    $result->{content} = $content if defined $content;
    
    return $result;
}

=head2 gridfs_remove_data(self, collection, {filename => "any_name"})

    Удаляем данные из mongodb(gridfs)
    
    Возвращает структуру:
        {
            metadata => obj MongoDB::GridFS::File
            , status => {err => '', ok => 1, ... }
        }

=cut

sub gridfs_remove_data($$$)
{
    my ($self, $collection, $metadata) = @_;

    die "Not specified filename" unless $metadata->{filename};

    my $profile = Yandex::Trace::new_profile('mongodb:gridfs_remove_data', tags => "$self->{db}~remove");

    my $grid = $self->{_database}->get_gridfs($collection);
    my $file = $grid->remove({"filename" => $metadata->{filename}});

    my $status = $self->_check_error();

    $self->{log}->out({collection => $collection, rm => $metadata}, {status => $status});
    
    return _process_result($status, $file);
}

=pod

    Функции для работы с MongoDB как ключ=значение
        можно найти в https://svn.yandex.ru/direct-utils/yandex-lib/mongodb/lib/Yandex/MongoDB.pm@1505

    Список функций:
        mongo_get_data_short - получение значения из коллекции
        mongo_put_data_short_mass - массовые запросы на получение данных
        mongo_update_one_data - обновление значений ключей
        mongo_remove_data_mass - удаление

=cut

1;
